<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/ /** */

require_once APPPATH.'controllers/inbox.php';

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class Folder_controller extends Inbox {

    /** 
	* Archives all messages in a folder, then deletes the folder. 
	*
	* @todo This would be better as an AJAX-only method - instead of reloading the inbox every time we delete a folder, let's just make an AJAX call and display the resulting message appropriately
	*
	* @param int ID for a custom folder 
	*/
    public function archive_folder($id) {
		$folder = $this->find_custom_folder_or_show_404($id);
		
		//actually delete the folder
		if(Folder::delete($folder->id))
			$this->session->set_success_message('The '.$folder->name_for_display().' folder has been archived successfully.');
		else
			$this->session->set_error_message('An error occured and the '.$folder->name_for_display().' folder was not archived.  Please try again in a moment, and contact an administrator if this problem persists.');
        
        redirect('inbox');
    }
	
	public function create(){

		require_library('form_markup_generator');
		$markup_generator = new Form_markup_generator();
		
		$folder_options = array('' => 'None');
		foreach($this->mailbox->folders_in_hierarchical_order as $id => $folder){
			if(!$folder->is_custom_folder()) continue;
			if(($folder->depth + 1) >= NESTED_FOLDER_MAX_DEPTH) continue;
			$folder_options[$id] = $folder->indented_name_for_display();
		}
		
		
		$markup_generator->set_fields( array('parent_id' => array('type' => 'dropdown', 
																  'label_text' => 'Please select a parent folder:',
																  'referral_label' => 'Parent Folder',
																  'options' => $folder_options),
											 'name' => array('type' => 'text_input', 
											 				 'required' => true,
															 'label_text' => 'Please choose a new folder name:', 
															 'referral_label' => 'Name')
									  ));
		
		//if we have POST data, validate the data and create the folder if possible
		if(!empty($_POST)){
			$markup_generator->set_values_from_post();
			
			//validate that the data meets the validation set up in the markup generator AND that the name has not already been used in this mailbox
			$folder_name_already_exists = in_array($markup_generator->field('name')->value, collect('name_for_display', $this->mailbox->folders));
			if(!$markup_generator->validates() || $folder_name_already_exists){
				$validation_messages = $markup_generator->validation_messages();
				if($folder_name_already_exists)
					$validation_messages[] = 'There is already a folder for this mailbox named "'.$markup_generator->field('name')->value.'."  Please enter another name.';
				$this->template->set('danger_message', '<strong>We need you to make a few changes to this form.</strong> '.ul_if_multiple($validation_messages));
			}else{
				$values = $markup_generator->values();
				if(empty($values['parent_id']))
					unset($values['parent_id']);
				$folder = Folder::create($values);
	
				if(Folder::is_an_entity($folder)){
					$this->session->set_success_message('Successfully created the '.$folder->name.' folder.');
					
					$redirect_location = 'inbox';

					//for ajax, set the success message, close the window && redirect 
					if(!IS_AJAX) redirect($redirect_location);

					//we should be able to do some sort of explicit fancybox close or just $('#fancybox-close').click();, but it's causing js errors and weird ghosting while saving.
					//so for now, we'll just plan on removing all the fancybox divs
					echo '<script>location.reload(); $("#save_btn").click();$("[id^=fancybox]").remove();</script>';		
				}else{
					$this->session->set_error_message('An error occured and the folder was not created.  Please try again in a moment, and contact an administrator if this problem persists.');
					if($this->api->message() == 'Access denied. The application is not authorized for this action.'){
						$this->session->set_service_permission_error_message('Manage', 'Failed to create folder because Manage Direct Service is disabled.');
					}			
				}		
			}
		}

		$markup = form_open(current_url(), array('role' => 'form'));
		$markup .= $markup_generator->markup();
		$markup .= '<div class="form-group createFolder"><div class="form-control-container no-label">';
		$markup .= form_submit('submit', 'Create Folder', 'class="btn btn-primary btn-sm"' );
		$markup .= form_button('cancel_button','Cancel','onclick="$(\'#fancybox-close\').click(); return false;" class="btn btn-default btn-sm"');
		$markup .= '</div></div>';
		$markup .= form_close();


		//if we have errors, or if we're displaying the form for the first time, display the form
		$this->template->set('header', 'Create New Folder');	
		$this->template->load_string('form_dialog_template.php', $markup);
	}
	
	
	/**
	* Change the parent folder for a given folder.
	* Folder data is provided via POST.  Note that this method is used both by AJAX calls and by normal POST calls.  
	* @todo It would be simpler & easier to avoid bugs if we identified both folders by id instead of name.
	*/
    public function change_parent($folder_id) {
		$folder = $this->find_custom_folder_or_show_404($folder_id);

		//make sure that a target parent has been provided and belongs to a folder in this mailbox
    	$parent_id = $this->input->post('parent_folder_select',TRUE);		
		if(empty($parent_id)) $parent_id = null;
		if(!is_null($parent_id) && !Folder::formatted_like_an_id($parent_id)){
			if(is_on_local()) $this->error->should_be_null_or_a_folder_id($parent_id);
			show_404();
		}
				
    	
		//make the change, and report the result appropriately depending on whether or not this is an AJAX call
		$status = $folder->change_parent($parent_id);
		if($status){
    		$message = 'Successfully moved the '.$folder->name_for_display().' folder.';
    		if(!IS_AJAX) $this->session->set_success_message($message);
    	}
    	else {
    		$message = 'An error occured and the '. $folder->name_for_display() .' folder could not be moved.  '.
					   'Please try again in a moment, and contact an administrator if this problem persists.';
    		if(!IS_AJAX) $this->session->set_error_message($message);
    	}
    	
		if(!IS_AJAX) redirect('inbox'); 
		echo $this->json->encode(compact('status', 'message'));
    }		
	
	public function change_parent_form($folder_id) {
		$folder = $this->find_custom_folder_or_show_404($folder_id);

		$this->load->view('inbox/change_parent_form', compact('folder'));
	}
    
     /**
	 * Renames custom folders that the user has created with this application.
     * The function will check for the prefix that the application attaches to custom folders so that it replaces only
     * custom folders. It accepts the new folder name as a post value from the jeditable editable input and the old folder name
     * as a parameter passed to the function.
     */
#TODO - we should really be looking up folders by id, not by name -- much less potential for problems.	
    function rename_folder($folder_id, $folder_name=null, $show_unseen = null) {
		$folder = $this->find_custom_folder_or_show_404($folder_id);
		$folder_name = $folder->name_for_display();
		
		//make sure a new folder name has been specified		
		$new_folder_name = trim($this->input->post('value',TRUE));	
		if(empty($new_folder_name)) trim($new_folder_name = $this->input->post('value_'.hash('sha256',$folder->name_for_display),TRUE));
		if(empty($new_folder_name)){
			echo $this->json->encode(array('error' => 'You must provide a folder name.'));
			return;
		}
		
		//if this folder already has that name, we're done
		if($folder->name_for_display == $new_folder_name){
			echo $this->json->encode(	array('id' => $folder->id,
											  'name' => $folder->name_for_display(),
											  'name_with_message_count' => $folder->name_for_display_with_message_count(),
											  'hashed_name' => hash('sha256', $folder->name_for_display)));
			return;
		}
		
		if(in_array($new_folder_name, collect('name_for_display', $this->mailbox->folders))){
			echo $this->json->encode(array('error' => 'A folder with this name already exists.'));
			return;
		}
		
		if($folder->rename($new_folder_name) && $folder->name_for_display == $new_folder_name){
			echo $this->json->encode(	array('id' => $folder->id,
											  'name' => $folder->name_for_display(),
											  'name_with_message_count' => $folder->name_for_display_with_message_count(),
											  'hashed_name' => hash('sha256', $folder->name_for_display)));
			return;
		}
		
		echo $this->json->encode(array('error' => 'An error occurred and the folder was not renamed.  Please try again in a moment, and contact an administrator if the problem persists.'));
	}

////////////////////////////////////
// PROTECTED HELPER METHODS
////////////////////////////////////

	/**
	* Finds and validates custom folders by id.
	* Since most of our controller methods look up custom folders by id, this method centralizes the lookup process.  Will trigger a 404
	* error if the given value is not an id or not the id for a custom folder in this mailbox.
	* @param int
	* @return Folder 
	*/
	protected function find_custom_folder_or_show_404($folder_id){
		
		if(!Folder::formatted_like_an_id($folder_id)){
			if(is_on_local()) $this->error->should_be_the_id_of_a_custom_folder($folder_id, $offset=1);
			show_404();
		}
		
		$folder = element($folder_id, $this->mailbox->folders);
		if(!Folder::is_an_entity($folder) || !$folder->is_custom_folder()){
			if(is_on_local()) $this->error->should_be_the_id_of_a_custom_folder($folder_id, $offset=1);
			show_404();
		}
		
		return $folder;
	}

	
}
/* End of file inbox.php */
/* Location: ./application/controllers/inbox.php */