<?php
/**
* @package vler
* @subpackage libraries
*/

/* Grab icarus dependencies.  Note that if we're in the CI context, we want to use the Loader functionality to ensure that any application-specific extensions are loaded */
require_library('Object');
load_helpers('form', 'html', 'string');

/**
* Handles logic, validation, and markup generation for an individual form field.
*
* This is an abstract class for field markup generators; if you haven't read the {@link Form_markup_generator} explanation, go do so now.
* Also note that this extends the {@link Object} class; if you haven't read the {@link Object} class documentation, you should do so now.
*
* Field markup generators:
* - Generate the markup for a form field and handle all the configuration logic needed to do so
* - Determine if the value of a form field is valid
* - Generate a standard validation/error message if the value isn't valid.
*
* <b>Basic Use of Field Markup Generators</b>
*
* The examples below demonstrate the basic features and configuration of a field markup geneator.  Keep in mind, though, that most of the time you'll
* be interacting through a {@link Form_markup_generator}.  Once you're coding for real, you should follow the example in that class; these examples are 
* just meant to help you understand the purposes of a field markup generator.
*
* <code>
* //When you set up a field, you need to give it a name
* //This will be the actual name attribute on the field tag
* //We'll also auto-generate a label and and id from it
* $field = new Text_input_field_markup_generator('my_example_input');
* 
* //you can then configure it using any of its settable variables.  for example ...
* $field->max_length = 20;
* $field->required = true;
* $field->attributes = array('class' => "green-collar");
*
* //you can add a "comment" (helper text) if you like 
* //for most generator types, this will display beneath the input
* $field->comment = "Please limit your brilliance to 20 characters."
* 
* //when you're done configuring, echo out the markup
* echo $field->markup();
* </code>
*
* The output markup will vary depending on the markup partials that you're using (see below), but by default it will look like this:
*
* <code>
* <div class="field-markup-container text-input-markup-container">
*	 <span class="label-container text-input-label-container">
*	  <label for="my-example-input-field">My Example Input</label>
*	 </span>
*	 <span class="field-container text-input-field-container">
*	  <input type="text" name="my_example_input" value="" id="my-example-input-field" class="text-input-field green-collar" maxlength="20" size="20">
*   <div class="form-field-comment">
*    Please limit your brilliance to 20 characters.
*   </div>
*	 </span>
*	</div>
* </code>
*
* Note that the containers have both a generic class ("field-container") and a class specific to the type of markup generator ("text-input-field-container").
* If we were working with a markup generator that inherited from other types (say, a numeric text input type that inherited from the text input generator), the 
* parent generator's classes would be included as well.  This comes in handy when you want to apply CSS to a field type and all of its descendents.
* 
* <code>
*	//set the value
*	$field->value = "Here is a sentence that is way more than twenty characters."
*	if(!$field->validates()){
* 	//displays a message telling the user that the value is too long
*		echo $field->validation_messages; 
* 	//markup now includes an error class, so you can style in an error indicator
*		echo $field->markup;
*	}
* </code>	
*
* In our example above, the text for the label was auto-generated from the field name. You can set it to something else by setting 
* {@link Field_markup_generator::$_label_text label_text}.
* You may find once you do so that the label text is unwieldy or looks odd in the validation messages (if, for example, your label was a long question, or included
* punctuation at the end).  If so, you can set a {@link Field_markup_generator::$_referral_label referral_label} value, which will be what your label is called
* whenever it's referred to in validation messages.
*
* <b>Partials and Markup Generation</b>
*
* Field markup generator objects control all the logic and validation for a form field, but the markup itself takes place in partials.  Think of this as analogous to 
* controller and views in the MVC model.  (The term "partial" is borrowed from Ruby on Rails -- it refers to any snippet of markup that doesn't make up an entire view by 
* itself.)
*
* Field markup generators each have three partials:
* <ul>
* 	<li>A partial for the form field by itself.  By convention, this is named after the field generator's {@link type} with "_field" at the end -- so, the form field partial 
*			  for the {@link Text_input_markup_generator} is {@link _text_input_field.php}.  This partial is used by the {@link field_markup()} method.</li>
*		<li>A partial for the field label by itself.  By convention, this is named after the field generator's {@link type} with a "_label" appended, but most markup generators 
*				use the default {@link _label.php} partial. This partial is used by the {@link label_markup()} method.</li>
*		<li>A partial for the field and label together, using the markup from {@link field_markup()} and {@link label_markup()}.  The structure of the markup in this partial is
* 			designed to be easy to style using CSS so that you can change the look and feel of your forms from application to application without having to customize partials or 
*				change the markup.  Using this markup will also include any {@link Field_markup_generator::$_comment comments} assigned to the field.  By convention, this is named 
* 			after the field generator's {@link type} -- so, the form field partial for the {@link Text_input_markup_generator} is {@link _text_input.php}.  This partial is used by
*				the {@link markup()} method.</li>
* </ul>
*
* Partials are designed to follow object-oriented-esque patterns of inheritance. If a partial doesn't exist for a given field markup generator, the corresponding 
* partial for the field markup generator's parent will be used.  If the standard Icarus partial doesn't meet your needs, you may also "override" the icarus partial
* using any of the following methods:
* 
* <ul>
* 	<li>Set {@link Field_markup_generator::$_partial partial} to a different file.</li>
*		<li>Create an "override" file with the same name as this file, located in the default extension directory 
*				({@link APPPATH}/libraries/form_markup_generator/field_markup_partials).</li>
*		<li>If you're working outside of Codeigniter or would prefer to store your overrides in a different directory, set the 
* 			{@link Field_markup_generator::$_extension_directory extension_directory} to your directory of choice.  Most of the time you'll want to set this in the 
*				{@link Form_markup_generator::$_extension_directory Form_markup_generator} object, which will automatically pass the value on to all of the field markup generators
* 			that belong to it. Note that your extension directory should still store partials in a subdirectory named "field_markup_partials."</li>
*	</ul>
*
*	<b>How to create a new form markup generator</b>
* <ol>
* 	<li>Set up a new class that extends the {@link Field_markup_generator} or one of its descendents.  The name of the class should end with "_markup_generator".
* 			Please stick to convention and name the class using underscores instead of camelcasing.</li>
* 	<li>Give the class a {@link $_type type} that is a lower case version of the class name, with "_markup_generator" stripped from the end.</li>
*		<li>If you're directly extending {@link Field_markup_generator}, make sure to set up field and field container partials (see the Partials and Markup Generation section
* 			for more information on this).</li>
* </ol>
*
*
* @todo Move this very long tutorial to the wiki.
* @package vler
* @subpackage libraries
*/
class Field_markup_generator extends Object{
	
	/**
	* The name of the field markup generator type.
	* This should match the name of the class, with "_markup_generator" stripped from the end.  This value will be used to identify the partials for this 
	* markup generator, and will be included in the {@link classes} applied to the form field field element. 
	* @var string
	*/
	protected $_type;
	
	/**
	* Name for this field.
	* This is the unique identifier that will be used to reference this field markup generator in the parent {@link Form_markup_generator}.
	* It will also usually be the value of the name attribute on the HTML tag, though that may differ if {@link _name_for_markup name_for_markup} is set.
	* By convention, words should be lower-case and separated by underscores.
	* @var string
	*/
	protected $_name;	
	
	/**
	* Name that will be used as the value of the name attribute on the HTML tag.
	* By default, this will be the same as {@link _name name} unless {@link _model_class model_class} is set.
	* @see name_for_markup()
	* @var string
	*/
	protected $_name_for_markup;
	
	/**
	* The text to be used as the value of the <label> element.
	* You can set a separate {@link _referral_label referral_label} if you wish to refer to the element differently invalidation messages;
	* this is useful when the label text is a question or otherwise unwieldy.
	* @var string
	*/
	protected $_label_text;
	
	/**
	* The text to be used when referring to this field in validation messages and all contexts other than the <label> field.
	* By default, this will be auto-generated from {@link _label_text label_text}, but you may choose to set it if the label text is unwieldy or would be
	* confusing in the context of a validation message.  For example, if the label for a field was "How much wood can a woodchuck chuck?" you might want to refer to
	* it in validation messages as the "Woodchuck" field.
	* @var string
	*/
	protected $_referral_label; //How should we refer to this field in validation feedback, etc?
	
	/**
	* Any HTML attributes that should be applied to the <label> tag.
	* @var array Format ('class' => "my-class", 'id' => "my-id")
	*/
	protected $_label_attributes;
	
	/**
	* @var int The maximum number of characters acceptable in the value.
	*/
	protected $_max_length;
	
	/**
	* @see attributes()
	* @see set_attributes()
	* @var array Associative array of attributes to be applied to the field element, e.g. array('class' => 'this-is-a-class', 'href' => 'http://google.com').
	*/	
	protected $_attributes = array();
	
	/**
	* Helper text for the user that gives additional instructions or clarification about a field.
	* This will only be applied if the {@link _partial partial} for this markup generator makes use of it.  In general, it appears underneath the field element, usually
	* in a smaller font-size than the labels and main text for the form.
	* @see comment()
	* @var string
	*/
	protected $comment;
	
	/**
	* True if the label and form field should appear on the same line.
	* True by default; when false, the class "not-inline" will be included in this field's {@link classes()}.  You'll need to apply styling in your CSS as necessary
	* to ensure that inline/not-inline behavior appears as intended.
	* @var boolean
	*/
	protected $inline = true;
	
	/**
	* True if this field is required.
	* This affects field validation via {@link has_field_if_required} and will add the "required" attribute to {@link $_attributes attributes}. 
	* @var boolean
	*/
	protected $_required;
		
	protected $_value;
	protected $raw_value;
	
	protected $_model_class;
	
	//If you have any application-specific extensions, put your application's equivalent here.  (default for Codeigniter: APPATH.libraries/form_markup_generator)
	protected $_extension_directory; //add a setter to make sure that this is a real directory //deprecated for better performance - 2016-05-04 MG
	
	//where we should look for partials - this will be in the form_markup_generator library directory in either the application or VLER libraries
	protected $_partial_directory = 'field_markup_partials/'; 

	protected $_label_partial; //partial for just the label
	protected $_field_partial; //partial for the field 
	protected $_partial; //partial for the field + the label
	protected $_width_class; //a non-semantic class used to determine width - for example, using bootstrap this might be col-xs-6
	protected $_label_width_class; //a non-semantic class used to determine width - for example, using bootstrap this might be col-xs-6
	
	
	protected $validation_method; //needs to be set in child classes.
	protected $_security_validation_method;
	protected $_security_sanitization_method;
	protected $_security_validation_args = array();
	protected $_security_sanitization_args = array();
	

	
	/** Validation message for the user to explain that this field is required.
	* This message will be included in {@link validation_messages()} if the field value fails {@link has_value_if_required()}.  
	* Use "%s" to signify the {@link $_referral_label referal_label} for the field.
	* @var string */
	protected $_feedback_required_field = "The %s field is required.";
	protected $_feedback_invalid_input = "Please enter a value for the %s field."; //child classes should implement a specific description of the value
	protected $_feedback_too_long = "Please limit your response to %m characters for the %s field (currently %c)."; //child classes should implement a specific description of the value
	protected $_parent_field_group;
	protected $_parent_referral_label;
	protected $_is_part_of_field_group = false;
	
	protected $_has_error = false; //will be set when validates() is called
	
	protected $_property_validation_rules= array(	'name_for_markup' => 'nonempty_string_with_no_whitespace',
													'max_length' => 'unsigned_integer',
													'model_class' => 'model_class',
													'label_text' => 'string',
													'comment' => 'nonempty_string',
													'inline' => 'boolean',
													'partial_directory' => 'nonempty_string_with_no_whitespace',
													'label_partial' => 'icarus_view',
													'field_partial' => 'icarus_view',
													'partial' => 'icarus_view',
													'extension_directory' => 'directory_path', //deprecated for better performance - 2016-05-04 MG
													'required' => 'boolean',
													'security_validation_method' => 'nonempty_string_with_no_whitespace',
													'security_sanitization_method' => 'nonempty_string_with_no_whitespace',
													'security_validation_args' => 'array',
													'security_sanitization_args' => 'array',
													'referral_label' => 'nonempty_string',
													'feedback_required_field' => 'string',
													'feedback_invalid_input' => 'string',
													'feedback_too_long' => 'string',
													'is_part_of_field_group' => 'boolean',
													'parent_referral_label' => 'string',
													'has_error' => 'boolean',
													'label_attributes' => 'associative_array', 
													'width_class' => 'nonempty_string',
													'label_width_class' => 'nonempty_string'
												);	

	function __construct($name = NULL, $values = array()){
		parent::__construct();
		$this->is = $this->validator;
		if(!is_null($name)){
			if(!$this->is->nonempty_string($name) || !$this->is->string_with_no_whitespace($name))
				return $this->error->property_value_should_be_a_nonempty_string_with_no_whitespace('name', get_class($this), $name); 
			$this->_name = $name;
		}
		
		//if(!$this->is->associative_array($values)) return $this->error->should_be_an_associative_array($values); //this is better validation-wise, but we're trying to improve performance
		if(!is_array($values)) return $this->error->should_be_an_associative_array($values); 
		foreach($values as $property => $value){
			if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property, $this);
			$this->$property = $value;
		}
	}
	
	function describe(){
		return '#'.$this->id();
	}
	
	//mostly, this is a convenience method -- id should always be set/handled through attributes, but we need to reference it pretty requently
	function id(){
		return $this->attributes['id'];
	}
	
	function id_for_container(){
		return replace_last_with('-field', '-markup-container', $this->id);
	}
	
	function link_to_field(){
		$id = $this->id_for_container();
		if($this->is_part_of_field_group)
			$id = $this->id; //fields in a field group are within their parent's markup container.
		
		return '<a href="#'.$id.'">'.strip_tags($this->referral_label).'</a>'; //strip tags in case there are links inside the label.
	}
	
////////////////////////////////
// MARKUP METHODS
////////////////////////////////

	/**
	* Returns the markup for the label of this form field.
	* Uses the partial specified by {@link label_partial} to generate the label markup.  The partial will be passed the {@link Field_markup_generator} object as a 
	* parameter.  This partial generates the markup for the <label> tag only - to generate the label and field together, see {@link markup()}.
	* 
	* @param array HTML attributes to be applied to the <label> tag.
	* @return string
	*/		
	function label_markup($label_attributes = array()){
		if(!is_array($label_attributes)) return $this->error->should_be_an_array($label_attributes);
		
		if(!$this->property_is_empty('label_attributes'))
			$label_attributes = merge_attributes($this->label_attributes, $label_attributes);
		
		if($this->property_is_empty('label_partial')) 
			return $this->error->warning("I can't generate the label markup until you set a valid label partial for the ".$this->type.' field markup generator');
	
		return $this->markup_for_partial($this->label_partial, array('markup_generator' => &$this, 'label_attributes' => $label_attributes), $return_as_string = TRUE);
	}
	
	/**
	* Returns the markup for the form field element.
	* Uses the partial specified by {@link field_partial} to generate the field markup.  The partial will be passed the {@link Field_markup_generator} object as a 
	* parameter.  This partial generates the markup for the field itself only - to generat ethe label and field together, see {@link markup()}
	* 
	* @return string
	*/		
	function field_markup(){
		if($this->property_is_empty('field_partial')) 
			return $this->error->warning("I can't generate the field markup until you set a valid field partial for the ".$this->type.' field markup generator');
		
		return $this->markup_for_partial($this->field_partial, array('markup_generator' => &$this), $return_as_string = TRUE);
	}
	
	/**
	* Returns the markup for the form field and its label within a container.
	* Uses the partial specified by {@link partial}, which will be passed the {@link Field_markup_generator} object as a 
	* parameter.  This partial generates the markup for both the field and its label - to generate label and field separately, use {@link label_markup()} and {@link field_markup()}.
	* 
	* @return string
	*/		
	function markup(){		
		if($this->property_is_empty('partial')) 
			return $this->error->warning("I can't generate the markup until you set a valid partial for the ".$this->type.' field markup generator');	
		
		$label_attributes = array();
		if(isset($this->label_width_class) && $this->inline)
			$label_attributes['class'] = $this->label_width_class;
		
		$parameters = array('label_markup' => $this->label_markup($label_attributes), 'field_markup' => $this->field_markup(), 'markup_generator' => &$this);
		return $this->markup_for_partial($this->partial, $parameters, $return_as_string = TRUE);
	}

///////////////////////////////////
// SANITIZATION/VALIDATION METHODS
///////////////////////////////////
	
	/**
	* True if the value of this form field is valid.
	* Note that this method assumes that the form has been filled out already, and that if {@link $_value value} has not been set, it's because the field was not filled out.
	* @return boolean
	*/
	function validates(){
		if(!$this->has_valid_value_if_any() || !$this->has_value_if_required() || !$this->value_does_not_exceed_max_length()){
			$this->has_error = true;
			return false; 
		} 
		return true;
	}
	
	/**
	* True if value of this form field is within length limits.
	* Internal method, intended to be overriden by child classes; to see if a this form field has a value that meets all validation requirements, see {@link validates()}.
	* @return boolean
	*/
	protected function value_does_not_exceed_max_length(){		
		if(isset($this->max_length) && isset($this->value)){
			if(is_numeric($this->value))
				return (number_length($this->value) <= $this->max_length);
			return (strlen($this->value) <= $this->max_length);
		}
		return true;
	}
	
	/** 
	* True if this field has a value set
	* @return boolean */
	protected function has_value(){
		return (!empty(trim($this->raw_value)) || trim($this->raw_value) === '0');
	}
	
	/**
	* True if this form field is either not required or has a value.
	* Internal method, intended for overriding by child classes; to see if a this form field has a value that meets all validation requirements, see {@link validates()}.
	* @return boolean
	*/
	protected function has_value_if_required(){
		if($this->required && !$this->has_value()) return false;
		return true;
	}
	
	/**
	* True if the value of this field is valid.
	* Note that this internal method validates the value against the validation method set in {@link $_validation_method validation_method},
	* and does not take into account whether or not the field is required or fits within length requirements.
	* To see if this form field has a value that meets all validation requirements, see {@link validates()}.
	* @return boolean
	*/
	protected function has_valid_value_if_any(){
		//every generator should at least validate that the value is a string, if nothing else
		//if we don't have a way of validating the value, we should make that very obvious in dev.
		if(!isset($this->validation_method))	return $this->error->error('There is no validation method set for '.get_class($this));
		if(!$this->is->validation_rule($this->validation_method)) 
			return $this->error->error('There is no method called "'.$this->validation_method.'" for class '.get_class($this->validator));

		if(!isset($this->value)) return true;
			
		$validation_method = $this->validation_method;
		return $this->is->$validation_method($this->value);	
	}
			
	/**
	* Generates a message for the user describing any way in which the value of this form field field to validate.
	* Like {@validates}, this method should not be called unless the form has already been filled out -- since the field markup generator
	* only receives a value if the field has been filled out, it has no way to distinguish between a required field that was mistakenly left
	* blank and a field that hasn't been filled out yet.
	*
	* @return string
	*/		
	function validation_messages(){
		if($this->property_is_empty('feedback_required_field')) $this->error->should_be_a_nonempty_string($this->feedback_required_field);
		if($this->property_is_empty('feedback_invalid_input')) $this->error->should_be_a_nonempty_string($this->feedback_invalid_input);
		
		$messages = array();		
		if(!$this->has_value_if_required())
			$messages[] = $this->feedback_required_field;
		if(!$this->has_valid_value_if_any())
			$messages[] = $this->feedback_invalid_input;
		if(!$this->value_does_not_exceed_max_length()){
			$messages[] = $this->feedback_too_long; //replacements are made in the getter
		}
				
		return implode_nonempty(" ", $messages);
	}		
	
	/**
	* Applies a Bahlibs sanitization method to the form field value.
	* Uses the validation and sanitization methods specified by {@link security_validation_method} and {@link security_sanitization_method}.
	*
	* @param mixed The raw value, as entered by the user
	* @return mixed The value after being sanitized
	*/
	protected function sanitize_value($value){
		require_library('form_markup_generator/security.class');
		//the lack of a valid security method is a major issue, so trigger a fatal issue if we don't have valid methods for this to make absolutely sure this is noticed ASAP
		if(!method_exists('Security', $this->security_validation_method))
			return $this->error->error("I can't validate the '".$this->name."' form field without a validation method, but you gave me ".$this->security_validation_method);
		if(!method_exists('Security', $this->security_sanitization_method))
			return $this->error->error("I can't validate the '".$this->name."' form field without a sanitization method, but you gave me ".$this->security_validation_method);

		return Security::secureVariable($value, 
										'Security::'.$this->security_validation_method, 
										'Security::'.$this->security_sanitization_method, 
										$this->security_validation_args, 
										$this->security_sanitization_args);
	}
	
	/**
	* Restore the value to the default value for this form field type (usually null).
	* This should be used instead of {@link unset}, which won't clear both the raw value and the vaue.
	*/
	function clear_value(){
		//determine the default value
		$class = get_class($this);
		$dummy_field_markup_generator = new $class();
			$this->raw_value = $dummy_field_markup_generator->raw_value; 
		$this->_value = $dummy_field_markup_generator->value;
	}
	
///////////////////////////
// GETTERS FOR CLASS VARS
////////////////////////////
//DEVELOPERS, PLEASE NOTE - the #@+ in the docblock below indicates that this docblock will be applied to every method until we hit the /**#@-*/ dockblock.
//Please only place getter methods for protected variables in this section, or the documentation will be inaccurate.
 
/**#@+ 
* Accessor for a protected variable.
*
* This is a getter method for a protected variable that has the same name as this method, but prefixed by an underscore.  
* 
* In general, you should not call on this method directly.  Call on the variable, but without the underscore in the name -- the {@link __get} method will
* call on any necessary accessor methods.
*/
	
	/** 
	* @see $_attributes
	* @return array
	*/
	function attributes(){
		$id = $this->make_string_css_friendly($this->name_for_markup).'-field';
		$default_attributes = array('id' => $id, 'class' => implode(' ', $this->classes()));
		$attributes = $this->_merge_with_parent_array('_attributes');
		if($this->required)
			$attributes['required'] = 'required';
			
		return merge_attributes($default_attributes, $attributes);
	}
	
	/**
	* @see $_comment
	* @return string
	*/
	/*function comment(){
		//We should probably have a toggle for this.  For now, let's not include it, hasn't been tested enough.
		//if($this->has_error)
			//return implode_nonempty(' ', array('<span class="validation-message">'.$this->validation_messages.'</span>', $this->_comment));
		//}
			
		return $this->_comment;	
	}*/
	
	/**
	* @see $_extension_directory
	* @return extension_directory
	*/
	 function extension_directory(){
		$return_value = null;
		if(isset($this->_extension_directory))
			$return_value = $this->_extension_directory;
		elseif(defined('CI_VERSION') && defined('APPPATH') && file_exists(APPPATH.'libraries/form_markup_generator')) 
			$return_value = APPPATH.'libraries/form_markup_generator/';
		
		if(!is_null($return_value) && !string_ends_with('/', $return_value))
			$return_value .= '/';
		
		return $return_value;
	} 

	/**
	* Generates a message for the user that explains that this field did not meet the validation requiremens.
	* This message will be included in {@link validation_messages()} if the field value fails {@link has_valid_value_if_any}.
	* You may customize the message by setting {@link $_feedback_invalid_input feedback_invalid_input}.  Use "%s" to signify the name of the field.
	* @return string
	*/
	function feedback_invalid_input(){
		return str_replace("%s", $this->link_to_field(), $this->_feedback_invalid_input);
	}

	/**
	* @see _feedback_required_field
	* @return string
	*/
	function feedback_required_field(){
		return str_replace("%s", $this->link_to_field(), $this->_feedback_required_field);
	}	
	
	/**
	* @see _feedback_too_long
	* @return string
	*/
	function feedback_too_long(){
		$message = str_replace("%s", $this->link_to_field(), str_replace("%m", $this->max_length, $this->_feedback_too_long));
		if(is_numeric($this->value))
			$message = str_replace("%c", number_length($this->value), $message);
		else
			$message = str_replace("%c", strlen($this->value), $message);
		return $message;
	}
	
	/**
	* @see _field_partial
	* @return string
	*/
	function field_partial(){
		if(isset($this->field_partial) && $this->is->icarus_view())	return $this->_field_partial;
		return $this->_find_partial('field');
	}	

	/**
	* @see _label_partial
	* @return string
	*/	
	function label_partial(){
		if(isset($this->label_partial) && $this->is->icarus_view())	return $this->_label_partial;
		return $this->_find_partial('label');
	}
	
	/**
	* @see _label_text
	* @return string
	*/			
	function label_text(){
		if(isset($this->_label_text)) return $this->_label_text;
		if(!$this->property_is_empty('name'))	return humanize(strip_from_end('_id', $this->name));
		return $this->error->warning("I can't generate a label_text value until you give me a value for ".get_class($this).'::name');
	}
	
	//not doing anything fancy, but this is a read-only var and we won't be able to access it without a getter 
	//wait, what?  self, you were not being remotely clear when you wrote that comment. -- MG 2012-08-06
	/**
	* @see _name
	* @return string
	*/	
	function name(){
		return $this->_name;
	}
	
	/**
	* @see _name_for_markup
	* @return string
	*/
	function name_for_markup(){
		if(isset($this->_name_for_markup)) return $this->_name_for_markup;
		if($this->property_is_empty('name')) return $this->error->warning("I can't generate a name_for_markup value until you give me a value for ".get_class($this).'::name');
		if(!$this->property_is_empty('model_class')) return $this->model_class.'['.$this->name.']';
		return $this->name;
	}		
	
	/**
	* @see _partial
	* @return string
	*/		
	function partial(){
		if(isset($this->partial) && $this->is->icarus_view()) return $this->_partial;
		return $this->_find_partial();
	}
	
	/**
	* @see _referral_label
	* @return string
	*/	
	function referral_label(){
		if(isset($this->_referral_label)) return $this->_referral_label;	
		
		if($this->is_part_of_field_group){
			$parent_label = $this->parent_referral_label;
			//it's pretty common to end up with double words, like "Scenario Time Time Hour" field -- let's strip one of them if that happens
			if(first_word($this->label_text) == last_word($parent_label))
				$parent_label = trim(strip_from_end(last_word($parent_label), $parent_label));
			return $parent_label.' '.$this->label_text;
		}
		
		return $this->label_text;
	}
/**#@-*/	
	
///////////////////////////
// SETTERS FOR CLASS VARS
//////////////////////////
//DEVELOPERS, PLEASE NOTE - the #@+ in the docblock below indicates that this docblock will be applied to every method until we hit the /**#@-*/ dockblock.
//please only place setter methods for protected variables in this section, or the documentation will be inaccurate.
 
/**#@+ 
* Accessor for protected variable.
*
* This is a setter method for a protected variable that has the same name as this method, but prefixed by an underscore.  
* 
* In general, you should not call on this method directly.  Call on the variable, but without the underscore in the name -- the {@link __set} method will
* call on any necessary accessor methods.
*/

	/**
	* @see _attributes
	* @param array
	*/
	function set_attributes($new_attributes){
		if(!is_array($new_attributes)) return $this->error->property_value_should_be_an_array('attributes', $this, $new_attributes);
		
		//grab the literal attributes value -- don't want to double up values when we autogenerate default values in attributes()
		$this->_attributes = merge_attributes($this->_attributes, $new_attributes);
	}

	function set_attribute($attribute, $value){
		if(!$this->is_a_nonempty_string($attribute)|| $this->is_a_string_with_no_whitespace($attribute)) 
			return $this->error->should_be_a_nonempty_string_with_no_whitespace($attribute);
		if(!$this->is_a_string($value)) return $this->error->should_be_a_string($value);
		
		$this->set_attributes(array($attribute => $value));
	}	
	
	/**
	* @see _parent_field_group
	* @param Field_group_markup_generator
	*/
	function set_parent_field_group(&$value){
		if(!is_a($value, 'Field_group_markup_generator')) return $this->error->property_value_should_be_a_field_group_markup_generator('field_group', $this, $value);
		$this->parent_referral_label = $value->referral_label;
		$this->is_part_of_field_group = true;
		$this->_parent_field_group =& $value;
	}

	/**
	* Sanitizes the user input before setting the field value.
	* The un-sanitized field value is stored in {@link raw_value}.
	*
	* @see _value
	* @param mixed
	*/
	function set_value($value){
		if(is_numeric($value))
			$value = (string)$value;
		if(is_bool($value) && !$value)
			$value = '';	
		if(!is_string($value)) 
			return $this->error->property_value_should_be_a_string('value', $this, $value);
		$this->raw_value = $value;
		$this->_value = $this->sanitize_value($value);
	}
	
/**#@-*/ 	
	
///////////////////////
/// OTHER METHODS
//////////////////////
	
	protected function _find_partial($partial_type = ''){
		//check to see if we've cached the partial location in CI config.  (better approach for someday in the future - run a make file for every build that records where the partial is so we can stop looking this up.)
		$CI = get_instance();
		if(!isset($CI->config->field_markup_partials)) $CI->config->field_markup_partials = array();
		if(!isset($CI->config->field_markup_partials[$this->type])) $CI->config->field_markup_partials[$this->type] = array();
		if(array_key_exists($partial_type, $CI->config->field_markup_partials[$this->type])) 
			return $CI->config->field_markup_partials[$this->type][$partial_type];
			
		//not in the cache?  check to see if we have a partial named after this type
		if(empty($partial_type))
			$partial_name = '_'.$this->type;
		else
			$partial_name = '_'.$this->type.'_'.$partial_type;
		
		$partial_directories[] = APPPATH.'libraries/form_markup_generator/'.$this->partial_directory;
		if(!$this->property_is_empty('extension_directory'))
			$partial_directories[] = $this->extension_directory.$this->partial_directory;
		
		foreach($partial_directories as $partial_directory){
			$path = $partial_directory.$partial_name.'.php';
			if(file_exists($path)){
				$CI->config->field_markup_partials[$this->type][$partial_type] = $path;
				return $path;
			}
		} 
		
		//no view specific to this type?  check the parent
		$parent_type = $this->get_parent_default_value('type');
		if(!empty($parent_type)){
			$parent_class = ucfirst($parent_type).'_markup_generator';
			$parent = new $parent_class();
			if(!$this->property_is_empty('extension_directory'))
				$parent->extension_directory = $this->extension_directory;
			$path = $parent->_find_partial($partial_type);
			if(!empty($path)){
				return $path;
			}
		}
		
		//nothing from the parent?  if it's a label, we can just use the default label
		if($partial_type == 'label'){
			$path = $partial_directory.'_label.php';
			if(file_exists($path)){
				$CI->config->field_markup_partials[$this->type][$partial_type] = $path;
				return $path;			
			}
		}
			
		return $this->error->warning("I can't find a ".$partial_type.' partial for the '.$this->type.' field markup generator');
	}


	/**
	* Generate an array of the default classes that should be included the HTML attributes for this generator.
	* This will reflect the ancestry of the field markup generator, so that CSS rules may be broadly applied to a markup generator type and its children.
	* For example: array('great-grandmother-field', 'grandmother-field', 'mother-field', 'self-field')
	* @uses parent_types
	* @uses make_string_css_friendly
	* @param string
	* @return array
	*/		
	function classes($suffix = '-field'){
		$ancestry = array_reverse($this->parent_types());
		array_push($ancestry, $this->type);
		foreach($ancestry as &$type) $type = $this->make_string_css_friendly($type).$suffix;
		$classes = $ancestry;
		if($this->has_error && !is_a($this, 'Field_group_markup_generator'))
			$classes[] = 'has-error';
		if($suffix == '-form-control-container'	&& isset($this->width_class) && $this->inline)
			$classes[] = $this->width_class;
			
		return $classes;
	}	

	/**
	* Find the names of the parents of this field markup generator type.
	* The parent types will be returned in an array ordered from the lowest in the hiearchy to the highest, e.g. array('mother', 'grandmother', 'great-grandmother').
	* Note that the array will contain the type names, not the class names.
	* @return array 
	*/	
	protected function parent_types(){			
		$parent_class = get_parent_class($this);
		$parent_type = $this->get_parent_default_value('type');
		if(empty($parent_class) || empty($parent_type) || !method_exists($parent_class, 'parent_types')) return array();
		$parent_object = new $parent_class();
		return array_merge(array($parent_type), $parent_object->parent_types());
	}
	
	/**
	* Replaces spaces and underscores in a string with dashes to fit our CSS standards.
	* @param string
	* @return string
	*/
	function make_string_css_friendly($string){
		if(!$this->is->nonempty_string($string)) return $this->error->should_be_a_nonempty_string($string);
		
		return strip_from_end('-', preg_replace('/-{2,}/', '-', str_replace(']','-', str_replace('[', '-', str_replace('_', '-', str_replace(' ', '-', strtolower($string)))))));
	}
	
	/**
	* Renders a partial and returns the content as a string.
	* This is analogous to using CI's {@link CI_loader::view} method with the optional $return_as_string parameter set to true.
	* @param string File path to the partial, in a format that will be understood by PHP's require() command.
	* @param array Array of variables to pass to the partial, formatted array(variable_name => variable_value)
	* @return string
	*/
	function markup_for_partial($partial_path, $variables_for_partial){
		if(!$this->is->string_like_a_file_path($partial_path)) return $this->error->should_be_a_string_like_a_file_path($partial_path);
		if(!is_array($variables_for_partial)) return $this->error->should_be_an_associative_array($variables_for_partial);
		//if(!$this->is->associative_array($variables_for_partial)) return $this->error->should_be_an_associative_array($variables_for_partial); //this is better validation-wise, but we're trying to improve performance
		
		extract($variables_for_partial);
		ob_start();
		require $partial_path;
		$buffer = ob_get_contents();
		@ob_end_clean();
		return $buffer;
	}
	
}