<?PHP  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
 * @package vler
 * @subpackage libraries
 */ 

require_library('authentication');

/**
 * @package vler
 * @subpackage libraries
 */ 
abstract class Certificate_authentication extends Authentication {
	protected $_certificate_info; //array of user information from the cert - see certificate_info()
	protected $_name_as_array; //array ('first_name' => $first_name, 'middle_name' => $middle_name, 'last_name' => $last_name, 'suffix' => $suffix);
	protected $organization_id_field_name; //the database field that corresponds to the user's organization id



	/**
	* Finds an array of information about the user based on their certificate.
	*
	* This typically includes name information and organization id, but it may vary in child classes depending on the type of certificate.
	*
	* @todo It would be great to have child classes extend this to add meaningful keys and prevent further parsing
	*
	* @return array 
	*/
	protected function certificate_info(){
		if(!isset($this->_certificate_info)){
			$parsed_cert = openssl_x509_parse($this->clean_cert_string($_SERVER['HTTP_CERT']));
			if(is_array($parsed_cert) && array_key_exists('subject', $parsed_cert) && array_key_exists('CN', $parsed_cert['subject'])){
				$this->_certificate_info = explode(' ', trim($parsed_cert['subject']['CN']));
			}
		}
		return $this->_certificate_info;
	}
	
	protected function email(){
		if(!isset($this->_email)){
			$parsed_cert =  openssl_x509_parse($this->clean_cert_string($_SERVER['HTTP_CERT']));
			
			if(array_key_exists('extensions', $parsed_cert) && array_key_exists('subjectAltName', $parsed_cert['extensions']) && string_contains('email:', $parsed_cert['extensions']['subjectAltName'])){
				$email = substr($parsed_cert['extensions']['subjectAltName'], strpos($parsed_cert['extensions']['subjectAltName'], 'email:') + 6);

				if(string_contains(',', $email))
					$email = strstr($email, ',', true);

				$this->_email = trim($email);
			}
			
			//if that didn't work, look for the uid
			if(array_key_exists('subject', $parsed_cert) && array_key_exists('UID', $parsed_cert['subject'])){
				$email = trim($parsed_cert['subject']['UID']);
				if(get_instance()->is->string_like_an_email_address($email))
					$this->_email = $email;
			}
			
		}
		return $this->_email;
	}
	
	protected function given_name(){
		return element('given_name', $this->name_as_array);	
	}
	
	protected function middle_name(){
		return element('middle', $this->name_as_array);	
	}
	
	protected function family_name(){
		return element('family_name', $this->name_as_array);	
	}
	
	protected function suffix(){
		return element('suffix', $this->name_as_array);	
	}	
	
	/**
	* The user's name as defined by their certificate.
	* @return array
	*/
	protected function name_as_array(){
		if(!isset($this->_name_as_array)){
			if(!$this->is_valid()) return $this->error->warning("I can't find the name when the authentication is not valid");
			
			$pieces = array_slice($this->certificate_info, 0, array_search($this->organization_id, $this->certificate_info));
						
			//check to see if the name includes a suffix as long as we have more than two pieces for the name (since we always have first name and last)
			if(count($pieces) > 2){ 
				$last_key = array_last_key($pieces);
				if(in_array(strtolower($pieces[$last_key]), array('jr', 'sr', 'jr.', 'sr.'))){
					$suffix = ucfirst(strtolower($pieces[$last_key]));
					unset($pieces[$last_key]);
				}elseif(is_roman_numeric($pieces[$last_key])){
					$suffix = strtoupper($pieces[$last_key]);
					unset($pieces[$last_key]);
				}
			}
			
			//names tend to be all caps on the certs, and we don't want that
			//unfortunately, this is not going to help us if the user is named DeWerd or McDonald or DeLint, but admins can manually correct as needed
			$pieces = array_map('ucfirst', array_map('strtolower', $pieces));
			
			//find the last name
			$family_name = last_element($pieces);
			unset($pieces[array_last_key($pieces)]);
			
			//find the first name
			$given_name = first_element($pieces);
			unset($pieces[array_first_key($pieces)]);
			
			if(!empty($pieces))
				$middle_name = implode_nonempty(' ', $pieces);
			
			$this->_name_as_array = compact('given_name', 'middle_name', 'family_name', 'suffix');
		}
		return $this->_name_as_array;
	}

	/**
	* The user corresponding to the current credentials.
	* Note that this user may or may not be authorized to use the site.  This just identifies the user based on the organization id (CAC or PIV)
	* Note also that child classes must define {@link_organization_id_field_name}
	* @return User
	*/
	protected function user(){
		if(empty($this->organization_id_field_name)) return $this->error->warning('I can\'t find the user when Certificate_authentication::$organization_id_field_name has not been defined by the child class');
		
		if(!isset($this->_user) && $this->is_valid() && !$this->property_is_empty('organization_id')){
			$user = User::find_one(array($this->organization_id_field_name => $this->organization_id));			
			if(User::is_an_entity($user)){
				$this->_user = $user;
			}
		}
		return $this->_user;
	}

	protected function clean_cert_string($cert) {
		$output = preg_replace("/-----[A-Z]+\sCERTIFICATE-----/","",$cert);
		$output = preg_replace("!\s+!"," ",$output);
		$cert = str_replace(" ","\n",$output);
		$cert = "-----BEGIN CERTIFICATE-----\n".trim($cert)."\n-----END CERTIFICATE-----";
		return $cert;
	}
	
}
?>