<?PHP  if ( ! defined('BASEPATH')) exit('No direct script access allowed'); 
/**
* @package vler
* @subpackage libraries
*/

/**
* @package vler
* @subpackage libraries
*/
abstract class XML_document{
	protected $is;
	protected $error; 
		
	protected $dom;
	protected $xpath;
	protected $xpath_namespaces = array(); //namespaces that xpath should be configured to understand; name maps to uri.  If there isn't a prefix, use x or another arbitrary prefix to make PHP happy
	
	//array
	//name -> array( attribute => attribute if we're targeting one (optional), single_value => booolean (default false), xpath_query => valid xpath string, default_value = '', 
	protected $values_to_parse = array();
	
	protected $_raw_values = array();
	protected $_values = array();
		
	
	public function __construct($xml_string){
		$CI = get_instance();
		$this->is = $CI->is;
		$this->error = $CI->error;
		
		if(empty($xml_string) || !$this->is->xml_string($xml_string)){
			log_message('error', 'Invalid XML string provided for new '.get_class($this).' object: '.$this->error->describe($xml_string));
			throw new Exception('Invalid XML string provided for new '.get_class($this).' object');
		}
		
		$string_matches_schema = static::string_matches_schema($xml_string); //makes phpdocumentor happier to put this on a separate line
		if(!$string_matches_schema){
			log_message('error', 'XML string does not match schema for '.get_class($this).' object: '.$this->error->describe($xml_string));
			throw new Exception('XML string does not match schema for '.get_class($this).' object');			
		}
		
		$this->dom = $this->dom_from_xml($xml_string);
		if(!is_a($this->dom, 'DOMDocument')) return $this->error->warning('Unable to create a DOMDocument object from XML string: '.$this->error->describe($xml_string));
		$this->xpath = $this->xpath_from_dom($this->dom);
		if(!is_a($this->xpath, 'DOMXPATH')) return $this->error->warning('Unable to create a DOMXPath object from DOMDocument: '.$this->error->describe($this->dom));
	}
	
	/**
	* Parses a value from the document and returns it without further formatting.
	* Values may need to be further processed, or we may want to apply a default value for empty values, so in general you'll want to use {@link value()} instead.
	* Values must be set up in the [@link values_to_parse} array in order to be parsed.
	* @param string
	* @return string|array
	*/
	public function raw_value($value_to_parse){
		if(!array_key_exists($value_to_parse, $this->values_to_parse)) return $this->error->warning('Cannot parse unknown value: '.$this->error->describe($value_to_parse));
		
		if(!array_key_exists($value_to_parse, $this->_raw_values)){
			$config = $this->values_to_parse[$value_to_parse];

			$method = 'values_for_tag';				
			if(element('single_value', $config))
				$method = replace_first_with('values', 'value', $method);
				
			if(empty($config['attribute'])){
				$this->_raw_values[$value_to_parse] = $this->$method($config['xpath_query']);	
			}else{
				$method = replace_last_with('tag', 'attribute', $method);
				$this->_raw_values[$value_to_parse] = $this->$method($config['attribute'], $config['xpath_query']);
			}							
		}
		return $this->_raw_values[$value_to_parse];
	}	
	
	
	/**
	* Parses a value from the document and returns a formatted version of it.
	* To get an unformatted version of the value, see {@link raw_value()}. If you'd like to supply a default value on a case-by-case basis, supply a non-null value
	* for the second parameter; to add a default value in all cases, add a default_value in the {@link values_to_parse} array.
	* @param string
	* @param mixed
	* @return mixed
	*/
	public function value($value_to_parse, $default_value = NULL){
		if(!array_key_exists($value_to_parse, $this->values_to_parse)) return $this->error->warning('Cannot parse unknown value: '.$this->error->describe($value_to_parse));
		
		$config = $this->values_to_parse[$value_to_parse];
		if(is_null($default_value)) $default_value = element('default_value', $config, '');
		if(method_exists($this, $value_to_parse)) return $this->$value_to_parse($default_value);
				
		$value = $this->raw_value($value_to_parse);
		if(is_scalar($value) && empty($value) && $value !== 0) $value = $default_value;
		return $value;	
	}
		
	public function values( $just_these_values = NULL ){
		if(is_null($just_these_values)) $just_these_values = array_keys($this->values_to_parse);
		if(!is_array($just_these_values)) return $this->error->should_be_an_array();
		
		//if we have all the values we need already, don't parse more of them
		if(empty(array_diff($just_these_values, array_keys($this->_values)))) 
			return array_intersect_key($this->_values, array_flip($just_these_values)); 
		
		//parse any values that haven't already been parsed	
		foreach($just_these_values as $value_to_parse){
			$this->_values[$value_to_parse] = $this->value($value_to_parse);
		}
		
		return array_intersect_key($this->_values, array_flip($just_these_values)); 
	}
	
	public function raw_values( $just_these_values = NULL){
		if(is_null($just_these_values)) $just_these_values = array_keys($this->values_to_parse);
		if(!is_array($just_these_values)) return $this->error->should_be_an_array();
		
		//if we have all the values we need already, don't parse more of them
		if(empty(array_diff($just_these_values, array_keys($this->_raw_values)))) 
			return array_intersect_key($this->_raw_values, array_flip($just_these_values)); 
		
		//parse any values that haven't already been parsed	
		foreach($just_these_values as $value_to_parse){
			$this->_raw_values[$value_to_parse] = $this->raw_value($value_to_parse);
		}
		
		return array_intersect_key($this->_raw_values, array_flip($just_these_values)); 		
	}
	
	/**
	* Creates & configures a PHP DOMDocument object from an XML string.
	* Called by the constructor and should not be called elsewhere; separated into its own function only so that child classes can customize configuration as needed.
	* @param string
	* @return DOMDocument
	*/
	protected function dom_from_xml($xml_string){		
		$dom = new DOMDocument('1.0', 'UTF-8');
		$dom->encoding = 'utf-8';
		$dom->formatOutput = FALSE;
		$dom->preserveWhiteSpace = TRUE;
		$dom->substituteEntities = TRUE;
		$dom->loadXML($xml_string);
		return $dom;
	}
		
	protected function value_for_attribute($attribute, $xpath_for_tag){
		$values = $this->values_for_attribute($attribute, $xpath_for_tag);
		if(!is_array($values)) return false; //we encountered an error
		if(count($values) > 1) $this->error->warning('More than one tag with attribute '.$attribute.' matched '.$xpath_for_tag.'; ignoring all but the first');
		return first_element($values);
	}	
	
	protected function value_for_tag($xpath_for_tag){
		$values = $this->values_for_tag($xpath_for_tag);
		if(!is_array($values)) return false; //we encountered an error
		if(count($values) > 1) $this->error->warning('More than one tag matched '.$xpath_for_tag.'; ignoring all but the first');
		return first_element($values);
	}
	
	protected function values_for_attribute($attribute, $xpath_for_tag){
		if(!$this->is->nonempty_string($attribute)) return $this->error->should_be_a_nonempty_string($attribute);
		if(!$this->is->nonempty_string($xpath_for_tag)) return $this->error->should_be_a_nonempty_string($xpath_for_tag);
	
		$xpath_results = $this->xpath->query($xpath_for_tag);
		
		if(!is_a($xpath_results, 'DOMNodeList') || $xpath_results->length < 1) return array();
		
		$values = array();
		foreach($xpath_results as $xpath_result){					
			if($xpath_result->hasAttribute($attribute)) 
				$values[] = $xpath_result->getAttribute($attribute);
		}
		return $values;
	}
	
	protected function values_for_tag($xpath_for_tag){
		if(!$this->is->nonempty_string($xpath_for_tag)) return $this->error->should_be_a_nonempty_string($xpath_for_tag);
	
		$xpath_results = $this->xpath->query($xpath_for_tag);
		if(!is_a($xpath_results, 'DOMNodeList') || $xpath_results->length < 1) return array();
		
		$values = array();
		foreach($xpath_results as $xpath_result){
			$values[] = $xpath_result->textContent;
		}
	
		return $values;
	}	
	
	/**
	* Creates & configures a PHP DOMXPath object from a DOMDocument object.
	* Called by the constructor and should not be called elsewhere; separated into its own function only so that child classes can customize configuration as needed.
	* @param DOMDocument
	* @return DOMXPath
	*/
	protected function xpath_from_dom($dom){
		$xpath = new DOMXPath($dom);	
		foreach($this->xpath_namespaces as $prefix => $uri){
			$xpath->registerNamespace($prefix, $uri);
		}
		return $xpath;
	}	
	
	function __get($property){
		$raw_property = 'raw_'.$property;	
		
		if(array_key_exists($property, $this->values_to_parse)){
			return $this->value($property);
		}elseif(array_key_exists(strip_from_beginning('raw_', $property), $this->values_to_parse)){
			return $this->raw_value($property);
		}

		get_instance()->error->property_does_not_exist($property, get_class($this), 1);
		return null;
	}		
	
///////////////////////
// STATIC FUNCTIONS
///////////////////////	
	
	//this used to be abstract, PHP is now giving runtime notices saying that's not possible for static methods, 
	//it's possible that this should be some kind of interface instead of an abstract class, but I am too lazy 
	public static function string_matches_schema($xml_string){
		get_instance()->error->warning('This method must be implemented by child classes.');
	}
	
}
?>