<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package vler
* @subpackage libraries
*/ /** */

require_library('object');
require_model('user');

/**
* Identifies the user who is attempting to log in.
*
* This library looks for something that will identify the user who is attempting to log in, and uses it to pull the user entry from the
* database and LDAP.  Child classes will implement the correct way to determine that identifier based on the authentication type being used 
* (CAC, PIV, VA SSO, etc.).  The identifier for the user is referred to throughout this class as the organization id.  
*
* Child classes must:
* * Determine the user's organization id and set the _organization_id variable if the organization id is valid and the user has the correct credentials for their organization
* * Set the _user variable if they are able to look up the user based on the organization id.
*
* Once we have the user object, most other checks and operations needed to sign in or sign out will take place on the User object.  This library exists primarily
* to identify the user.
*
* @package vler
* @subpackage libraries
*/
abstract class Authentication extends Object {
	protected $type; //the type of authentication that is set based on the child class being called
	protected $_email;
	protected $_given_name;
	protected $_middle_name;
	protected $_family_name;
	protected $_suffix;
	
	protected $_organization_id; //set as the unique identifier for the user
	
	protected $_user; //the user object that was established after successful authentication 
	
	protected $_destination_after_login; //url that the user should be directed to after the user has been authenticated.
	protected $_url_for_registration_page; //url that the user will be directed to if registration is needed. 
	protected $_url_for_logout; //url that will log the user out

	/**
	* True if we're able to determine the user's organization id and verify that it belongs to them.
	*
	* Note that it is the responsibility of the child class to make sure that the organization id is only set if their certs/mechanism of authentication is in order.
	*
	* @return boolean
	*/
	function is_valid(){
		return !$this->property_is_empty('organization_id');
	}
	
	////////////////////////
	// GETTERS
	/////////////////////////
	
	
	abstract protected function given_name();
	abstract protected function middle_name();
	abstract protected function family_name();
	abstract protected function suffix();
	abstract protected function user();
	
	/**
	* @return string URL
	*/
	function destination_after_login(){
		if(!isset($this->_destination_after_login)){
			$CI = get_instance();
			if(empty($CI->session->flashdata('uri_before_logout')) || $CI->session->flashdata('uri_before_logout') == site_url('auth/logout')){
				$this->_destination_after_login = site_url('inbox');
			}else{
				$CI->session->keep_flashdata('uri_before_logout');
				$this->_destination_after_login = $CI->session->flashdata('uri_before_logout');
			}
		}
		
		return $this->_destination_after_login; 
	}
	
	/**
	* @return string URL
	*/
	function url_for_registration_page(){
		if(!isset($this->_url_for_registration_page)) 
			$this->_url_for_registration_page = site_url('registration');
		return $this->_url_for_registration_page;
	}	

}
