<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package vler
* @subpackage libraries
*/ /** */

/**
* A standardized library to make calls to the API service.
* Developers originally wrote their own CURL calls throughout the application whenever they needed to access the API.  By standardizing
*  our calls and using this library instead, we're able to reduce duplicate code, make it easier to make improvements to how we access 
* the API, and include helpful debugging tools like including the API calls in the CI profiler or logging data about API calls to a CSV
*  for performance debugging.  
*
* Debugging tips for API calls:
* - To view profiler data, turn on the Codeigniter Profiler
* - To log data about API calls, set ENABLE_PERFDNS   E_LOG to true in your .ini files.
* - To write the actual response data to the logs while debugging, look for the is_on_dev() check in call() and (temporarily) uncomment 
    that code.  (Don't do this outside your local environment - it'll fill up the log files very quickly and could reveal PII/PHI.)
*
* Keep in mind that there are very likely old API calls still in the code that were written before this library existed and therefore
* will not show up in any of these logs.  You should be able to find them easily by searching for CURL calls; update them to use this 
* library as time allows, especially if you need better debugging information about that API call.
* 
* Note that this library is similar to the Table library in that CI will run a singleton instance of this library and you'll need to
* run the {@link clear()} method between calls.  If you're seeing unexpected behavior, check to make sure that you've run the clear
* method before making a new call.
* 
* @package vler
* @subpackage libraries
*/ 
class API {
	protected $CI;
	var $public_key = WEBSERVICE_PUBLIC_KEY;
	var $private_key = WEBSERVICE_PRIVATE_KEY;
	
	var $date; //if left unset, will be set when call() is called
	var $http_method = 'GET'; //GET, POST, etc.
	var $resource; //the URL to send to: e.g., /direct/send

	var $data = array(); //vars to send via post
	var $files_to_send = array(); //separate array of files to send
	var $http_status;
	var $raw_output;
	var $format = 'json';
	var $perfDNS   e_log = array();
	var $profiler_log = array(); 
	
	//For performance and profiler data, track when the call starts and when we receive the response. 
	//We used to do this via benchmark, but there's no real advantage of doing so; now that we're using benchmark to track perfDNS   e, better to keep this internal
	protected $service_call_start;
	protected $service_call_start_standardized_timezone; 
	protected $service_call_end;
	
	//vars preceded by underscore are for internal use
	//don't call these!  call the method with the same name instead.
	protected $_authorization_header; 
	protected $_boundary;
	protected $_content_type = 'application/x-www-form-urlencoded'; //multipart/form-data, etc.	
	protected $_headers;
	protected $_formatted_output;
	protected $_output_as_array;
	protected $_output;
	protected $_target_url;

	
	public function __construct(){
		$this->CI = get_instance();
		$this->CI->load->helper('date');
	}	 
	
	
	/** 
	* Giant monster method that actually makes the API call.
	* Most other methods in this class are helper methods to this one, or which can be used to pull information about the response to debug as needed.
	* Typically, you'll only need to call this method and your output method of choice.  
	* @return boolean True if the call was successful
	*/
	public function call($resource = null, $data = null, $http_method = null){
		if(!is_null($resource)) $this->resource = $resource;
		if(!is_null($data))$this->data = array_merge($this->data, $data);
		if(!is_null($http_method)) $this->http_method = $http_method;
		
		//verify that any files that we're sending are being sent correctly
		if(!empty($this->files_to_send) && $this->http_method != 'POST'){
			$this->CI->error->warning('Files must be sent via POST. Files ('.array_to_human_readable_list(collect('filename', $this->files_to_send)).') will not be sent to '.$this->target_url());	
		}	
	
		if(!isset($this->date)) $this->date = now();
		$this->resource = strip_from_end('/', $this->resource);
		
		$this->format = strtolower($this->format);
		if($this->format != 'xml' && !string_contains('/format/', $this->resource))
			$this->resource .= '/format/'.$this->format;	
		
		$this->service_call_start = microtime(TRUE);
		$this->service_call_start_standardized_timezone = now(PERFDNS   E_LOG_TIMEZONE); //make sure that we've got the timestamp in a standardized timezone for display purposes	
		
		# TODO -- ADD SOME VALIDATION OF DATA, FIGURE OUT IF YOU HAVE POST DATA, TRIGGER WARNING 
		$headers = $this->headers();
		if(empty($headers) || !is_array($headers)){
			trigger_error("I can't perform the service call without valid headers", E_USER_WARNING);
			return false;
		}  
		
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->target_url());
		if($this->http_method == 'POST'){
	        curl_setopt($ch, CURLOPT_POST,1);
			curl_setopt($ch, CURLOPT_POSTFIELDS, $this->data_for_post());
		}
		else if($this->http_method == 'DELETE' || $this->http_method == 'PUT') {
			 curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $this->http_method);
			 if(!empty($this->data_for_post())) { 
			 	curl_setopt($ch, CURLOPT_POSTFIELDS, $this->data_for_post()); 
			 }
		}
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($ch, CURLOPT_HTTPHEADER,$this->headers());

		$this->raw_output = curl_exec($ch);
		if($this->raw_output === FALSE){
			trigger_error(curl_error($ch), E_USER_ERROR);
			$this->format_errors[] = 'cURL call failed';
		}
		$this->http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);		
		curl_close($ch);
		
		//trigger a warning if the call didn't work.  note that 403 is an application attempting to do something they're unauthorized; developers debugging unexpected behavior may want to show the error for 403s
		

		if($this->http_status >= 200 && $this->http_status < 300){
			//all 200 codes indicate success, but some of them are less definitive than others (eg 202)
			//at the time of this writing, we only expect 200 and 201 responses from the api, and other status codes deserve the developer's attention
			//if that changes in the future, modify this code to ensure this notice is only triggered when we're getting unexpected behavior -- MG 2016-08-17
			if($this->http_status > 201) $this->error->notice('Unexpected http status code: '.$this->http_status.' while accessing '.$this->target_url().' via '.$this->http_method.'.  We expected 200 or 201.');
		}elseif($this->http_status != '403'){ 
			get_instance()->error->warning(implode_nonempty('; ', array('Error '.$this->http_status.' while accessing '.$this->target_url().' via '.$this->http_method, $this->message())));
		}
		
		$this->raw_output = trim($this->raw_output);
		if(empty($this->raw_output)){
			$this->CI->error->warning('No output for API call: '.$this->target_url().' with data '.$this->CI->error->describe($this->data));	
			$this->format_errors[] = 'No output';
		}
		
		if($this->format == 'xml'){
			//check xml for formatting errors
			libxml_use_internal_errors(true);
			$xml = simplexml_load_string(trim($this->raw_output), 'SimpleXMLElement', LIBXML_NOCDATA);
			if(empty($xml) && !empty($this->raw_output)){ 
				$this->format_errors = collect('message', libxml_get_errors()); 
				foreach($this->format_errors as $error_message){
					$this->CI->error->warning('XML parse error for API call '.$this->target_url().': '.trim($error_message).'.  Data for API call: '.$this->CI->error->describe($this->data));	
				}
			}
			libxml_use_internal_errors(false);
		}
		
		if($this->format == 'json'){
			$json = json_decode(trim($this->raw_output)); //rare case when we don't use $this->json->decode() - we want to trigger the error ourselves
			if(json_last_error() != JSON_ERROR_NONE){
				
				$error_message = 'JSON parse error for API call '.$this->target_url().': '.json_last_error_msg().'. ';		
#				if(is_on_dev())
#					$error_message .= "<br>\n\n Response: ".$this->raw_output."<br>\n\n"; //only do this during testing because it fills the logs up pretty fast
				$error_message .= 'Data for API call: '.$this->CI->error->describe($this->data);
				$this->CI->error->warning($error_message);	
				$this->format_errors = array($error_message);
			}	
		}
		
		$this->service_call_end = microtime(TRUE);
		$this->add_call_to_perfDNS   e_log();
		$this->add_call_to_profile_log();	
		return ($this->http_status >= 200 && $this->http_status < 300);
	}
	
	/**
	* Restores class variables to their default values.  
	* This should be called when we're totally done with an API call and have grabbed any data that we're going to need from the response.
	* To preserve a value, add it to the $vars_to_preserve array in this method.
	*/
	function clear(){
		$vars_to_preserve = array('CI', 'public_key', 'private_key', 'perfDNS   e_log', 'profiler_log');	
		foreach(get_class_vars(get_class($this)) as $var => $value){
			if(!in_array($var, $vars_to_preserve)){
				$this->$var = $value; 
			}
		}	
	}

	/** 
	* Formats the output from the API with an eye to human legibility (for instance, adding whitespace).
	* @return string 
	*/
	public function formatted_output(){
		if(isset($this->raw_output) && !isset($this->_formatted_output)){
			if(empty($this->format_errors) && $this->format == 'xml'){
				//format the output
				$dom = new DOMDocument;		
				$dom->preserveWhiteSpace = FALSE;
				$dom->loadXML($this->raw_output);
				$dom->formatOutput = TRUE;
				$this->_formatted_output = $dom->saveXml();	
			}else{
				$this->_formatted_output = $this->CI->json->encode($this->output_as_array(), JSON_PRETTY_PRINT);
			}
		}
		return $this->_formatted_output;
	}			

	/** 
	* Grab the message from the API service's response to our call.
	* This is most frequently used for grabbing error messages from the API output.  May be empty if there wasn't a top-level value called "message" in the response
	* @return string
	*/
	public function message(){
		if(is_array($this->output_as_array()))
			return element('message', $this->output_as_array());
	}	

	/**
	* Formats the raw output of the API call as a generic object.  
	* Recommend generally using {@link output_as_array()} instead, objects that are stdObjects instead of a class with methods are not often that useful.
	* @return object
	*/
	public function output(){
		if(isset($this->raw_output) && !isset($this->_output)){
			if(empty($this->format_errors)){				
				if($this->format == 'xml'){				
					$this->_output = $this->CI->json->decode($this->CI->json->encode(simplexml_load_string(trim($this->raw_output), 'SimpleXMLElement', LIBXML_NOCDATA)));
				}elseif($this->format == 'json'){
					$this->_output = $this->CI->json->decode(trim($this->raw_output));
				}
			}else{
				$this->_output = (object)array();
			}	
		}
		return $this->_output;
	}

	/** 
	* Formats the raw output of the API call as an array, since that's generally the most useful format for this sort of data. 
	* @return array
	*/	
	public function output_as_array(){
		if(isset($this->raw_output) && !isset($this->_output_as_array)){
			if(empty($this->format_errors)){
				if($this->format == 'xml'){									
					$this->_output_as_array = $this->CI->json->decode($this->CI->json->encode(simplexml_load_string(trim($this->raw_output), 'SimpleXMLElement', LIBXML_NOCDATA)), TRUE);
				}elseif($this->format == 'json'){
					$this->_output_as_array = $this->CI->json->decode(trim($this->raw_output), TRUE);
				}
			}else{
				$this->_output_as_array = array();
			}	
		}
		return $this->_output_as_array;
	}

	/**
	* The id of the last request call - this will correspond to the api's request table, which logs calls made to the API.  This can be useful for debugging.
	* @return int
	*/
	public function request_id(){
		return element('request_id', $this->output_as_array());
	}
	
	/**
	* The API resource without including the domain, but still including the query string.  This can be useful for debugging.
	* @return string
	*/
	public function resource_with_query_string(){
		return strip_from_beginning(WEBSERVICE_URL, $this->target_url());
	}
	
	/**
	* The full URL that we'll be reaching out to, including any GET data as a query string.
	*/
	public function target_url(){
		if(!isset($this->_target_url)){
			//MG, resist the urge to use site_url(), you're accessing a different site, it will only cause sorrow and woe. - MG  
			if(!empty($this->resource) && !string_begins_with('/', $this->resource)) $this->resource = '/'.$this->resource;	 		
			$this->_target_url = WEBSERVICE_URL . $this->resource;
			if($this->http_method == 'GET' && !empty($this->data)){
				$this->_target_url .= '?'.raw_http_build_query($this->data);
			}
		}
		return $this->_target_url;
	}
	
	/** 
	* Creates a CSV compilation of the API calls made.  Note that this does not include GET or POST data or received output so that we can avoid PHI/PII. 
	* The data for this CSV will only have been logged if ENABLE_PERFDNS   E_LOG is true.
	*/
	public function write_perfDNS   e_log_to_csv(){
		if(empty($this->perfDNS   e_log)) return;
		
		$this->CI->load->helper('file');
		$directory = APPPATH.'logs/api_calls';
		$csv_file = $directory.'/api_calls_'.strftime("%Y-%m-%d", now(PERFDNS   E_LOG_TIMEZONE)).'.csv';
		if(!directory_exists($directory)) mkdir($directory);
		if(!file_exists($csv_file)) write_file($csv_file, implode(',', array_map('humanize', array_keys(first_element($this->perfDNS   e_log)))));
		
		//at the moment, undecided whether it's better to keep opening and closing the file or to load all the calls into memory twice to store as a string - make up your mind later
		foreach($this->perfDNS   e_log as $csv_data)
			write_file($csv_file, "\n".implode(',', $csv_data), 'a+');  
	}	
	
///////////////////////////////
// PROTECTED HELPER METHODS
////////////////////////////////

	/**
	* Generates the header for the API call that the APi service will used to verify that we're authorized to make this call.
	* @return string
	*/
	protected function authorization_header(){
		if(!isset($this->_authorization_header)){
			$this->http_method = strtoupper($this->http_method);
			$known_http_methods = array('GET', 'POST', 'PUT', 'DELETE');
			if(empty($this->http_method) || !in_array($this->http_method, $known_http_methods)){
				trigger_error('I expected a known http_method, but you gave me '.$this->http_method ); 
				return false;
			}			
			if(empty($this->date)) { trigger_error("I can't create an authorization header without a date"); return false; }
			if(empty($this->content_type())) { trigger_error("I can't create an authorization header without a content_type"); return false; }
			if(empty($this->resource)) { trigger_error("I can't create an authorization header without a resource"); return false; }
			
			if(empty($this->public_key)) { trigger_error("I can't create an authorization header without a public key"); return false; }
			if(empty($this->private_key)) { trigger_error("I can't create an authorization header without a private key"); return false; }  		
			
					
			$components = array($this->http_method,
								$this->date,
								//note - the auth header wants something like "multipart/form-data" or similar to be the content type, but the content type may have additional
								//components (boundary, charset, etc.).  This isn't the most effective way to get those, but it works for now
								substr($this->content_type(), 0, strpos($this->content_type(), ';')),//for the auth header, we only want the multipart/form-data or similar component.
								$this->resource_with_query_string());
											
			
			$this->_authorization_header =  'DAAS ' . $this->public_key . ':' . base64_encode(hash_hmac('sha256', implode("\n", $components), $this->private_key));
		}
		return $this->_authorization_header;
	}
	
	/**
	* Generates the string used as a boundary between different sections of the string that will become our call to the API service.
	* @return string
	*/
	protected function boundary(){
		if(!isset($this->_boundary)){
			return hash('sha256',(time()));
		}
		return $this->_boundary;
	}	
	
	/**
	* Generates the string that will indicate our content type to the API service.
	* @return string
	*/
	protected function content_type(){
		if($this->http_method == 'POST' && !empty($this->files_to_send)){
			return 'multipart/form-data; charset=utf-8; boundary='.$this->boundary();
		}
		
		if(!string_contains('charset', $this->_content_type))
			$this->_content_type  .= '; charset=utf-8;';
		
		return $this->_content_type;
	}	
	
	/**
	* Converts the data array for this call to a string that the API service will understand.
	* @return string
	*/
	protected function data_for_post(){		
		//if there aren't any files, we can piece together the data pretty simply
		if(empty($this->files_to_send) || !string_begins_with('multipart/form-data', $this->content_type())) return raw_http_build_query($this->data); 
		
		//if there are files, we need to do a more complicated processing of the data
		$post = '';
		$post .= '--'.$this->boundary()."\r\n";
		foreach($this->data as $name => $value) {
			$post .= 'Content-Disposition: form-data; name="'.$name.'"'."\r\n\r\n";
			$post .= $value."\r\n";
			$post .= '--'.$this->boundary()."\r\n";
		}
		$i = 0;
		foreach($this->files_to_send as $key => &$file) {
			$post .= 'Content-Disposition: form-data; name="'.$key.'"; filename="'.$file['filename'].'"'."\r\n";
			$post .= 'Content-Type: application/octet-stream'."\r\n";
			$post .= 'Content-Transfer-Encoding: binary'."\r\n\r\n";
			$post .= $file['binary_string']."\r\n";
			if($i === count($this->files_to_send)) { $post .= '--'.$this->boundary()."--\r\n"; } else { $post .= '--'.$this->boundary()."\r\n"; }
			$i++;
		}
/*		if(is_on_dev()){ //for testing duplicate attachment names in the API
			for($i=1; $i < 3; $i++){
				foreach($this->files_to_send as $key => $file) {
					$key .= $i;
					$post .= 'Content-Disposition: form-data; name="'.$key.'"; filename="'.$file['filename'].'"'."\r\n";
					$post .= 'Content-Type: application/octet-stream'."\r\n";
					$post .= 'Content-Transfer-Encoding: binary'."\r\n\r\n";
					$post .= $file['binary_string']."\r\n";
					if($i === count($this->files_to_send)) { $post .= '--'.$this->boundary()."--\r\n"; } else { $post .= '--'.$this->boundary()."\r\n"; }
					$i++;
				}
			}
		} */
				
		
		return $post;
	}
	
	 
	/** 
	* Generates the headers for this API call, including the authorization header.
	* @todo Consider including the optional content MD5 header
	* @return array
	*/
	protected function headers(){
		if(!isset($this->_headers)){	
			$authorization_header = $this->authorization_header();
			if(empty($authorization_header)){
				trigger_error("I can't generate headers without a valid authorization header", E_USER_WARNING);
				return false;
			}
		
			$this->_headers = array( 'Authorization: '.$authorization_header,
									 'Date: '.$this->date,
									 'Content-Type: '.$this->content_type());
		}
		return $this->_headers;				  
	}	
	
	
	/** 
	* Add an entry to the log of API calls made that will be used by {@link write_perfDNS   e_log_to_csv()}.  Note that this does not include GET or POST data or received output so that we can avoid PHI/PII.
	*/
	protected function add_call_to_perfDNS   e_log(){
		if(defined('ENABLE_PERFDNS   E_LOG') and ENABLE_PERFDNS   E_LOG){
			
			$csv_data = array('User ID' => ''); //this makes sure our data doesn't get misaligned in the CSV even when the id isn't available (e.g. login/logout process, registration, public places in the application)
			if(class_exists('User') && method_exists('User', 'find_from_session')){
				$user = User::find_from_session();  //this won't cause an actual user lookup unless we're hitting this before the user is stored on $CI.  Could be an issue for logouts/logins - check
				if(User::is_an_entity($user)){
					$csv_data['User ID'] = $user->id();
				}
			}
			
			$csv_data = array_merge( $csv_data,
									 array('session_id' => session_id(),
										  'called_from' => current_url(),
										  'called_at' =>  strftime("%Y-%m-%d %H:%M:%S", $this->service_call_start_standardized_timezone), 
										  'time' => round($this->service_call_end - $this->service_call_start, 4),
										  'target_resource' => WEBSERVICE_URL.$this->resource,
										  'http_status' => $this->http_status,
										  'http_method' => $this->http_method,
										  'format' => strtoupper($this->format),
										  'data_sent_in_bytes' => '', //this is a little complicated, so we'll sort it out further down
										  'data_received_in_bytes' => string_length_in_bytes($this->raw_output)));
										  
			if($this->http_method == 'GET')
				$csv_data['data_sent_in_bytes'] = string_length_in_bytes(raw_http_build_query($this->data));
			
			if($this->http_method == 'POST')
				$csv_data['data_sent_in_bytes'] = string_length_in_bytes($this->data_for_post());
				
														   
			$this->perfDNS   e_log[] = $csv_data;
		}
	}
	
	/**
	* Stores data about the most recent API call so that we can include it in the CI Profiler.  Note that we won't store this data if the profiler isn't enabled.
	*/
	protected function add_call_to_profile_log(){		
		//Add this service call to the profiler
		if($this->CI->output->profiler_is_enabled()){
			$log_entry = array('time' => number_format($this->service_call_end - $this->service_call_start, 4),
							   'http_method' => $this->http_method,       
							   'http_status' => $this->http_status,
							   'format' => strtoupper($this->format),
							   'data' => $this->data,
							   'url' => $this->target_url(),
							   'output' => $this->formatted_output() );
							   
			if(!empty($this->format_errors))
				$log_entry['output'] = $this->raw_output;							   
										   
			$backtrace = debug_backtrace();
			$query_backtrace = array();
			foreach($backtrace as $row => &$data){
				unset($data['object']);	//not necessary, just makes it easier to see things when you're debugging
				unset($data['args']);
				if(!isset($data['file'])) $data['file'] = false;
						
				$data['function_for_display'] = $data['function'];
				if(!empty($data['class'])){
					if($data['class'] == 'Entity' &&  $data['file'] != str_replace('/', '\\', APPPATH.'models/entity.php')){
						$data['class'] = ucfirst(strip_from_end('.php', strip_from_beginning(str_replace('/', '\\', APPPATH.'models/'), $data['file'])));
					}					
					$data['function_for_display'] = $data['class'].$data['type'].$data['function'];
				}
				
				$data['short_file'] = '';
				if(is_string($data['file'])){
					$last_slash = strrpos($data['file'], '/');
					$data['short_file'] = '..'.substr($data['file'], $last_slash);
				}
				$query_backtrace[] = $data;
			}	
			
			$files_to_ignore = array( /*BASEPATH.'database/DB_active_rec.php', BASEPATH.'database/original_DB_driver.php', APPPATH.'libraries/Active_record_model.php', APPPATH.'models/entity.php' */ );
			foreach($files_to_ignore as $file_to_ignore) $files_to_ignore[] = str_replace('/', '\\', $file_to_ignore); //windows uses weird slashes just to annoy us
			foreach($backtrace as $row => $data){
				if(!in_array($data['file'], $files_to_ignore) && array_key_exists('file', $data) && array_key_exists('line', $data)){			
					$file = $data['file'];
					$line = $data['line'];
					$short_file = $data['short_file'];
					$function = $data['class'].$data['type'].$data['function'];
					break;
				}
			}
			$log_entry['backtrace'] = array('line' => $line, 'file' => $file, 'short_file' => $short_file, 'backtrace' => $query_backtrace, 'function' => $function); 	
			$this->profiler_log[] = $log_entry;							   
		}	
	}
}
