<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models-user
*/

require_model('mailbox');

/**
* @package direct-as-a-service
* @subpackage models
*/
class Group extends Mailbox {
	
	protected $_admin_ldap_connection;
	protected $_ldap_connection;
	protected $_ldap_values;

	protected $_readonly_fields = array( 'is_group' ); 
	

////////////////////////////////
// INSTANCE METHODS
////////////////////////////////
	
	public function activate(){
		if($this->is_active) return true; //if this ist active, we're already done		
		
		$mailbox = $this->mailbox;
		$mailbox_activated = false;
		if(Mailbox::is_an_entity($mailbox) && !$mailbox->is_active){
			$mailbox->is_active = true;
			$success = $mailbox_activated = $mailbox->save();
			if(!$success) return $this->error->warning('Cannot activate '.$this->describe().'; '.$mailbox->describe.' could not be activated.');
		}
		
		$this->_ldap_values = null; //force a reload of our ldap information when we're done
		
		if(!is_resource($this->admin_ldap_connection))return $this->error->warning('Cannot activate '.$this->describe().' without an LDAP connection');			
		$success = ldap_rename($this->admin_ldap_connection, $this->disabled_dn,'uid=' . $this->name, LDAP_ACCOUNT_GROUP, TRUE);
		if(!$success){
			$this->error->warning('Unable to activate '.$this->describe().': LDAP says '.$this->error->describe(ldap_error($this->admin_ldap_connection)));
			if($mailbox_activated){
				$mailbox->is_active = false;
				if(!$mailbox->save()) $this->error->warning('Unable to deactivate '.$mailbox->describe().'; please manually update the database.');
			}
		}
		return $success;
	}	
	
	public function deactivate(){
		if(!$this->is_active) return true; //if this isn't active, we're already done		
		
		$mailbox_deactivated = false;
		$mailbox = $this->mailbox;
		if(Mailbox::is_an_entity($mailbox) && $mailbox->is_active){
			$mailbox->is_active = false;
			$success = $mailbox_deactivated = $mailbox->save();
			if(!$success) return $this->error->warning('Cannot deactivate '.$this->describe().'; '.$mailbox->describe.' could not be deactivated.');
		}
		
		$this->_ldap_values = null; //force a reload of our ldap information when we're done
		
		if(!is_resource($this->ldap_connection))return $this->error->warning('Cannot deactivate '.$this->describe().' without an LDAP connection');			
		$success = ldap_rename($this->ldap_connection, $this->dn(),'uid=' . $this->name,LDAP_DISABLED_ACCOUNT_GROUP, TRUE);
		if(!$success){
			$this->error->warning('Unable to deactivate '.$this->describe().': LDAP says '.$this->error->describe(ldap_error($this->ldap_connection)));
			if($mailbox_deactivated){
				$mailbox->is_active = true;
				if(!$mailbox->save()) $this->error->warning('Unable to reactivate '.$mailbox->describe().'; please manually update the database.');
			}
		}
		return $success;
	}	
	
	
////////////////////////////////
// GETTERS
//////////////////////////////

	public function application_names(){
		$entries = ldap_get_entries($this->ldap_connection(), ldap_search($this->ldap_connection(), LDAP_BASE_RDN, '(&(member='.$this->dn().'))', array('cn')));
		
		$display_names = array();
		for($i = 0; $i < $entries['count']; $i++) { 
			$display_names[] = $entries[$i]['cn'][0];
		}
		
		return $display_names;
	}

	public function display_name(){
		return element('cn', $this->ldap_values());
	}

	public function dn(){
		return 'ou='.$this->name.','.LDAP_GROUPS_GROUP;
	}
	
	public function disabled_dn(){
		return 'ou='.$this->name.','.LDAP_DISABLED_GROUPS_GROUP;
	}
	

	public function admin_ldap_connection() {
		if(!isset($this->_admin_ldap_connection)){
			$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
			if($ldap_conn === FALSE) return $this->error->warning('Could not connect to LDAP host: '.$this->error->describe(LDAP_HOSTNAME));
			
			if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
			if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
			
			//get the currently logged user's password from the database to bind to LDAP (NOT the user for this object);
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$this->_admin_ldap_connection = $ldap_conn;
		}
		return $this->_admin_ldap_connection;
	}	
	
	public function ldap_connection() {
		if(!isset($this->_ldap_connection)){
			$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
			if($ldap_conn === FALSE) return $this->error->warning('Could not connect to LDAP host: '.$this->error->describe(LDAP_HOSTNAME));
			
			if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
			if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
			
			//get the currently logged user's password from the database to bind to LDAP (NOT the user for this object);
			$logged_in_user = User::find_from_session();	
			if(!User::is_an_entity($logged_in_user) && in_api_context()) return $this->admin_ldap_connection();		
			$ldap_bind = ldap_bind($ldap_conn, $logged_in_user->dn, get_instance()->encrypt->decode($logged_in_user->user_ep));				
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$this->_ldap_connection = $ldap_conn;
		}
		return $this->_ldap_connection;
	}	
	
	function ldap_values(){
		if(!isset($this->_ldap_values) && !$this->property_is_empty('name')){
			$ldap_fields = array('cn', 'description');		
			$ldap_entries = ldap_first_entry($this->ldap_connection(), 
											 ldap_search($this->ldap_connection(), 
														 $this->is_active? LDAP_GROUPS_GROUP : LDAP_DISABLED_GROUPS_GROUP, 
														 '(&(ou='.$this->name.')(cn=*))', 
														 $ldap_fields));
			
			if($ldap_entries){
				$entry = ldap_get_attributes($this->ldap_connection(), $ldap_entries);
				foreach($ldap_fields as $field){
					$this->_ldap_values[$field] = $entry[$field][0];
				}
			}
		}		
		return $this->_ldap_values;		
	}
	
	public function readonly_fields(){
		$readonly_fields = $this->_merge_with_parent_array('readonly_fields');
		if(isset($this->id)){
			$readonly_fields[] = 'name';
		}
		
		return $readonly_fields;
    }
	
	public function values($just_these_values=null){
		if(is_null($just_these_values)){
			return array_merge($this->ldap_values(), parent::values());
		}
		return parent::values($just_these_values);
	}
	
//////////////////////////////////////
// SETTERS
//////////////////////////////////////	



	
//////////////////////////////////////
// DATA MANAGEMENT
//////////////////////////////////////

	protected function _run_before_create(){
		$success = $this->_set_field_value('user_created_time', time(), $error_offset = 0, $override_validation = TRUE);
		return $success;
	}
	

#TODO - ADD LDAP CODE HERE INSTEAD OF NEEDING TO RUN USERSMODEL::CREATE_USER
	protected function _run_after_create(){

		return true;
	} 
	
/////////////////////////////////////////
// PHP MAGICAL FUNCTIONS
////////////////////////////////////////	

    function __get($property){		
        $indirect_property = '_'.$property; 
		if(!method_exists(get_class($this), $property) && !property_exists(get_class($this), $indirect_property) 
			&& !property_exists(get_class($this), $property) && !array_key_exists($property, $this->_values)
			&& array_key_exists($property, $this->ldap_values())){
			return $this->_ldap_values[$property];
		}
        return parent::__get($property);
    }
	

/////////////////////////////////
// STATIC METHODS
/////////////////////////////////

	public static function count($id_or_conditions = array()){
        static::db()->where('is_group', true);
        return parent::count($id_or_conditions);
    }

	public static function find($id_or_conditions = array(), $key_by = null){
		static::db()->where('is_group', true);
		return parent::find($id_or_conditions, $key_by);
	}

	
}
