<?php
/**
* @package direct-as-a-service
* @subpackage libraries
*/

/**
* Format class
*
* Help convert between various formats such as XML, JSON, CSV, etc.
*
* @author Phil Sturgeon
* @license		http://philsturgeon.co.uk/code/dbad-license
*
* @package direct-as-a-service
* @subpackage libraries
*/
class Format {

	// Array to convert
	protected $_data = array();

	// View filename
	protected $_from_type = null;

	/**
	 * Returns an instance of the Format object.
	 *
	 *     echo $this->format->factory(array('foo' => 'bar'))->to_xml();
	 *
	 * @param   mixed  general date to be converted
	 * @param   string  data format the file was provided in
	 * @return  Factory
	 */
	public function factory($data, $from_type = null)
	{
		// Stupid stuff to emulate the "new static()" stuff in this libraries PHP 5.3 equivalent
		$class = __CLASS__;
		return new $class($data, $from_type);
	}

	/**
	 * Do not use this directly, call factory()
	 */
	public function __construct($data = null, $from_type = null)
	{
		get_instance()->load->helper('inflector');

		// If the provided data is already formatted we should probably convert it to an array
		if ($from_type !== null)
		{
			if (method_exists($this, '_from_' . $from_type))
			{
				$data = call_user_func(array($this, '_from_' . $from_type), $data);
			}

			else
			{
				throw new Exception('Format class does not support conversion from "' . $from_type . '".');
			}
		}

		$this->_data = $data;
	}

	// FORMATING OUTPUT ---------------------------------------------------------

	public function to_array($data = null)
	{
		// If not just null, but nothing is provided
		if ($data === null and ! func_num_args())
		{
			$data = $this->_data;
		}

		$array = array();

		foreach ((array) $data as $key => $value)
		{
			if (is_object($value) or is_array($value))
			{
				$array[$key] = $this->to_array($value);
			}

			else
			{
				$array[$key] = $value;
			}
		}

		return $array;
	}

	public function to_xml($data = null, $structure = null, $basenode = 'xml')
	{
		if ($data === null and ! func_num_args())
		{
			$data = $this->_data;
		}

		// turn off compatibility mode as simple xml throws a wobbly if you don't.
		if (ini_get('zend.ze1_compatibility_mode') == 1)
		{
			ini_set('zend.ze1_compatibility_mode', 0);
		}

		$return_as_string = false;
		if ($structure === null){
			$return_as_string = true; //make sure that we return the results of this function as a string (recursive calls will return a structure)
			
			//set up the dom document
			$this->dom = new DOMDocument('1.0', 'UTF-8');
			$this->dom->encoding = 'utf-8';
	
			#$this->dom = DomDocument::loadXML($xml);
			$this->dom->formatOutput = FALSE;
			$this->dom->preserveWhiteSpace = TRUE;
			$this->dom->substituteEntities = TRUE;
			
			//set up the structure var			
			$structure = $this->dom->appendChild($this->dom->createElement($basenode));
			#$structure = $this->dom->getElementsByTagName($basenode)->item(0);
		}

		// Force it to be something useful
		if ( ! is_array($data) AND ! is_object($data)){
			$data = (array) $data;
		}

		foreach ($data as $key => $value){

			//change false/true to 0/1
			if(is_bool($value)){
				$value = (int) $value;
			}

			// no numeric keys in our xml please!
			if (is_numeric($key)){
				// make string key...
				$key = (singular($basenode) != $basenode) ? singular($basenode) : 'item';
			}

			// replace anything not alpha numeric
			//$key = preg_replace('/[^a-z_\-0-9]/i', '', $key); 
			$original_key = $key;
			$key = preg_replace('/[^\p{L}_\-0-9]/u', '', $key); //make sure that this allows UTF-8 characters -- MG 2014-05-22
			if(empty($key)) get_instance()->error->warning('Invalid XML key: '.get_instance()->error->describe($original_key)); //make sure that we know if we end up with an empty key -- MG 2014-05-22

			$node = $this->dom->createElement($key);
			
			// if there is another array found recursively call this function
			if (is_array($value) || is_object($value)){
				$node = $this->to_xml($value, $node, $key); //recursive call
			}else{
				//protect HTML and special chars by putting them in in CDATA instead of having the parser convert them to entities
				//from a strictly-XML standpoint, it makes no difference whether it's in CDATA or converted to entities; from a PHP perspective, we're going to have fewer charset issues 
				//if we don't have escape/un-escape html entities. -- MG 2014-05-23
				if(!empty($value) && is_string($value) && (string_contains('<', $value) || string_contains('>', $value) || string_contains('&', $value) || string_contains("'", $value) || string_contains("\r\n", $value)))
					$node->appendChild($this->dom->createCDATASection($value)); 
				else
					$node->appendChild($this->dom->createTextNode($value));
				
			}
			$structure->appendChild($node);
			
		}

		if($return_as_string){ 
			$xml = '<?xml version="1.0" encoding="utf-8"?>'.$this->dom->saveXML($structure);
			return $xml;
		}
		return $structure;
	}

	// Format HTML for output
	public function to_html()
	{
		$data = $this->_data;

		// Multi-dimensional array
		if (isset($data[0]) && is_array($data[0]))
		{
			$headings = array_keys($data[0]);
		}

		// Single array
		else
		{
			$headings = array_keys($data);
			$data = array($data);
		}

		$ci = get_instance();
		$ci->load->library('table');

		$ci->table->set_heading($headings);

		foreach ($data as &$row)
		{
			$ci->table->add_row($row);
		}

		return $ci->table->generate();
	}

	// Format CSV for output
	public function to_csv()
	{
		$data = $this->_data;

		// Multi-dimensional array
		if (isset($data[0]) && is_array($data[0]))
		{
			$headings = array_keys($data[0]);
		}

		// Single array
		else
		{
			$headings = array_keys($data);
			$data = array($data);
		}

		$output = implode(',', $headings).PHP_EOL;
		foreach ($data as &$row)
		{
			$output .= '"'.implode('","', $row).'"'.PHP_EOL;
		}

		return $output;
	}

	// Encode as JSON
	public function to_json()
	{
		return get_instance()->json->encode($this->_data); //preserves unicode characters, triggers errors as needed -- MG 2014-05-22
	}

	// Encode as Serialized array
	public function to_serialized()
	{
		return serialize($this->_data);
	}

	// Output as a string representing the PHP structure
	public function to_php()
	{
		return var_export($this->_data, TRUE);
	}

	// Format XML for output
	protected function _from_xml($string)
	{
		return $string ? (array) simplexml_load_string($string, 'SimpleXMLElement', LIBXML_NOCDATA) : array();
	}

	// Format CSV for output
	// This function is DODGY! Not perfect CSV support but works with my REST_Controller
	protected function _from_csv($string)
	{
		$data = array();

		// Splits
		$rows = explode("\n", trim($string));
		$headings = explode(',', array_shift($rows));
		foreach ($rows as $row)
		{
			// The mb_substr removes " from start and end
			$data_fields = explode('","', trim(mb_substr($row, 1, -1)));

			if (count($data_fields) == count($headings))
			{
				$data[] = array_combine($headings, $data_fields);
			}
		}

		return $data;
	}

	// Encode as JSON
	private function _from_json($string)
	{
		return get_instance()->json->decode(trim($string)); //triggers errors as needed -- MG 2014-05-22
	}

	// Encode as Serialized array
	private function _from_serialize($string)
	{
		return unserialize(trim($string));
	}

}

/* End of file format.php */
