<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/*
* @package direct-project-innovation-initiative
* @subpackage controllers
*/


/**
* Widget Controller: To present data to the widget views. 
* This page is used to manage the authentication for the widget to the user. 
* This page is used to display specific data for the widget views as well. 
*
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class Widget_controller extends CI_Controller{
	
	public $feedback_message; //if the user isn't able to view the widget for some reason (system unavailable, account disabled, etc), the feedback message for them will be stored here.
	public $patient; //if show_widget_for_patient() is called, the patient object will be stored here

	/**
	* Outputs a JSON array of mailboxes for the currently logged in user
	*/
	public function ajax_mailboxes(){
		$this->load->library('json');
		$mailboxes = array(); //sometimes the user won't be in session and we won't have any mailboxes
		
		if($this->show_widget()){ //need to verify that the user has access
			$user = User::find_from_session();
			$mailboxes = $user->mailboxes();
		}

		//returning encoded JSON object
		header('Content-Type: application/json');
		echo $this->json->encode($mailboxes);
	}	

	/**
	* Displays the number of new messages in each of the user's mailboxes
	* @param int The maximum number of mailboxes that should be shown
	*/
	public function new_message_count($limit = 4){
		if(!$this->is->nonzero_unsigned_integer($limit)) redirect('widgets/new_message_count'); //go with the default if the user supplied in invalid number
		
		//if the user isn't logged in, just show them the relevant feedback message inside of the widget box
		if(!$this->show_widget()){
			$this->load->view('widgets/new_message_count', array('feedback_message' => $this->feedback_message));
			return;
		}
			
		$user = User::find_from_session();

		$this->load->view('widgets/new_message_count', compact('limit', 'user'));
	}
	
	/**
	* If an ICN is provided AND the system can find the patient information in MVI, the user
	* will be able to send a message with the user's CCDA attached.
	*
	* Note that we need to look up the patient info in order to log a disclosure, so it makes more sense to look
	* up the patient first before looking up the (presumably larger and slower) attachment.  The patient data will 
	* be stored in the session to avoid repeat lookups.
	*
	* When making future changes, note that we need to ensure that the height of the widget stays below the height
	* documented in API Admin even when error messages are displayed.  Keep in mind that any changes in behavior
	* to the widget could affect other applications that make use of this widget.
	*/
	public function send_patient_document($icn=null){

		if(!$this->show_widget_for_patient($icn)){
			$this->load->view('widgets/send_patient_document', array('feedback_message' => $this->feedback_message));
			return;
		}
		
		$success_message = $this->session->flashdata('widget_ccd_success_message');
		$this->form_validation->set_rules('to', 'To', 'required|valid_email');
		$this->form_validation->set_error_delimiters('<p class="text-danger feedback">', '</p>');
		
		if(!empty($_POST) && $this->form_validation->run()){
			$user = User::find_from_session();	
		
			//create the message
			$message = new Message();
			$message->subject = $user->display_name().' has sent you a CCDA';
			$message->sender = $user->email_address();
			$message->to = $this->input->post('to');
			$message->body = 'Please find the CCDA for '.$this->patient->full_name().' attached.';
			$user->clear_attachment_cache(); //make sure no stray atttachments get attached this message (to the best of our limited abilities while the cache is set up in this stupid way)
			$message->save();
			
			if($message->attach_ccd_for_patient($this->patient) && $message->send()){
				$this->session->set_flashdata('widget_ccd_success_message', '<strong>Success!</strong>  The CCDA for '.$this->patient->full_name().' has been sent.');
				redirect(current_url()); //redirect so that the user doesn't resubmit the same data on accident
			}
			
			//could show this in the normal view if we had a more succinct (and still user-friendly) way of giving this information - needs to fit on one line
			$error_message = 'An error occurred, and your message could not be sent.  Please try again in a moment, or manually attach this document to a message from the full site.';
			$this->load->view('widgets/send_patient_document', array('feedback_message' => $error_message));
			return;			
		}

		$this->load->view('widgets/send_patient_document', array_merge(compact('error_message', 'success_message'), array('patient' => $this->patient)));
	}
	
///////////////////////////////////////
// PROTECTED METHODS
////////////////////////////////////////	

	/**
	* True if the user has an active Direct account and can be shown the typical contents of the widget.
	* A feedback message explaining the situation will be stored in {@link feedback_message} so that the user can be shown useful information if they can't be shown the usual widget.
	* @return boolean
	*/
	protected function show_widget(){

		//make sure the authentication library has been loaded successfully
		if(!isset($this->authentication) || !is_a($this->authentication, 'Authentication')){
			$this->feedback_message = 'The system is currently unavailable. Please try again in a moment and contact an administrator if the problem persists.'; 
			if(IS_AJAX) return show_error($this->feedback_message, 500);
			return false;
		}
		
		//check if the user has an account
		if(!User::is_an_entity($this->authentication->user)){
			$this->feedback_message = 'You currently do not have an account for this application.  ' .link_to($this->authentication->url_for_registration_page,'Register for Direct', array('target' => 'blank'));
			if(IS_AJAX) return show_error($this->feedback_message, 401);
			return false;
		}

		//check if user has an active account
		if(!$this->authentication->user->is_active()) {
			$this->feedback_message = 'Your account has been disabled. Please contact an administrator for assistance.';
			if(IS_AJAX) return show_error($this->feedback_message, 401);
			return false;
		}

		//check if the user is logged in
		if(!$this->authentication->user->log_in()){
			$this->feedback_message = 'An error has occurred. Please try again by refreshing the page. Contact an administrator if the problem persists.';
			if(IS_AJAX) return show_error($this->feedback_message, 401);
			return false;
		}

		//if the user doesn't have a mailbox available, they don't have anywhere to go
		if(empty($this->authentication->user->mailbox_names())){
			$this->feedback_message = 'Your account is not configured for personal use, and you are not a member of any group accounts. Please contact an administrator to resolve the situation.';
			if(IS_AJAX) return show_error($this->feedback_message, 400);
			return false;
		}

		return true;
	}
	
	//True if we've been given a valid ICN, all criteria for show_widget() is met, and we're able to look up the patient information
	protected function show_widget_for_patient($icn){
		if(empty($icn) || !is_string($icn)){
			$this->feedback_message = 'The system is currently unavailable. Please try again in a moment and contact an administrator if the problem persists.';
			return false;
		}
		
		if(!$this->show_widget()) return false;		
		
		$this->load->library('patient_data/patient_id');
		
		if(!empty($_SESSION['patients']) && is_array($_SESSION['patients']) && array_key_exists($icn, $_SESSION['patients'])){
			$this->patient = new Patient($_SESSION['patients'][$icn]); //patient class can be initialized from an array of values - we'll store that in the $_SESSION to avoid unneeded lookups
			return true;
		}
		
		$patient = $this->patient_id->patient_matching_id($icn);
		if(!is_a($patient, 'Patient')) {
			$this->feedback_message = 'The system is currently unavailable. Please try again in a moment and contact an administrator if the problem persists.';
			return false;
		}
		
		if(empty($_SESSION['patients'])) $_SESSION['patients'] = array();
		$_SESSION['patients'][$icn] = $patient->raw_values();
		
		$this->patient = $patient;
		
		return true;
	}
	
}