<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models
*//** */

/**
* @package direct-as-a-service
* @subpackage models
*/
class UsersModel extends CI_Model {
        const GROUP_TYPE_ROLES = 'roles'; 
        const GROUP_TYPE_APPLICATIONS = 'applications';
        const GROUP_TYPE_GROUPS = 'groups';
    
	//this is for sorting by value of a multiple dimension array
	private $array_key_name_for_sorting;
	
	/* This function will return an associative array with the SQL result and LDAP result of the user
	 * specified by the user id (database index) parameter
	 */
	public function get_user($user_id) {
		$username = $this->get_username_from_id($user_id); 
		if($username) { 
			$result = $this->get_user_from_username($username);
			if($result) { 
				$result->disabled = FALSE;
				return $result; 
			}
			else {
				//if it's not an active account, check if it is disabled
				$result = $this->get_user_from_username($username,FALSE);
				if($result) {
					$result->disabled = TRUE;
					return $result;
				}
			}
		}
		return FALSE;
	}
	
	public function get_users($search_input = null) {
		$users = $this->get_user_from_username('*');
		if($search_input) {
			$search_input = strtolower($search_input);
			$filtered_users = array();
			foreach($users as $user) {
				if (strpos(strtolower($user->username), $search_input) !== false ||
					strpos(strtolower($user->user_org_id), $search_input) !== false ||
					strpos(strtolower($user->cn), $search_input) !== false ||
					strpos(strtolower($user->departmentnumber), $search_input) !== false ||
					strpos(strtolower($user->o), $search_input) !== false ||
					strpos(strtolower($user->physicaldeliveryofficename), $search_input) !== false ||
					strpos(strtolower($user->telephonenumber), $search_input) !== false ||
					strpos(strtolower($user->mobile), $search_input) !== false ||
					strpos(strtolower($user->user_ext_mail), $search_input) !== false ||
					strpos(strtolower($user->title), $search_input) !== false) {
					array_push($filtered_users, $user);
				}
			}
			$users = $filtered_users;
		}
		if(!is_array($users)) { 
			$users = array($users); 
		}
			
		return $users;
	}
	
	public function get_users_by_page_number($start, $size, $users) {
		$users_by_page_number = array();
		$j = 0;
		for($i = 0; $i < sizeOf($users); $i++) {
			
			if($i >= $start - 1 && $i < $start - 1 + $size) {
				$users_by_page_number[$j] = $users[$i];
				$j++;
			}
		}
		return $users_by_page_number;
	}
	
	function cmp($a, $b)
	{
		return strcmp($a->uid, $b->uid);
	}
	/* This function will return an result object with the SQL result and LDAP result of all users
	 */
	public function get_user_from_username($username,$active = TRUE) {
		$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
		$fields = array('uid','cn','givenname','initials','sn','title','departmentnumber','o','physicaldeliveryofficename','telephonenumber','mobile');
		$base_dn = LDAP_ACCOUNT_GROUP;
		if(!$active) { $base_dn = LDAP_DISABLED_ACCOUNT_GROUP; }
		$search = ldap_search($ldap_conn, $base_dn, '(&(uid='.$username.'))', $fields);
		//commented out ldap_sort since it doesn't seem to support sorting string in case-insensitive manner
		//ldap_sort($ldap_conn, $search, 'uid');
		$entries = ldap_get_entries($ldap_conn, $search);
		
		$users = array();
		for($i = 0; $i < $entries['count']; $i++) {
			
			$entry = $entries[$i];
			$user = $this->get_db_user_from_username($entry['uid'][0])->result();
			
			if($user) {
				$user = $user[0]; //there should only be one result
				foreach($fields as $key) {
					if($key !== 'count' && isset($entry[$key][0])) { $user->$key = $entry[$key][0]; } else { $user->$key = FALSE; }
				}
                                $groups = $this->get_user_groups($user->user_id);
 				$user->groups = $groups['groups']; 
                                $user->group_types = $groups['group_types'];
                                
				if($base_dn === LDAP_DISABLED_ACCOUNT_GROUP) { $user->disabled = TRUE; } else { $user->disabled = FALSE; }
				array_push($users,$user);
			}
			//else { return FALSE; }
		}
		if(count($users) === 1) { return $users[0]; } //if only one user, return just the one object
		
		//sort $users by uid in case-insensitive manner
		usort($users, create_function('$a, $b','return strnatcasecmp($a->uid, $b->uid);'));
		
		return $users;
	}
	
	/* This function will return an associative array with the SQL result and LDAP result of all disabled users
	 */
	public function get_disabled_users($search_input = null) {
		$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
		$fields = array('uid','cn','title','departmentnumber','o','physicaldeliveryofficename','telephonenumber','mobile');
		$search = ldap_search($ldap_conn, LDAP_DISABLED_ACCOUNT_GROUP, '(&(uid=*))', $fields);
		$entries = ldap_get_entries($ldap_conn, $search);
		$users = array();
		for($i = 0; $i < $entries['count']; $i++) {
			$entry = $entries[$i];
			$user = $this->get_db_user_from_username($entry['uid'][0])->result();
			if($user) {
				$user = $user[0]; //there should only be one result
				foreach($fields as $key) {
					if($key !== 'count' && isset($entry[$key][0])) { $user->$key = $entry[$key][0]; } else { $user->$key = FALSE; }
				}
				if($search_input) {
					if (strpos($user->username, $search_input) !== false ||
					strpos($user->user_org_id, $search_input) !== false ||
					strpos($user->cn, $search_input) !== false ||
					strpos($user->departmentnumber, $search_input) !== false ||
					strpos($user->o, $search_input) !== false ||
					strpos($user->physicaldeliveryofficename, $search_input) !== false ||
					strpos($user->telephonenumber, $search_input) !== false ||
					strpos($user->mobile, $search_input) !== false ||
					strpos($user->user_ext_mail, $search_input) !== false ||
					strpos($user->title, $search_input) !== false) {
						$groups = $this->get_user_groups($user->user_id);
                                                $user->groups = $groups['groups']; 
                                                $user->group_types = $groups['group_types'];
						array_push($users,$user);
					}
				}
				else {
					$groups = $this->get_user_groups($user->user_id);
                                        $user->groups = $groups['groups']; 
                                        $user->group_types = $groups['group_types'];
					array_push($users,$user);
				}
			}
			else { return FALSE; }
		}
	
		return $users;
	}
	
	public function get_disabled_users_by_page_number($start, $size) {
		$users_by_page_number = array();
		$users = $this->get_disabled_users();
		if(!is_array($users)) { $users = array($users); }
		$j = 0;
		for($i = 0; $i < sizeOf($users); $i++) {
			
			if($i >= $start - 1 && $i < $start - 1 + $size) {
				$users_by_page_number[$j] = $users[$i];
				$j++;
			}
		}
		
		return $users_by_page_number;
	}
	
	/* This function returns an array of the common names of the groups a user belongs to (specified by
	 * user database id), this could be expanded to give more information on the groups.
	 */
	public function get_user_groups($user_id) {
		$username = $this->get_username_from_id($user_id);
		if(isset($username) && $username) {
			$dn = $this->get_dn_from_username($username);
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(member='.$dn.'))', array('cn'));
			$entries = ldap_get_entries($ldap_conn, $search);
			$groups = array();
                        $group_types = [
                            self::GROUP_TYPE_APPLICATIONS => [],
                            self::GROUP_TYPE_ROLES => [],
                            self::GROUP_TYPE_GROUPS => []
                        ];
			for($i = 0; $i < $entries['count']; $i++) { 
                            array_push($groups,$entries[$i]['cn'][0]); 
                            $group_type = $this->getGroupType($entries[$i]);
                            $group_types[$group_type][$entries[$i]['cn'][0]] = $entries[$i]['cn'][0]; 
                        }

 			return ['groups' => $groups, 'group_types' => $group_types]; 
		}
		return FALSE;
	}
	
   /*
    * This function will return an array of uids who are DaaS administrators
    */
    public function get_daas_admins(){
    	//connect to ldap
    	$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if(!$ldap_bind)	$this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
    	$user_arr = array();
    	if($ldap_bind) {
    		//TODO: make this ou=api_admins configurable
    		//search for the api_admins group and pull all members
    		$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(ou=api_admins))', array('member'));
			$entries = ldap_get_entries($ldap_conn, $search);
			//grab the member entries
			$users = $entries[0]['member'];
			//loop through the members
			foreach ($users as $user){
				//grab the members that have a uid and return it
				if (preg_match("/uid=[^,]+/", $user, $uid) == true){  
					$user_arr[] = mb_substr($uid[0],4);  
				}
			}
    	}
    	return $user_arr;
    }
    
    /*
     * This function will return an array of uids who are DaaS administrators
    */
    public function get_app_admins($app_id){
    	//connect to ldap
    	$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if(!$ldap_bind)	$this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
    	$user_arr = array();
    	
    	if($ldap_bind) {
    		//TODO: make this ou=api_admins configurable
    		//search for the api_admins group and pull all members
    		    		
    		$search = ldap_search($ldap_conn, LDAP_APPLICATION_GROUP, '(&(member=*))', array('member'));
    		$entries = ldap_get_entries($ldap_conn, $search);

    		foreach ($entries as $entry){
    			
    			$exploded_dn = ldap_explode_dn($entry['dn'],0); //explode dn so we can find parent dn
    			unset($exploded_dn['count'], $exploded_dn[0]); //remove values unncessary for parent dn
    			$parent_dn = implode(',',$exploded_dn); //impode array to get parent dn
    			
    			//get parent entry and grab the uid (which corresponds to the application's unique id in the database)
    			$parent_search = ldap_read($ldap_conn, $parent_dn, '(objectclass=*)', array('uid'));
    			$parent_entry = ldap_get_entries($ldap_conn, $parent_search);

    			if (isset($parent_entry[0]['uid'])){
    				$uid = (int)$parent_entry[0]['uid'][0];
    				
    				if ($uid == $app_id){
	    				$sec_search = ldap_read($ldap_conn, 'ou=admins,'.$parent_dn, '(&(ou=admins))', array('member'));
	    				$sec_entry = ldap_get_entries($ldap_conn, $sec_search);
	    				
	    				$users = $sec_entry[0]['member'];
	    				//loop through the members
	    				foreach ($users as $user){
	    					//grab the members that have a uid and return it
	    					if (preg_match("/uid=[^,]+/", $user, $uid) == true){
	    						$user_arr[] = mb_substr($uid[0],4);
	    					}
	    				}
	    				
    				}
    			}
    		}
    	}
    	
    	return $user_arr;
    }
    
    /*
     * This function will return an array of email addresses of app admins/pocs
     */
    public function get_request_emails($app_id){
    	$emails = array();
		$application_request = Application_request::find_one($app_id);
		if(Application_request::is_an_entity($application_request)){
    		$requestor_id = $application_request->requestor;
    		$emails[] = $application_request->poc_email;
			$requester = $application_request->requestor_user;
    		if(User::is_an_entity($requester)){
    			$emails[] = $requester->user_ext_mail;
    		}
    	}
    	
    	//remove any duplicates
    	$emails = array_unique($emails);
    	return $emails;
    }
    
    /*
     * This function will return an array of email addresses of daas admins
     */
    public function get_emails_from_uid_array($admins){
    	$emails = array();
    	foreach($admins as $admin){
    		$query = $this->db->query('SELECT user_ext_mail FROM users WHERE username='.$this->db->escape($admin));
    		if ($query){
    			$email = $query->result();
    			if(!empty($email[0])) {
    				array_push($emails,$email[0]->user_ext_mail);
    			}
    		}
    	}
    	return $emails;
    }
	
	/* This function creates a user account with the provided attributes
	 */
	public function create_user($username, $application_id, $id, $mail, $facility_id, $domain, $ldap_attributes) {

		//the user properties will be validated when we create the user, but validate the other things before we start
		if(!$this->is->nonzero_unsigned_integer($application_id)) return $this->error->should_be_a_nonzero_unsigned_integer($application_id);
		if(!is_array($ldap_attributes)) return $this->error->should_be_an_array($ldap_attributes);
		
		//make sure that the application id is valid & we can determine it's dn before we go to the trouble of creating a user
		$this->load->model('applicationmodel');
		$application_dn = $this->applicationmodel->get_users_dn_from_app_id($application_id);
		if(empty($application_dn) || !is_string($application_dn)) return $this->error->warning('Could not create user '.$this->error->describe($username).'; could not determine dn for application#'.$application_id);
		
		//make sure that the username is available as a mailbox name
		if(!Mailbox::name_is_available($username)) return $this->error->warning('Could not create user: '.$this->error->describe($username).' is not available as a mailbox name');
		
		
		//make sure we have valid ldap attributes
		if(empty($ldap_attributes) || !is_array($ldap_attributes)) return $this->error->should_be_a_nonempty_array($ldap_attributes);
		foreach( array('givenName', 'sn') as $required_attribute){
			if(!array_key_exists($required_attribute, $ldap_attributes)) return $this->error->warning('Could not create user '.$this->error->describe($username).' without a value for '.$required_attribute);
		}	
		
		$default_ldap_attributes = array( 'objectClass' => array('posixAccount', 'top', 'person', 'organizationalPerson', 'inetOrgPerson'),
										  'gidNumber' => '5000',
										  'uidNumber' => '5000',
										  'cn' => $ldap_attributes['givenName'].' '.$ldap_attributes['sn']); //uid, homeidrectory, & mail will be added once we've successfully saved the user
		$ldap_attributes = array_merge($ldap_attributes, $default_ldap_attributes);	
		if(!array_key_exists('displayName', $ldap_attributes)) 
			$ldap_attributes['displayName'] = $ldap_attributes['sn'].', '.$ldap_attributes['givenName'];											

		//make sure that we can connect to LDAP before we go to the trouble of creating a user
		
		/* TO-DO: It should be possible to allow API requests to pass a client cert through nginx, which will check for validity and set
		 * certain server variables. We should be able to check those variables, and if valid, look up the user from the cert then perform
		 * an LDAP bind as the actual user.
		 */
		 //TO-DO: re-examine LDAP permissions for FOC
		 //because this is exposed as an API function, we cannot get client cert to link to a user account
		//so use anonymous LDAP admin account and log that the request came from this application
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
		if(!$ldap_bind)	return $this->error->warning("I couldn't create user ".$this->error->describe($username)." because I couldn't connect to LDAP; please check your config file");				
		
		//create the user, now that we know we have the right set-up
		$this->load->library('encrypt');
		$password = $this->random_password();
		$ldap_attributes['userPassword'] = $this->ssha256_encode($password);
		
		$user = User::create( array('username' => $username,
									'user_org_id' => $id,
									'user_ext_mail' => $mail,
									'user_ep' => $this->encrypt->encode($password))); 
		
		if(!User::is_an_entity($user)) return false; //the User class will trigger an error as needed.  Mailbox creation/cleanup will be handled by the User class as needed.
		
		//set up the remainder of our ldap_attributes
		$ldap_attributes['uid'] = $user->username;
		$ldap_attributes['homeDirectory'] = '/var/mailboxes/'.$user->username;
		$ldap_attributes['mail'] = $user->username.'@'.$domain;

		$success = ldap_add($ldap_conn, $this->get_dn_from_username($username),$ldap_attributes);
		if(!$success){
			$this->error->warning('Could not create LDAP entry for '.$user->describe());
			if(!User::delete($user->id)) $this->error->warning('Could not remove '.$user->describe().' from the database; please manually remove the user and mailbox entries from the database as needed.');
			return FALSE;
		}

		//TO-DO: check here for success and delete other entries if this fails, then return FALSE
		$success = ldap_mod_add($ldap_conn, $application_dn, array('member' => $user->dn() ));
		if(!$success) $this->error->warning('Unable to add '.$user->describe().' to application#'.$application_id);
		
		#TODO - IT WOULD BE NICE TO GET THIS ADDED TO RUN_AFTER_CREATE FOR MAILBOX, IF WE COULD FIGURE OUT HOW TO COMMUNICATE THE APPLICATION ID.  ALSO, WE NEED A BOOLEAN SUCCESS FOR MAILBOX_SETTINGS_SETUP SO WE HAVE AN ERROR IF IT DOESN'T WORK
		$this->load->model('usersettingsmodel');
		$mailbox = $user->mailbox;
		$this->usersettingsmodel->mailbox_settings_setup($mailbox->id, $application_id, 'add');
		if(!empty($facility_id))
			$mailbox->facility_id = $facility_id; 
		if(!empty($domain))
			$mailbox->domain = $domain;
		$success = $success && $mailbox->save();
		if(!$success) $this->error->warning('Unable to save mailbox for '.$mailbox->describe());

		return $user;
    }
	
	/* This function creates a user account with the provided attributes
	 * TO-DO: write corresponding Admin API function, if we decided apps will be able to update user settings
	 */
	public function update_user($user_id, $ext_mail, $facility_id, $attributes) {
		$user = $this->get_user($user_id);
		if($user) {
			$username = $user->username;
			$ldap_conn = $this->prepare_ldap_conn();
			//TO-DO: re-examine LDAP permissions for FOC
			//because this potentially will be exposed as an API function, we cannot get client cert to link to a user account
			//so use anonymous LDAP admin account and log that the request came from this application
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$user_dn = $this->get_dn_from_username($username); 
			if($user->disabled) { $user_dn = $this->get_disabled_dn_from_username($username); }
			$result = array();
			foreach($attributes as $key => $attribute) {
				//if the attribute is blank, we can't set it to blank, we have to delete the entry
				if(!empty(trim($attribute))) { 
					$attribute_arr = array($key => $attribute);
					$result[$key] = ldap_modify($ldap_conn,$user_dn,$attribute_arr);
				}
				//suppress warnings because it will complain if it didn't exist and we tried to delete it again
				else {	
					$attribute_arr = array($key => array());
					$remove = @ldap_mod_del($ldap_conn,$user_dn,$attribute_arr); 
					//if the attribute didn't exist before (error 16), then we didn't need to get rid of it and therefore we did what we wanted
					if(!$remove && ldap_errno($ldap_conn) === 16) { $result[$key] = TRUE; }
					else { $result[$key] = $remove; }
				}
			}
			$this->load->model('usersettingsmodel');
			if($ext_mail !== FALSE){
				$result['ext_mail'] = $this->update_ext_mail($user->user_id,$ext_mail);
			}
			$result['facility_select'] = $this->usersettingsmodel->update_facility_id_in_mailbox($username,false,$facility_id);
			return $result;
		}
		return FALSE;
	}
	
	/* This function adds/removes the user specified by the user id (database index) to/from the group
	 * specified by the common name LDAP attribute given. 
	 * Add/Remove according to the specified action string (options 'add' or 'remove').
	 * TO-DO: add function in admin API for this?
	 */
	public function change_group_membership($action, $user_id, $group_cn) {
		if($action !== 'add' && $action !== 'remove') { return FALSE; }
		$username = $this->get_username_from_id($user_id);
		$user_dn = $this->get_dn_from_username($username); //even disabled users are stored in groups as their non-disabled dn
		
		$ldap_conn = $this->prepare_ldap_conn();
		//TO-DO: re-examine LDAP permissions for FOC
		//because this potentially will be exposed as an API function, we cannot get client cert to link to a user account
		//so use anonymous LDAP admin account and log that the request came from this application
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
		if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
		$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(cn='.$group_cn.')(objectClass=groupOfNames))', array('dn'));
		$entries = ldap_get_entries($ldap_conn, $search);
		if($entries['count'] === 1) { //has to be just one entry or it can't be sure what to remove
			$group_dn = $entries[0]['dn'];
			if($action === 'remove') { 
				$remove = @ldap_mod_del($ldap_conn,$group_dn,array('member'=>$user_dn));
				//if the attribute didn't exist before (error 16), then we didn't need to get rid of it and therefore we did what we wanted
				if(!$remove && ldap_errno($ldap_conn) === 16) { return TRUE; }
				else { return $remove; }
			}
			if($action === 'add') { 
				return ldap_mod_add($ldap_conn,$group_dn,array('member'=>$user_dn));
			}
		}
		
		return FALSE;
	}
	
	/* This function will return a list of groups that the user with the permission set
	 * passed to this function as a parameter is allowed to add other users to, or remove them from.
	 */
	public function get_allowed_groups_for_access($permissions) {
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
		//if the user is part of the API admins group, return all groups
		if($permissions['API']['admins'] || $permissions['Role'][0] == "accountgroupmaintenance" ){
			$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(objectClass=groupOfNames)', array('dn','cn'));
			$entries = ldap_get_entries($ldap_conn, $search);
			$result = array();
                        $group_types = [
                            self::GROUP_TYPE_ROLES => [],
                            self::GROUP_TYPE_APPLICATIONS => [],
                            self::GROUP_TYPE_GROUPS => []
                        ];
 			foreach($entries as $key => $entry) {
				//don't include count array entry, and don't show groups that are for application entries only
				if(is_numeric($key) && $entry['dn'] !== LDAP_DIRECT_API_PERMISSIONS_GROUP && $entry['dn'] !== LDAP_ADMIN_API_PERMISSIONS_GROUP && $entry['dn'] !== LDAP_DISCLOSURE_API_PERMISSIONS_GROUP) { 
               
					array_push($result,$entry['cn'][0]); 
                                        $group_type = $this->getGroupType($entry); 
                                        $group_types[$group_type][$entry['cn'][0]] = $entry['cn'][0];
				}
			}
 			return ['groups' => $result, 'group_types' => $group_types]; 
		}
		//TO-DO: for now, everyone else gets nothing, this will need to change
		return array();
	}
	
	/* This function updates a user's external mail field in the database
	 */
	public function update_ext_mail($user_id, $ext_mail) {
		return $this->db->query('UPDATE users SET user_ext_mail='.$this->db->escape($ext_mail).' WHERE user_id='.$this->db->escape($user_id));
	}


	/* This function checks if an organizational id (EDIPI for DoD) is already in use for a given application.
	 */
	public function org_id_linked($id, $application) {
		if(gettype($application) !== 'integer') { $application = (int)$application; } 
		$id_query = $this->db->query('SELECT username FROM users WHERE user_org_id='.$this->db->escape($id));
		if($id_query->num_rows() === 0) { return FALSE; } //if no user records for organizational id
		if($id_query->num_rows() > 1) { return NULL; } //if too many user records for organizational id
		if($id_query) {
			$row = $id_query->row_array(0);
			$username = $row['username'];
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if(!$ldap_bind)	$this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			if($ldap_bind) {
				$dn = $this->get_dn_from_username($username);
				//search for groups that this user is in
				$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(member=' . $dn . '))', array('dn'));
				$entries = ldap_get_entries($ldap_conn, $search);
				if($entries['count'] === 0) { return FALSE; } //if no entries, then it does not exist
				foreach($entries as $entry) {
					if(isset($entry['dn'])) {
						$exploded_dn = ldap_explode_dn($entry['dn'],0); //explode dn so we can find parent dn
						unset($exploded_dn['count'], $exploded_dn[0]); //remove values unncessary for parent dn
						$parent_dn = implode(',',$exploded_dn); //impode array to get parent dn

						//get parent entry and grab the uid (which corresponds to the application's unique id in the database)
						$parent_search = ldap_read($ldap_conn, $parent_dn, '(objectclass=*)', array('uid'));
						$parent_entry = ldap_get_entries($ldap_conn, $parent_search);
						
						if (isset($parent_entry[0]['uid']))
							$uid = (int)$parent_entry[0]['uid'][0];
				
						//get raw values of dn parts so we can check group name
						$exploded_dn_values = ldap_explode_dn($entry['dn'],1); 
						//if the uid matches the application we are checking and the user is in the application's user group
						if(($exploded_dn_values[0] === LDAP_APPLICATION_USER_GROUP_NAME) && ($uid === $application)) { return TRUE; }
					}
				}
				return FALSE;
			}
			else { return NULL; } //return NULL if status can't be determined due to ldap bind fail
		}
		else { return NULL; } //return NULL if status can't be determined due to query fail
	}
	
	/* This function returns a user id in the database that is associated with a provided direct address
	 */
	public function get_id_from_direct_address($address) {
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if(!$ldap_bind)	$this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
		if($ldap_bind) {
			//search for groups that this user is in
			$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(mail=' . $address . '))', array('dn','uid'));
			$entries = ldap_get_entries($ldap_conn, $search);
			if($entries['count'] === 0 || $entries['count'] > 1) { return FALSE; } //if no entries, or too many entries
			if(isset($entries[0]['uid']) && $entries[0]['uid']['count'] === 1) {
				return $this->get_id_from_username($entries[0]['uid'][0]);
			}
		}
		return FALSE;
	}
	
	/* This function gets the user record in the database associated with a username.
	 */
	public function get_db_user_from_username($username) {
		return $this->db->query('SELECT * FROM users WHERE username='.$this->db->escape($username));
	}
	
	/* This function gets the user id in the database associated with a username.
	 */
	public function get_id_from_username($username) {
		if(!$username){
			return FALSE;
		}
		$query = $this->db->query('SELECT user_id FROM users WHERE username='.$this->db->escape($username));
		if($query) {
			if($query->num_rows() === 1) {
				$row = $query->row_array(0);
				return $row['user_id'];
			}
		}
		return FALSE;
	}
	/* This function gets the user ids in the database associated with a username.
	 */
	public function get_ids_from_username($username) {
		return  $this->db->query('SELECT user_id FROM users WHERE username like '.$this->db->escape($username));
	}
	function get_orgs_from_username($username) {
		return  $this->db->query('SELECT user_org_id FROM users WHERE username like '.$this->db->escape($username));
	}
	
	/* This function gets the username in the database associated with a user id.
	 */
	public function get_username_from_id($user_id) {
		/* note - it would be better to not call on this method at all and grab a user object *once* so that we don't need multiple db queries to grab user data */
		//check if $user_id is a positive integer //note - we really should be triggering some error if it's not a valid input, not just accept *any* input.
		if(!get_instance()->is->nonzero_unsigned_integer($user_id)) return false;
		$user = User::find_one( $user_id );
		if(!User::is_an_entity($user)) return false;
		return $user->username;
	}
	
	/* This function gets the user id in the database associated with a organizational identifier.
	 * For DoD, the EDIPI of the CAC will be used as the organizational identifier.
	 */
	public function get_user_id_from_org_id($org_id) {
		/* note - it would be better to not call on this method at all and grab a user object *once* so that we don't need multiple db queries to grab user data */
		$user = User::find_one( array('user_org_id' => $org_id) );
		if(!User::is_an_entity($user)) return false;
		return $user->id;
	}
	
	/* This function gets the organizational identifier associated with the database user id
	 */
	public function get_org_id_from_user_id($user_id) {
		/* note - it would be better to not call on this method at all and grab a user object *once* so that we don't need multiple db queries to grab user data */
		$user = User::find_one($user_id);
		if(!User::is_an_entity($user)) return false;
		return $user->user_org_id;
	}
	
	public function get_user_id_from_user_name($user_name) {
		/* note - it would be better to not call on this method at all and grab a user object *once* so that we don't need multiple db queries to grab user data */
		$user = User::find_one( array('username' => $user_name) );
		if(!User::is_an_entity($user)) return false;
		return $user->username;
	}
	
	/* This function gets the username in the database associated with a organizational identifier.
	 * For DoD, the EDIPI of the CAC will be used as the organizational identifier.
	 */
	public function get_username_from_org_id($org_id) {
		/* note - it would be better to not call on this method at all and grab a user object *once* so that we don't need multiple db queries to grab user data */
		$user = User::find_one(array('user_org_id' => $org_id));
		if(!User::is_an_entity($user)) return false;
		return $user->username;
	}
	
	/* This function retrieves user permissions from LDAP based on group membership.
	 * Returns FALSE on failure to retrieve permissions.
	 * @deprecated Use User::permissions()
	 */
	public function get_user_permissions($username) {
		$default_permissions = array('Registered'=>FALSE);
		if(empty($username)) return $default_permissions;
		

		$user = User::find_one(array('username' => $username));
		if(!User::is_an_entity($user)) return $this->error->should_be_a_username($username);
		
		return $user->permissions;
	}
	
	
	/* Simple function to format a dn for a given username on this system
	 * abstracted to a function so that it can be changed in only
	 * one place if the format ever changes.
	 */ 
	public function get_dn_from_username($username) {
		return 'uid='.$username.','.LDAP_ACCOUNT_GROUP;
	}
	
	/* Similar to get dn, but gets the dn as if the account were disabled */
	public function get_disabled_dn_from_username($username) {
		return 'uid='.$username.','.LDAP_DISABLED_ACCOUNT_GROUP;
	}
	
	/* This function performs an ldap bind on the given ldap connection using the user org id stored in session
	 */
	 public function ldap_bind_current_user($ldap_conn) {
		$this->load->library(array('session','encrypt'));
		$enc_current_user_org_id = $this->session->userdata('user_id');
		if(!empty($enc_current_user_org_id)) {
			$current_user_org_id = $this->encrypt->decode($enc_current_user_org_id);
			$username = $this->get_username_from_org_id($current_user_org_id);
			$user_id = $this->get_id_from_username($username);
			$enc_pwd = $this->get_encrypted_password($user_id);
			$ldap_bind = ldap_bind($ldap_conn, $this->get_dn_from_username($username), $this->encrypt->decode($enc_pwd));
			return $ldap_bind;
		}
		return FALSE;
	 }
	 
	/* -----------------------------*
	 *  PRIVATE FUNCTIONS           *
	 * -----------------------------*/
	 
	 /* This function searches for the encrypted password of a user in the database given user id.
	  * Returns the encrypted password on success, FALSE on failure.
	  */
	 protected function get_encrypted_password($id) {
		$query = $this->db->query('SELECT user_ep FROM users WHERE user_id='.$this->db->escape($id));
		if($query) {
			if($query->num_rows() === 1) {
				$row = $query->row_array(0);
				return $row['user_ep'];
			}
		}
		return FALSE;
	 }
	 
	 /* This function prepares a connection to LDAP using the configured constants for the application
	  * and the LDAP options required for the connection. Returns FALSE on failure, LDAP connection resource
	  * on success.
	  */
	 protected function prepare_ldap_conn() {
		$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
		if($ldap_conn === FALSE) return $this->error->warning('Could not connect to LDAP host: '.$this->error->describe(LDAP_HOSTNAME));
		
		if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
		if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
		return $ldap_conn;
	 }
	
	/* This function generates a 32 character random string for use as a password,
	 * utilizing a mix of lower and upper case, numbers, and special characters.
	 */
	protected function random_password() {
		$chars = array("a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","1","2","3","4","5","6","7","8","9","0","!","@","#","$","%","^","&","*","(",")","{","}","?");
		$pass = "";
		$count = count($chars);
		for($i = 0; $i < 32; $i++) {
			$pass .= $chars[abs(hexdec(bin2hex(openssl_random_pseudo_bytes(6)))%$count)];
		}
		return $pass;
	}
	
	/* This function encodes the given text as a salted SHA string for use in OpenLDAP.
	 */
	protected function ssha256_encode($text) {
		$salt = hash('sha256', openssl_random_pseudo_bytes(32));
		$hash = "{SSHA256}".base64_encode(pack("H*",hash('sha256',$text.$salt)).$salt);
		return $hash;
	}
        
        /**
         * Determines the group type for an ldap group entry
         * 
         * @param array $ldapEntry
         * @return string
         */
        protected function getGroupType($ldapEntry) { 
            $dn = $ldapEntry['dn']; 
            $groupType = self::GROUP_TYPE_GROUPS; 
            
            if (preg_match('/ou=roles/', $dn)) {
                $groupType =  self::GROUP_TYPE_ROLES; 
            } elseif (preg_match('/ou=applications/', $dn)) { 
                $groupType = self::GROUP_TYPE_APPLICATIONS; 
            }
            
            return $groupType;
        }
        
        /**
         * Returns Group Descriptions
         * 
         * @return array
         */
        public function get_group_descriptions() {
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
		
		//if the user is part of the API admins group, return all groups
		 
                $search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(objectClass=groupOfNames)', array('dn','description','cn'));
                $entries = ldap_get_entries($ldap_conn, $search);
                $group_descriptions = []; 
                foreach ($entries as $entry) { 
                    if (is_array($entry) && array_key_exists('cn', $entry) && array_key_exists('description', $entry)) { 
                        $group_descriptions[$entry['cn'][0]] = $entry['description'][0]; 
                    }
                }
                
                return $group_descriptions; 
        }
}
