<?php defined('BASEPATH') OR exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage libraries
*/

 
/** */
require APPPATH.'/libraries/REST_Controller.php';
 
/**
* DaaS CodeIgniter Rest Controller Extension
*
* An extension of Phil Sturgeon's fully RESTful server implementation for CodeIgniter
*
* @author        	Adam Bell, Shawn Williams
*
* @package direct-as-a-service
* @subpackage libraries
*/
abstract class DaaS_REST_Controller extends REST_Controller
{
	var $router_method; //the method which the router is actually trying to call for this controller.  Usually this can be found as $this->router->method, but the REST controller uses a _remap method that messes this up.
	
	var $application_id;
	var $request_id;
	
	var $response_message = array();
	var $missing_required_fields = array();
	var $invalid_fields = array();
	
	protected $request_entity; //this is the row from the requests table, encapsulated as an entity.  for some reason, code also references some other request object. -- MG
	
	public function __construct(){	
		ini_set('display_errors', 0);  //don't display errors while we're in the API -- just write them to the log
	
		$this->add_benchmark('start_daas_rest_controller_constructor');
		$this->add_benchmark('start_rest_controller_constructor');
		parent::__construct();
		$this->add_benchmark('end_rest_controller_constructor');
		
				
		$this->add_benchmark('start_load_router_method');
		$this->_load_router_method();
		$this->add_benchmark('end_load_router_method');

				
		//load models, libraries, & helpers that will be used by all actions
		$this->load->model(array('applicationmodel', 'usersettingsmodel'));
		$this->load->library('permissions');
		
		$this->load->library('Validator', array(), 'is');	
		
		$this->add_benchmark('start_hmac_authenticate_request');
		$this->request->hmac_authenticated = $this->_hmac_authenticate_request();
		$this->add_benchmark('end_hmac_authenticate_request');
		
		$this->add_benchmark('start_is_direct_api_authorized');
		$this->request->direct_api_authorized = $this->_is_direct_api_authorized();
		$this->add_benchmark('end_is_direct_api_authorized');
		
		$this->add_benchmark('start_is_admin_api_authorized');
		$this->request->admin_api_authorized = $this->_is_admin_api_authorized();
		$this->add_benchmark('end_is_admin_api_authorized');
		
		$this->add_benchmark('start_is_disclosure_api_authorized');
		$this->request->disclosure_api_authorized = $this->_is_disclosure_api_authorized();
		$this->add_benchmark('end_is_disclosure_api_authorized');
		
		$this->request->uri = $_SERVER['REQUEST_URI'];
		
		
#TODO - REMOVE THIS WORKAROUND WHEN WE HAVE TIME TO BE MORE ELEGANT
		//Ideally, all of the controllers that inherit from the REST_Controller would be api services that respond with xml/json and require the same authentication, but we have a few 		
		//methods in child classes that are actually displaying HTML and behaving as normal pages on the site.  So as a workaround, we'll check the domain of the site before requiring
		//that the user has authenticated in the normal API - if the domain is api.dev.careinbox.com, we know that we need to require that the requester has been authorized and authenticated
		//in the usual API fashion.
		if( in_api_context() ){ 					
			
			$this->add_benchmark('start_request_creation');
			$request = Request::create(array('call' => $this->request->uri . ' ' . mb_strtoupper($this->request->method)));
			if(Request::is_an_entity($request)){
				$this->request_entity = $request;
				$this->request_id = $request->id;
				$this->application_id = $request->application_id;
			}
			$this->add_benchmark('end_request_creation');
			
			$this->response_message['request_id'] = $this->request_id;
	
			//validate that the requester should be allowed to make this request		
			if (!$this->request->hmac_authenticated) {//check if authenticated request
				$this->response('Access Denied. Authentication Failed.', 401);
			}
					
			if (!$this->request->direct_api_authorized) {//authorize request
				$this->response( 'Access Denied. Use Not Authorized.', 403);
			}				
		}
		$this->add_benchmark('end_daas_rest_controller_constructor');
	}
	
	
	/**
	* Logs the response automatically before calling on the parent method.
	* Additionally, automatically adds the request id and any information stored in 
	* {@link response_message} to the response.
	* @todo This should really be a protected method, but we'd need to change that in the parent class to make the switch.
	* @param string|array A message or an array to encode into the response.
	* @param int HTTP response code
	*/	
	public function response($message_or_data = array(), $http_code = null){
		$this->add_benchmark('start_response');
		if(is_array($message_or_data)){
			$this->response_message = array_merge($this->response_message, $message_or_data);
		}else{
			$this->response_message['message'] = $message_or_data;
		}
		
		//in rare cases, we might not have a request entity because the db isn't working
		if(Request::is_an_entity($this->request_entity)){
			$this->request_entity->response_code = $http_code;
			$this->request_entity->response = $this->response_message;
			$this->request_entity->save();
		}
		
		if($this->request->output_format == 'json' && !$this->json->is_encodable($this->response_message)){
			$http_code = 500;
			$this->response_message = array('message' => 'Internal Server Error: Unencodable Response.');
			log_message('error', 'Request#'.$this->request_id.' has an unencodable response, please check the logs.');
		}			
	
		parent::response($this->response_message, $http_code); //note - parent includes an exit statement, so our benchmark to end this response takes place in the parent
	}
	
	/**
	 * Parse GET
	 */
	protected function _parse_get()
	{
		// Grab proper GET variables
		parse_str(parse_url($_SERVER['REQUEST_URI'], PHP_URL_QUERY), $get);

		// Merge both the URI segments and GET params
		$this->_get_args = array_merge($this->_get_args, $get);
		
		//recreate query string
		$query_str = '';
		foreach($this->_get_args as $key=>$value) { $query_str .= $key.'='.urlencode($value).'&'; }
		$query_str = rtrim($query_str, '&');
		$this->request->body = $query_str;
	}
	
	/**
	 * Parse POST
	 */
	protected function _parse_post()
	{
		$headers = getallheaders();
		if(isset($headers['X-Real-Content-Type'])) { $this->_parse_raw_post(); }
		
		$this->_post_args = $_POST;
	
		$this->request->format and $this->request->body = file_get_contents('php://input');
		$this->request->body = file_get_contents('php://input');
	}
	
	/**
	 * Parse raw POST
	 * This function should be parsing POST requests based on RFC 2388 http://tools.ietf.org/html/rfc2388
	 */
	protected function _parse_raw_post(){			
		$headers = getallheaders();
		$content_type = (!empty($headers['X-Real-Content-Type'])) ? $headers['X-Real-Content-Type'] : $headers['Content-Type'];
		
		//no need to parse the headers if it's not multipart form data
		if(!string_begins_with('multipart/form-data', $content_type)) return false; 		
		
		// grab multipart boundary from content type header
		preg_match('/boundary=[\"]?(.*)$/', $content_type, $matches);
		if(empty($matches)) $this->response('Malformed headers', 400);
		$boundary = rtrim(element(1, $matches),'"');
		
		// read incoming data
		$success = $handle = fopen('php://input', 'rb');
		
		while(!feof($handle)){
			$headers = $this->_metadata_for_header($handle, $boundary);

			//we need to have a name to be able to record the data, but it's okay for it to be a number instead of a string
			//note that technically $_POST should never have numeric keys instead of strings, but we're allowing it because the original version of this application allowed it
			if(array_key_exists('name', $headers) && (is_numeric($headers['name']) || !empty($headers['name']))){
				if(!empty($headers['filename']))
					$this->_add_header_content_to_files($handle, $boundary, $headers); //as long as we have a value for filename, we'll assume it's a file
				else
					$this->_add_header_content_to_post($handle, $boundary, $headers); 
			}
		}
		fclose($handle);
	}
	
	protected function _metadata_for_header(&$handle, $boundary){
		$headers = array();
		
		//we know that we're done with this part of the data when we hit the boundary or an empty line
		while($content = fgets($handle)){
			
			if(!$content || string_begins_with('--'.$boundary, $content))
				break;
			
			$content = rtrim($content);
			if(empty($content)) break;
			
			$headers[strtolower(strstr($content, ':', TRUE))] = substr($content, strpos($content, ': ') + 2);
		}
		
		//the main thing we need from the header metadata is the name for the part, and if it's a file, the filename
		//parse them out here so that we don't need to do any repeat parsing later on
		if(!empty($headers['content-disposition'])){
			$headers['name'] = $this->_attribute_from_header_metadata('name', $headers);
			if(!empty($headers['content-type']) && !string_begins_with('text', strtolower($headers['content-type'])))
				$headers['filename'] = $this->_attribute_from_header_metadata('filename', $headers);
		}
		
		return $headers;
	}
	
	protected function _add_header_content_to_post(&$handle, $boundary, $metadata){
		$name = element('name', $metadata);
		if(is_null($name)) return $this->error->should_be_a_string_or_number($name); //dev should have verified that there's a name before this
		
		$_POST[$name] = '';
		
		while($content = fgets($handle)){
			if(!$content || string_begins_with('--'.$boundary, $content))
				break;
			
			$_POST[$name] .= $content;
		}
		
		$_POST[$name] = rtrim($_POST[$name]);
		return true;
	}

	//todo - too many params, reduce 
	protected function _add_header_content_to_files(&$handle, $boundary, $metadata){
		get_instance()->output->enable_profiler(false);	
		
		$name = element('name', $metadata);
		if(is_null($name)) return $this->error->should_be_a_string_or_number($name); //dev should have verified that there's a name before this
		
		$filename = element('filename', $metadata);
		if(!$this->is->nonempty_string($filename)) return $this->error->should_be_a_nonempty_string($filename); //dev should have verified that there's a name before this	

		$temp_path = 'files/'.$name.'-'.$filename; //todo - hash this, it's probably safer?

		$_FILES[$name] = array();
		$_FILES[$name]['name'] = $filename;
		$_FILES[$name]['type'] = null;
		$_FILES[$name]['tmp_name'] = $this->session->cache_root($temp_path);
		$_FILES[$name]['error'] = 1; //set this up as an error to begin with - we'll turn it to false after we've succesfully parsed the fiel
		
		$tmp_file_handle = null;
		
		while($content = fgets($handle)){
			if(!$content || string_begins_with('--'.$boundary, $content))
				break;
			
			if($tmp_file_handle){
				if(!fwrite($tmp_file_handle, $content))
					$this->error->warning('Error writing  to '.$_FILES[$name]['tmp_name']);
			}else{
				if(!$this->session->add_file_to_cache($temp_path, $content))
					$this->error->warning('Unable to write '.$temp_path.' to cache');
				$tmp_file_handle = fopen($_FILES[$name]['tmp_name'], 'ab');
				if(!$tmp_file_handle)
					$this->error->warning('Unable to create file handle for '.$_FILES[$name]['tmp_name']);
			}
		}
		
		fclose($tmp_file_handle);
		
		$_FILES[$name]['type'] = mime_content_type($_FILES[$name]['tmp_name']);
		$_FILES[$name]['error'] = 0;
		$_FILES[$name]['size'] = filesize($_FILES[$name]['tmp_name']);

		return true;
	}
	
	protected function _attribute_from_header_metadata($attribute, $metadata, $offset=0){
		if(!$this->is->nonempty_string($attribute)) return $this->error->should_be_a_nonempty_string($attribute, $offset+1);
		if(!is_array($metadata)) return $this->error->should_be_an_array($metadata, $offset+1);
		
		if(empty($metadata['content-disposition'])) return false;
		preg_match('/'.$attribute.'="([^"]+)"/', $metadata['content-disposition'], $matches);
		return element(1, $matches);
	}
	
	/* This function authenticates a request to the API. Requests to the API are authenticated using
	 * HMAC authentication. This is done by providing an authorization header containing the public key of
	 * the authenticating application, and a hash signed with the private key. The public key is used to
	 * look up the private key on our side and attempt to recreate the hash. The hash is a combination of
	 * the HTTP verb, the Date, the content MD5 (optional), the content type (optional), and the resource being requested.	 
	 */
	protected function _hmac_authenticate_request() {
		$this->load->model('applicationmodel');
		$headers = getallheaders();
		//check headers
		if(isset($headers['Authorization'])) { //if no authorization header, return false
			$auth_header = explode(' ',$headers['Authorization']);
			$key_hash = explode(':',$auth_header[1]);
			$public_key = $key_hash[0];
			$this->request->public_key = $public_key;
			$query = $this->applicationmodel->get_application_from_public($public_key);
			if ($query->num_rows() == 0){ return false; }
			$query_array = $query->row_array();
			$private_key = $query_array['private_key'];
			$hmac = base64_decode($key_hash[1]);
		}
		else { return FALSE; }
		
		$date = NULL;
		if(isset($headers['Date'])){ $date = $headers['Date']; }
		else {
			if(isset($headers['X-Daas-Date'])) { //if no date header, return false
				$date = $headers['X-Daas-Date'];
			}
			else { return FALSE; }
		}
		//check that date header was within the last 15 mins
		if (!is_null($date)){
			$now = time();
			$diff = 16;
			if(FALSE === strtotime($date)){ $diff = (($now - $date)/60); }
			else { $diff = (($now - strtotime($date))/60); }
			if ($diff > 15 || $diff < -15){ return FALSE; }
		}
		else{ return FALSE; }
		
		//if no canocialized resource, return false
		if(isset($_SERVER['REQUEST_URI'])) { 
			$resource = $_SERVER['REQUEST_URI']; //TO-DO: define canocialization and add logic here
		}
		else { return FALSE; }
		
		//get content-md5 header if it exists
		if(isset($headers['Content-Md5'])) { 
			$content_md5 = base64_decode($headers['Content-Md5']);
		}
		
		//get content-type header, if it exists (it won't for GET requests)
		if(isset($headers['Content-Type'])) { 
			$content_type = $headers['Content-Type'];
			if(isset($headers['X-Real-Content-Type'])) { $content_type = $headers['X-Real-Content-Type']; }
			if(mb_strpos($content_type, ";") > 0){
				$content_type = mb_substr($content_type, 0, mb_strpos($content_type, ";"));
			}
		}
		else {
			//if no content type on a POST request, return false
			if(mb_strtoupper($this->request->method) === 'POST') { return FALSE; }
		}
		
		//construct signed string hash
		$hash_content = mb_strtoupper($this->request->method) . "\n";
		if(isset($date)) { $hash_content .= $date . "\n"; }
		if(isset($content_md5)) { 
			if($content_md5 !==  md5($this->request->body)) { return FALSE; } //if content-md5 provided in header doesn't match, return false
			$hash_content .= md5($this->request->body) . "\n"; 
		}
		if(isset($content_type)) { $hash_content .= $content_type. "\n"; }
		if(isset($resource)) { $hash_content .= $resource; }
		
		//PHP uses hex hmac hash by default, but other languages often use the raw binary, we will allow either
		$hmac_check_raw = hash_hmac('sha256', $hash_content, $private_key, TRUE);
		$hmac_check_hex = hash_hmac('sha256', $hash_content, $private_key, FALSE);
		return (($hmac === $hmac_check_hex) || ($hmac === $hmac_check_raw));
	}
	
	protected function _is_direct_api_authorized() {
		$this->load->library('permissions');
		$this->load->model('applicationmodel');
		if(isset($this->request->public_key)) { $public_key = $this->request->public_key; }
		if(isset($public_key)) {
			$query = $this->applicationmodel->get_application_from_public($public_key);
			if($query && $query->num_rows() === 1) {
				$result = $query->row_array(1);
				$auth = $this->permissions->get_api_authorization($result['id']);
				return $auth['direct'];
			}
		}
		return FALSE;
	}
	protected function _is_disclosure_api_authorized() {
		$this->load->library('permissions');
		$this->load->model('applicationmodel');
		if(isset($this->request->public_key)) { $public_key = $this->request->public_key; }
		if(isset($public_key)) {
			$query = $this->applicationmodel->get_application_from_public($public_key);
			if($query && $query->num_rows() === 1) {
				$result = $query->row_array(1);
				$auth = $this->permissions->get_api_authorization($result['id']);
				return $auth['disclosure'];
			}
		}
		return FALSE;
	}
	
	protected function _is_admin_api_authorized() {
		$this->load->library('permissions');
		$this->load->model('applicationmodel');
		if(isset($this->request->public_key)) { $public_key = $this->request->public_key; }
			if(isset($public_key)) {
			$query = $this->applicationmodel->get_application_from_public($public_key);
			if($query && $query->num_rows() === 1) {
				$result = $query->row_array(1);
				$auth = $this->permissions->get_api_authorization($result['id']);
				return $auth['admin'];
			}
		}
		return FALSE;
	}

	protected function generate_required_fields_message($fields) {
		if(empty($fields)) return '';
		$fields = array_map('humanize', $fields);
		if (count($fields) == 1) { return first_element($fields).' is a required field.'; }
		return array_to_human_readable_list($fields).' are required fields.';
	}
	
	protected function generate_invalid_fields_message($fields) {
		if(empty($fields)) return '';
		$fields = array_map('humanize', $fields);
		if (count($fields) == 1) { return first_element($fields).' contains an invalid input.'; }
		return array_to_human_readable_list($fields).' contain invalid inputs.';
	}
	
	protected function generate_invalid_recipients_message($invalid_addresses) {
		if(empty($invalid_addresses)) return '';
		if (count($invalid_addresses) == 1) { return first_element($invalid_addresses).' is not a trusted recipient.'; }
		return array_to_human_readable_list($invalid_addresses).' are not trusted recipients.';
	}
	
	//normally in CI, you can find out what method was called on the controller by calling $this->router->method
	//this doesn't work in controllers like the REST controller, which use a _remap method
	//this method uses the same logic as a the _remap method to set a class variable to indicate which method is being used.
	protected function _load_router_method(){
		$object_called = $this->router->method; //the _remap method uses this terminology, so we'll stick with it
		
		//this snippet of code is from the REST_Controller::_remap() method, which is what determines the method for this route.
		$pattern = '/^(.*)\.('.implode('|', array_keys($this->_supported_formats)).')$/';
		if (preg_match($pattern, $object_called, $matches)){
			$object_called = $matches[1];
		}

		$this->router_method = $object_called.'_'.$this->request->method;
	}	

	/**
	* Sends an error response to the end user if they've failed to provide required fields for the service call.
	*/
	protected function respond_with_error_if_fields_are_missing(){
		if(empty($this->missing_required_fields)) return;	
		$this->response_message['fields'] = $this->missing_required_fields;
		$this->response($this->generate_required_fields_message($this->missing_required_fields), 422);
	}
	
	/**
	* Sendes an error response to the end user if the current application is not authorized to act on behalf of this mailbox.
	* @param string $permission The type of permission which the application needs for this action.
	* @param string $sender (Optional) The email address for the mailbox
	*/
	protected function respond_with_error_if_user_is_unauthorized($permission, $sender = null){
		if(is_null($sender) && (!isset($this->mailbox) || !Mailbox::is_an_entity($this->mailbox)))
			$this->response('Access denied. The application is not authorized for this action.', 403); //if there isn't a valid mailbox to check for, just return an error without trying to check for permissions
		
		//at this point, we should be able to determine a sender - make sure that this works
		if(is_null($sender) && isset($this->mailbox)) $sender = $this->mailbox->email_address();
		if(!$this->is->string_like_an_email_address($sender)) return $this->error->should_be_an_email_address($sender);
		
		$permissions = $this->permissions->get_user_permissions_for_address($sender,$this->application_id);
		if(!isset($permissions[$permission]) || $permissions[$permission] !== TRUE) {				
#			if(!is_on_local())
			$this->response('Access denied. The application is not authorized for this action.', 403);
		}
	}
	
	/**
	* Sends an error response to the end user if they've provided invalid values for the service call parameters.
	*/
	protected function respond_with_error_if_fields_are_invalid(){
		if(empty($this->invalid_fields)) return;
		$this->response_message['fields'] = $this->invalid_fields;
		$this->response($this->generate_invalid_fields_message($this->invalid_fields), 422);
	}
	
	protected function respond_with_error_if_mailbox_is_inactive(){
		if(!Mailbox::is_an_entity($this->mailbox))
			$this->response('Access denied. No valid mailbox was provided.', 403);

		if(!$this->mailbox->is_active)
			$this->response('Mailbox "'.$this->mailbox->name.'" has been disabled. No changes may be made to this mailbox unless it is re-activated.', 403);	
	}
	
}
