<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
*//** */
 
require_once APPPATH.'controllers/services/mailbox_controller.php';

/**
* @author M. Gibbs <gibbs_margaret@bah.com
* @package direct-as-a-service
* @subpackage controllers
*/
class Automatic_replies_controller extends Mailbox_controller{	
	
	/**
	* Everything that applies to all actions should go here.
	*/
	function __construct(){
		parent::__construct();
		$this->respond_with_error_if_fields_are_missing();  //make sure that we get an error right away if the mailbox is missing
		$this->respond_with_error_if_fields_are_invalid();
		
		
		require_model('automatic_reply');
	}	
		
	/**
	* @param string $mailbox Username of the mailbox
	* @param string $content Text to be sent as an automatic reply
	* @param int $start_at (Optional) Unix timestamp - must be after now()
	* @param int $end_at (Optional) Unix timestamp - must be before $start_at
	*/
	public function create_post(){
		
		if($this->mailbox->has_related('automatic_reply'))
			return $this->response('An automatic reply already exists for this mailbox.  Please use the update service instead.', 422);
	
		//VALIDATE THAT REQUIRED FIELDS ARE PRESENT
		//Parent constructor will make sure that mailbox is present.  We need to validate that there's always a content value, and that both datetimes are supplied if either is supplied
		$required_fields = array('content');
		if(array_key_exists('start_at', $_POST) || array_key_exists('end_at', $_POST)){
		 	$required_fields[] = 'start_at';
			$required_fields[] = 'end_at'; //if one datetime field is supplied, both must be supplied
		}
		
		foreach($required_fields as $required_field){
			if(!array_key_exists($required_field, $_POST)){
				$this->missing_required_fields[] = 'content';
			}
		}
		
		$this->respond_with_error_if_fields_are_missing(); 
		$this->respond_with_error_if_user_is_unauthorized('send');		
		
		//VALIDATE THAT FIELDS HAVE REASONABLE VALUES
		$content = trim($this->input->post('content'));
		if(empty($content))
			$this->invalid_fields[] = 'content';
		
		if(array_key_exists('start_at', $_POST)){
			foreach(array('start_at', 'end_at') as $datetime_field){
				$$datetime_field = $this->input->post($datetime_field);
				if(empty($$datetime_field) || !$this->is->unix_timestamp($$datetime_field))
					$this->invalid_fields[] = $datetime_field;
			}
			
			if($start_at < strtotime('today'))
				$this->invalid_fields[] = 'start_at';
			if($end_at <= $start_at)
				$this->invalid_fields[] = 'end_at';
		}

		$this->respond_with_error_if_fields_are_invalid();
		
		//ADD AUTOMATIC REPLY TO THE DATABASE
		$values = compact('content');
		if(isset($start_at)) $values = array_merge($values, compact('start_at', 'end_at'));
		
		$automatic_reply = $this->mailbox->add_automatic_reply($values);
		if(!Automatic_reply::is_an_entity($automatic_reply))
			$this->response('An unexpected error occurred and the automatic reply could not be set up for this mailbox.', 400);		
		
		return $this->response(array('automatic_reply' => $automatic_reply->values), 200);
	}
	
	/**
	* @param string $mailbox Username of the mailbox
	*/
	public function delete_post(){
		//all we need is the mailbox identifier, which the controller will get for us
		$this->respond_with_error_if_fields_are_missing(); 
		$this->respond_with_error_if_user_is_unauthorized('send');	
		$this->respond_with_error_if_fields_are_invalid();
		
		$automatic_reply = $this->mailbox->automatic_reply;
		
		if(!Automatic_reply::is_an_entity($automatic_reply))
			return $this->response('No automatic reply was set up for this mailbox.  No changes have been made.', 200);
		
		if(Automatic_reply::delete($automatic_reply->id))
			return $this->response('The automatic reply for this mailbox has been removed.', 200);
			
		$this->response('An unexpected error occurred and the automatic reply for this mailbox could not be removed.', 400);		
	}
	
	/**
	* @param string $mailbox Username of the mailbox
	* @param string (Optional) $content Text to be sent as an automatic reply
	* @param int|null $start_at (Optional) Unix timestamp - must be after now()
	* @param int|null $end_at (Optional) Unix timestamp - must be before $start_at
	*/
	public function update_post(){
		$this->respond_with_error_if_fields_are_missing(); 
		$this->respond_with_error_if_user_is_unauthorized('send');		
		
		//validate that the id corresponds to an existing automatic reply
		$automatic_reply = $this->mailbox->automatic_reply;
		if(!Automatic_reply::is_an_entity($automatic_reply)){
			return $this->response('There is no automatic reply set up for this mailbox.  Please use the create service.', 422);
		}
		
		//note - we only want to set the values that are actually in $_POST
		//and using $this->input->post(array('content', 'start_at', 'ended')) would give us default values for the ones that aren't there
		$values = array();
		foreach(array('content', 'start_at', 'end_at') as $field){
			if(array_key_exists($field, $_POST)){
				if(empty($_POST[$field]) && !is_numeric($_POST[$field]))
					$values[$field] = NULL;
				else
					$values[$field] = trim($_POST[$field]);					
			}
		}

		if(empty($values))
			$this->response('No changes have been specified for the automatic reply for this mailbox', 422);
		
		//since start_at and end_at validate values against each other, clear them before attempting to set new values to prevent getting errors when comparing to old data
		if(array_key_exists('start_at', $values) && !$automatic_reply->property_is_empty('start_at'))
			$automatic_reply->start_at = NULL;
		if(array_key_exists('end_at', $values) && !$automatic_reply->property_is_empty('end_at'))
			$automatic_reply->end_at = NULL;
			
		//make sure that the optional values are valid, using validation inherent to the automatic reply model		
		foreach($values as $field => $intended_value){			
			if($automatic_reply->property_has_validation($field) && !$automatic_reply->value_is_valid_for_property($field, $intended_value))
				$this->invalid_fields[] = $field;
			else{
				$automatic_reply->$field = $intended_value;
				//check to make sure that the field really got set to that value - if not, it's probably because it was an invalid value.  note that this will trigger php warning on dev
				if($automatic_reply->$field != $intended_value) $this->invalid_fields[] = $field;
			}		
		}		
		
		if(array_key_exists('start_at', $values) && empty($values['start_at']) && $automatic_reply->start_at != $automatic_reply->end_at)
			$this->invalid_fields[] = 'start_at';
			
		if(array_key_exists('end_at', $values) && empty($values['end_at']) && $automatic_reply->start_at != $automatic_reply->end_at)
			$this->invalid_fields[] = 'end_at';	
		
		$this->respond_with_error_if_fields_are_invalid();
		
		if(!$automatic_reply->save())
			return $this->response('An unexpected error occurred and the automatic reply could not be updated for this mailbox.', 400);		
		
		return $this->response(array('automatic_reply' => $this->mailbox->automatic_reply->values), 200);	
	}
	
	/**
	* @param string $mailbox Username of the mailbox
	*/
	public function find_get(){
		
		$this->respond_with_error_if_fields_are_missing(); 
		$this->respond_with_error_if_user_is_unauthorized('retrieve');		
		$this->respond_with_error_if_fields_are_invalid();
		
		
		$automatic_reply = $this->mailbox->automatic_reply;
		if(!Automatic_reply::is_an_entity($automatic_reply))
			return $this->response('There is no automatic reply configured for this mailbox', 200);	
		return $this->response(array('automatic_reply' => $automatic_reply->values), 200);	
	}
		
}
?>