<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

require_model('api_entity');

/**
* Determines if the parameter is a Message object or a message id, and if necessary, instantiates the Message object for the uid.
*
* @param int|Message
* @return false|Message
*/
function get_message($message_id_or_object){
	if(is_a($message_id_or_object, 'Message')) return $message_id_or_object;

	$message_id = $message_id_or_object;

	$CI = get_instance();
	if(!$CI->is->nonzero_unsigned_integer($message_id)) return $CI->error->should_be_a_message_id_or_object($message_id);
	$message = Message::find_one($message_id);
	if(!Message::is_an_entity($message))
		return $CI->error->warning('Unable to instantiate a message object for message#'.$message_id);
	return $message;
}

/**
* @package direct-project-innovation-initiative
*
* @subpackage models
*/
class Message extends API_entity {
	static $find_resource = 'direct/messages';
	static $find_one_resource = 'direct/message';
	static $count_resource = 'direct/messages/count';
	static $create_resource = 'direct/messages/save';
	static $update_resource = 'direct/messages/save';

	protected $_attachment_files;
	protected $_inline_attachment_files;
	protected $_attached_patient_documents;
	protected $_flag;
	protected $_user;

	protected $_readonly_fields = array('archived',
										'draft',
										'folder_id',
										'html',
										'patients',
										'plain',
										'size',
										'seen');
	protected $_workflow_item;
	protected $_property_validation_rules = array( 	#'body' => 'string',
													'priority' => 'nonzero_unsigned_integer',
													'sender' => 'string_like_a_direct_address',
													'original_sender' => 'nonempty_string_with_no_whitespace',
													'subject' => 'string',
													'workflow_item' => 'nonempty_array',
													'flag' => 'nonempty_array');

	protected $statuses = array ( 0 => 'Message sent',
								  1 => 'Message processed',
								  5 => 'Message delivered',
								  2 => 'Message displayed',
								  3 => 'Message denied',
								  4 => 'Message failed',
								  100 => 'Multiple recipients');


	/**
	* @param string
	* @return array
	*/
	public function addresses_for_display($field){
		/* if(empty($field) || !in_array($field, array('sender', 'to', 'cc', 'bcc','pscc'))) return $this->error-> */
		if(empty($field) || !in_array($field, array('sender', 'to', 'cc', 'bcc'))) return $this->error->should_be_an_address_field($field);
		if($this->property_is_empty($field)) return array();


		if(!$this->property_is_empty('headers')){
			if($field == 'sender') $field = 'from'; //we use 'sender' in the API terminology, but the headers use 'from'
			$headers = get_instance()->json->decode($this->headers);

			//if we can find this field in the headers and it seems to be filled out, use it (theoretically, we may get display names this way - currently, we don't supply these in webmail, but in case we get mail from somewhere else ...)
			//note that the field name may be all-lower case or capitalized in the headers
			foreach( array($field, ucfirst($field)) as $header_field){
				if(is_object($headers) && isset($headers->$header_field) && !empty($headers->$header_field))
					return get_instance()->mailformat->format_address_list_for_display($headers->$header_field);
			}
		}

		//if we couldn't parse the headers to find this value, just use our normalized version
		if($field == 'from') $field = 'sender'; //switch back in case this didn't work
		return get_instance()->mailformat->format_address_list_for_display($this->$field);
	}

	/**
	* @param string
	* @return string
	*/
	public function addresses_for_display_for_send($field){
		/* if(empty($field) || !in_array($field, array('sender', 'to', 'cc', 'bcc', 'pscc'))) return $this->error->should_be_an_address_field($field); */
		if(empty($field) || !in_array($field, array('sender', 'to', 'cc', 'bcc'))) return $this->error->should_be_an_address_field($field);
		if($this->property_is_empty($field)) return '';

		return implode(';', array_map( array(get_instance()->mailformat, 'display_address_for_send'), $this->addresses_for_display($field))).';';
	}

    /**
     * Looks up a cCD patient data file, and if found, attaches it to this message.
     * @param Patient $patient
     * @param string $purpose_of_disclosure
     * @return bool|false
     */
	public function attach_ccd_for_patient(Patient $patient, $purpose_of_disclosure = 'TREATMENT'){

	    if(!is_a($patient, 'Patient')) {
	        return $this->error->should_be_a_patient($patient);
        }
		if(!$this->is_outgoing()) {
		    return $this->error->warning("I can't attach a CCD to an incoming message like ".$this->describe());
        }
		if(!$this->draft) {
		    return $this->error->warning(ucfirst($this->describe).' is not a draft and cannot have attachments added to it.');
        }
		if(!$this->is->nonempty_string($purpose_of_disclosure)) {
		    return $this->error->should_be_a_nonempty_string($purpose_of_disclosure);
        }

		$CI = get_instance();
		$CI->load->library('patient_data/document_retrieve');
		$patient_id = $patient->icn;

		if(is_on_local()) {
		    $document_info = array(
		        'file' => array(
		            'filename' => 'va_ccd_'.$patient_id.'.xml',
                    'data' => file_get_contents(strip_from_end('application/', APPPATH).'images/va_c32_nwhinone.xml')
                )
            );
        } else {
            $document_info = $CI->document_retrieve->find_ccd_for_patient($patient_id);
        }

		if(empty($document_info) || !is_array($document_info) || !array_key_exists('file', $document_info)) {
		    // todo: create a validation process here to ensure we are working with a good file
            // either something went wrong with the SOAP call, or this patient doesn't have a CCD
		    return false;
        }

		//$this->_user = User::find_from_session();

		$filetype = ((!empty($document_info['filetype'])) ? $document_info['filetype'] : 'ccd');
		$filename =  $filetype . '_for_patient_' . $patient_id . '.xml';
		if(!empty($document_info['file']) && !empty($document_info['file']['filename'])) {
			$filename = element('filename', $document_info['file']);
		}
		$binary_string = $document_info['file']['data'];

        // Currently need to include all attachments every time we save the message.
		$this->add_existing_attachments_to_cache();
		if(array_key_exists($filename, $this->attachment_files)){
			// trigger a notice because looking up CCDs is resource-intensive and we don't want developers to do it unnecessarily
			$this->error->notice('The CCD file for '.$this->describe().' has already been attached: '.$filename);
			return true;
		}

		$filename_fullpath = $this->user->attachment_cache() . $filename;
		if(!file_exists($filename_fullpath) && file_put_contents($filename_fullpath, $binary_string) === FALSE) {
			return $this->error->warning("Unable to write CCD for patient id ".$patient_id.' to cache ('.$filename.')');
		}

		// Attempt to save the message
		if(!$this->save()) {
		    log_message('debug', 'Failed to save the message');
		    return false;
        }

		// Update the cache with all attachments
		if($this->has_attachments()) {
            log_message('debug', 'There are attachments');
		    $this->add_existing_attachments_to_cache();
        }

		// Check to make sure the attachment was successful before attempting to log a disclosure
		if(!array_key_exists(basename($filename), $this->attachment_files)){
		    log_message('debug','Failed to attach?');
			$this->error->notice('The CCD file for '.$this->describe().' could not be attached: '.$filename);
			return false;
		}

		// Log a purpose of disclosure statement that includes this patient's SSNs
		$attachment = element(
		    strip_from_beginning($this->user->attachment_cache(), $filename),
            $this->attached_patient_documents()
        );
		$attachment->set_ssn($patient->ssn);
		$attachment->purpose = $purpose_of_disclosure;
		$attachment->save_disclosure();
        // update session cache for this attachment
        log_message('debug','Saving disclosure to cache');
        $CI->session->add_to_session(
            'disclosures',
            $filename,
            array(
                'patient_id' => $patient_id,
                'ppid' => $patient->ssn,
                'purpose_of_disclosure' => $purpose_of_disclosure
            )
        );
		return true;
	}

	public function attachment($name){
		if(!array_key_exists($name, $this->attachment_files)) {
		    return $this->error->warning('There is no attachment named '.$this->error->describe($name).' for '.$this->describe());
        }
		return $this->attachment_files[$name];
	}

	public function attachment_names(){
		return get_instance()->json->decode($this->attachments);
	}

	function has_attachment($name, $binary_string){
		if(!array_key_exists($name, $this->attachment_files)) return false;
		$attachment = $this->attachment_files[$name];
		return ($binary_string === $attachment->binary_string);
	}


	function has_attachments(){

		//if this is a message that we've just created, then any attachments that will be saved for this exist only in the user cache
		if(!isset($this->id)){
			//$user = User::find_from_session();
			if(!directory_exists($this->user->attachment_cache())) return false;
			return !empty(get_filenames($this->user->attachment_cache()));
		}

#TODO - SHOULD WE BE TRIGGERING AN ERROR IF ATTACHMENTS WASN'T LOADED FROM THE API?
#TODO - WHAT IF A SAVE IS IN PROGRESS?
		//if this has an id, then we determine the attachments from the attachments field.
		if($this->property_is_empty('attachments')) return false;
		$attachments = json_decode($this->attachments, true);
		return !empty($attachments);
	}

	function has_attachments_with_schema($schema){
		if(!$this->is->nonempty_string($schema)) return $this->error->should_be_a_nonempty_string($schema);
		if(!$this->has_attachments()) return false;
		foreach($this->attachment_files as $attachment){
			if($attachment->property_exists('schema') && $attachment->schema == $schema) return true;
		}
		return false;
	}

	/**
	* True if the flags for this message indicate that the user replied to it.
	* Note that this will return true whether the user has just replied normally or replied all.
	* @return boolean
	*/
	function has_reply(){
		if(!$this->is_incoming()) return false;
		if(!array_key_exists('flags', $this->values)) return $this->error->warning("I can't determine if ".$this->describe." has a reply without a value for flags.");
		if($this->property_is_empty('flags')) return false;
		return string_contains('reply',  strtolower($this->flags));
	}

	/**
	* True if the flags for this message indicate that the user replied all to it.
	* @return boolean
	*/
	function has_reply_all(){
		if(!$this->is_incoming()) return false;
		if(!array_key_exists('flags', $this->values))  return $this->error->warning("I can't determine if ".$this->describe." has a reply-all without a value for flags.");
		if($this->property_is_empty('flags')) return false;
		return string_contains('replyall',  strtolower($this->flags));
	}

	/**
	* True if the flags for this message indicate that the user forwarded it.
	* @return boolean
	*/
	function has_forward(){
		if(!$this->is_incoming()) return false;
		if(!array_key_exists('flags', $this->values))  return $this->error->warning("I can't determine if ".$this->describe." was forwarded without a value for flags.");
		if($this->property_is_empty('flags')) return false;
		return string_contains('forward',  strtolower($this->flags));
	}


	/**
	* @return boolean
	*/
	public function is_incoming(){
		if(!isset($this->id)) return false;//any new messages we make, that don't have ids, are always outgoing messages - incoming messages are created in the RI
		if(!isset($this->draft)) return $this->error->warning("I can't determine if ".$this->describe().' is incoming without a value for draft; run $message->load_field_values() to load all db values into this message');
		if(!isset($this->sent)) return $this->error->warning("I can't determine if ".$this->describe().' is incoming without a value for sent; run $message->load_field_values() to load all db values into this message');

		return !$this->is_outgoing();
	}

	/**
	* True if this is an outgoing message.
	* This will be true if the message has not yet been saved (all new messages *must* be outgoing), or if the sender matches the mailbox.
	* @return boolean
	*/
	public function is_outgoing(){
		if(!isset($this->id)) return true;//any new messages we make, that don't have ids, are always outgoing messages - incoming messages are created in the RI
		if(!isset($this->draft)) return $this->error->warning("I can't determine if ".$this->describe().' is outcoming without a value for draft; run $message->load_field_values() to load all db values into this message');
		if(!isset($this->sent)) return $this->error->warning("I can't determine if ".$this->describe().' is outgoing without a value for sent; run $message->load_field_values() to load all db values into this message');

		return ($this->draft || $this->sent);
	}


	function header_array(){
		if(!$this->property_is_empty('headers')){
			return get_instance()->json->decode_as_array($this->headers);
		}
		return array();
	}



	/**
	* This function returns this message's most recent status code, translated into a string as defined in the $this->statuses array.
	* If the message has multiple recipients it returns "Multiple recipients".
	* If the function is called with a parameter it finds the most recent status code for a specified recipient.
	* @return string
	*/
	function delivery_status($target_recipient = null){
		if(!is_null($target_recipient) && !$this->is->string_like_an_email_address($target_recipient)) return $this->error->should_be_an_email_address($target_recipient);

		if(!$this->sent) return $this->error->warning("I can't determine the delivery status for ".$this->describe()." because it is not a sent message");
		if(!array_key_exists('message_status', $this->_values))
			return $this->error->warning("I can't determine the delivery status for ".$this->describe." because message_status was not pulled from the API");

		if(!is_null($target_recipient)){
			if (($this->property_is_empty('message_status'))||(!array_key_exists($target_recipient, $this->message_status)))
				return 'Message not yet processed';
			else
				return $this->statuses[$this->message_status[$target_recipient][0]];
		}

		$recipients = get_instance()->json->decode($this->recipients);
		if(count($recipients)>1) return $this->statuses[100];

		if($this->property_is_empty('message_status') || !array_key_exists(first_element($recipients), $this->message_status)) return 'Message not yet processed';
		return $this->statuses[$this->message_status[first_element($recipients)][0]];
	}

	/**
	* This function returns an array that lists the status code statistics for all of this message's recipients.
	* The array maps each status code to the number of recipients for which that status code is the most recent.
	* The first value of the array is a boolean (1/0) for whether the message has multiple recipients.
	* @return array
	*/
	function delivery_statistics() {
		if(!$this->sent) return $this->error->warning("I can't determine the delivery statistics for ".$this->describe()." because it is not a sent message");
		if(!array_key_exists('message_status', $this->_values))
			return $this->error->warning("I can't determine the delivery statistics for ".$this->describe." because message_status was not pulled from the API");

		$recipients = get_instance()->json->decode($this->recipients);

		if(count($recipients)<2) return array(0,0,0,0,0,0,0,0);

		if($this->property_is_empty('message_status')) return array(1,count($recipients),0,0,0,0,0,0);

		$statuses = $this->message_status;
		$messages = array(1,0,0,0,0,0,0,0);
		foreach($recipients as $target_recipient) {
			if (!array_key_exists($target_recipient, $statuses))
				$messages[1]++;
			else
				$messages[$statuses[$target_recipient][0]+2]++;
		}
		return $messages;
	}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// API METHODS
// Any additional methods beyond standard CRUD that need to directly access the API.  These methods need to be watched carefully to make sure they
// clear the API appropriately, don't accrue the overhead of accessing the API if we know the API call would fail, and trigger appropriate errors
// on failure.
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	* Sets the {@link archived} flag to true via the API.
	* Note that the entity will be updated with the most current values after the API call is successful.
	* @return boolean
	*/
	public function archive($level = MSG_ARCHIVED){
		if(!isset($this->id)) $this->error->warning($this->describe().' cannot be archived until it has been saved');
		if($this->archived == MSG_ARCHIVED) return true;

		$CI = get_instance();
		$CI->api->clear();
		$api_args = array(
                'mailbox' => Mailbox::find_from_session()->name,
                'id' => $this->id,
                'level' => $level
		);
		log_message('debug','Archive API Call: ' . print_r($api_args, true));
		$success = $CI->api->call(
		    'direct/messages/archive',
            $api_args,
            'POST'
        );
		log_message('debug','called archive api: ' . print_r($success,true));
		$this->load_field_values_from_db(); //make sure that the entity gets populated with its most current values
        log_message('debug','new archived value is ' . print_r($this->archived(), true));
        return ($success ? true : false);  //make sure that the API call reported success AND we're seeing the results of that success
	}

	function log_disclosures($disclosures){
		if(!$this->is->nonempty_array($disclosures)) return $this->error->should_be_a_nonempty_array($disclosures);
		if(!isset($this->id)) return $this->error->warning("I can't log disclosures for a message that hasn't been saved yet");

		//validate disclosure values
		foreach($disclosures as $disclosure){
			if(empty($disclosure['purpose']))
				return $this->error->warning("All disclosures must have a value for 'purpose', but you gave me ".$this->error->describe($disclosure));
			if(empty($disclosure['ssn']) || !is_numeric($disclosure['ssn']) || strlen($disclosure['ssn']) != 9)
				return $this->error->warning("All disclosures must have an ssn, but you gave me ".$this->error->describe($disclosure));

			if(array_key_exists('hash', $disclosure)){
				$attachments_with_hash = $this->attachments_matching('hash', $disclosure['hash']);
				if(empty($attachments_with_hash) ){
					return $this->error->warning($this->describe().' does not have an attachment with hash '.$this->error->describe($disclosure['hash']));
				}
				foreach($attachments_with_hash as $attachment){
					if(!is_a($attachment, 'Patient_document_attachment')){
						if(empty($disclosure['given_name']))
							return $this->error->warning("Disclosures for non-parseable attachments must have a value for 'given_name', but you gave me ".$this->error->describe($disclosure));
						if(empty($disclosure['family_name']))
							return $this->error->warning("Disclosures for non-parseable attachments must have a value for 'family_name', but you gave me ".$this->error->describe($disclosure));
					}
				}
			}else{
				//message body disclosures must have a given name / family name
				if(empty($disclosure['given_name']))
					return $this->error->warning("Disclosures for message bodies must have a value for 'given_name', but you gave me ".$this->error->describe($disclosure));
				if(empty($disclosure['family_name']))
					return $this->error->warning("Disclosures for  message bodies attachments must have a value for 'family_name', but you gave me ".$this->error->describe($disclosure));
			}
		}


		$CI = get_instance();
		$values_to_save = array('mailbox' => Mailbox::find_from_session()->name,
								'id' => $this->id,
								'disclosures' => $disclosures);

		$CI->api->clear();
		return $CI->api->call('direct/disclosures/log', $values_to_save, 'POST');
	}

	/**
	* Moves a message to a new location (system folder or custom folder).
	*
	* @todo It would be better if we handled moving messages just by saving an altered value on the message -- this would allows us to handle post-save processing consistently
	* @todo This method should verify that the move was successful by checking the updated values as well as checking the success of the API call.
	* @param string|int $destination_id The id of the folder - this will be a string for system folders and an integer id for custom folders
	* @return boolean True on success
	*/
	function move_to($destination_id){
		$CI = get_instance();

		if(!array_key_exists($destination_id, $CI->mailbox->folders)) $this->error->warning($this->error->describe($destination_id).' is not a known folder for '.$CI->mailbox->describe());
		if(!isset($this->id)) $this->error->warning('Cannot move '.$this->describe().' before it has been saved');
		if(!empty($this->altered_values())) $this->error->warning($this->describe().' has unsaved values; please save the entity before moving it to '.$destination_id);

		$values_to_save = array('mailbox' => Mailbox::find_from_session()->name,
								'id' => $this->id,
								'folder' => $destination_id);

		$CI->api->clear();
		$success = $CI->api->call('/direct/messages/move/format/json', $values_to_save, 'POST');
		$this->load_field_values_from_db(); //make sure that the entity gets populated with its most current values
		return $success;
	}



	function send(){
        log_message('debug','Message::send()');
		if(!isset($this->id)) {
		    return $this->error->warning('Cannot send message until it has been saved');
        }

		$values_for_send = array(
		    'id' => $this->id,
            'sender' => $this->sender,
            'original_sender' => $this->original_sender
        );

		//this shouldn't happen under normal circumstances, but can happen if there's been an error in saving
		if($this->to == ALL_USERS_MOCK_ADDRESS && $this->sender == EXTERNAL_EMAILS_FROM){
			//$user = User::find_from_session();
			$values_for_send['sender'] = $this->user->email_address();
		}

		$api = static::api();
		$api->clear();
		$success = $api->call('direct/messages/send/draft', $values_for_send, 'POST');

		log_message('debug','Loading message # ' . $this->id . ' data from DB');
		$this->load_field_values_from_db(); //reload values for this message
		if(!$success || !$this->sent) {
		    return $this->error->warning('Could not send '.$this->describe());
        }
		return true;
	}


/////////////////////
// GETTERS
///////////////////////

    protected function user() {
	    if (!isset($this->_user)) {
            $user = User::find_from_session();
            $this->set_user($user);
        }
	    return $this->_user;
    }

	protected function archived(){
		if(!isset($this->id)) return false; //if there's no id, this is a draft
		return (($this->_values['archived'] == MSG_ARCHIVED) ? MSG_ARCHIVED : false);
	}

	protected function discarded(){
		if(!isset($this->id)) return false; //if there's no id, this is a draft
		return (($this->_values['archived'] == MSG_DISCARDED) ? MSG_DISCARDED : false);
	}

	protected function deleted(){
		if(!isset($this->id)) return false; //if there's no id, this is a draft
		return (($this->_values['archived'] == MSG_DELETED) ? MSG_DELETED : false);
	}

	/**
	* A JSON-encoded array of the names of the attachments for this message.
	* This value will be automatically updated every time a value in {@link mime_fields} is altered
	* for this message.
	*
	* Getter for message.attachments, with a default derived from {@link mime()} if the message hasn't been
	* saved yet.
	*
	* @return string
	*/
	protected function attachments(){
		if(!isset($this->_values['attachments'])){
		    $data = array();
			if(!$this->property_is_empty('attachment_files')) {
			    $data = array_keys($this->attachment_files);
            }
			return json_encode($data);
		}
		return $this->_values['attachments'];
	}


	function attachment_files(){
        log_message('debug','Message::attachment_files()');
		if(!isset($this->_attachment_files)){
			$this->_attachment_files = $this->_attachment_files_of_type('attachment');
            log_message('debug','Message::attachment_files() found ' . count($this->_attachment_files) . ' files');
		}
		return $this->_attachment_files;
	}

	//TODO - RENAME THIS FUNCTION!  WRITTEN QUICKLY FOR VAD-824, NEEDS SOME TLC
	function attachments_matching($property, $value){
		$matches = array();
		foreach($this->attachment_files as $name => $attachment){
			if($attachment->property_exists($property) && $attachment->$property == $value)
				$matches[$name] = $attachment;
			if(method_exists($attachment, $property) && $attachment->$property() == $value)
				$matches[$name] = $attachment;
		}
		return $matches;
	}

	function attached_patient_documents(){
        log_message('debug','Message::attach_patient_documents()');
		if(!isset($this->_attached_patient_documents)){
			$this->_attached_patient_documents = array();
			foreach($this->attachment_files as $filename => $attachment){
				if(is_a($attachment, 'Patient_document_attachment') && is_object($attachment->parser)){ //doesn't count as a patient document if we don't have a parser to obtain the patient information
					$this->_attached_patient_documents[$filename] = $attachment;
				}
			}
		}
		return $this->_attached_patient_documents;
	}

	function body(){
		if(isset($this->_values['body'])) {
		    return $this->_values['body'];
        }

		if(!$this->property_is_empty('html')) {
		    return $this->html;
        }
		return $this->plain;
	}

	function body_for_display(){
		$body =  nl2br($this->plain);
		if(!$this->property_is_empty('html')){
			//N.B. I removed some old code that tried to strip out <html> and <body> tags because we now run purify() on the body and that strips out these tags more effectively -- MG 2014-04-10
			$body = $this->html;
			foreach($this->inline_attachment_files as $name => $attachment){
				if(!$attachment->property_is_empty('cid')){
					$cid_placeholder = 'cid:'.$attachment->cid;
					if(string_contains($cid_placeholder, $body))
						$body = str_replace($cid_placeholder, $attachment->url_for_view(), $body);
					continue;
				}

				//if there isn't a CID for the file or we didn't find it in the body, just append an image link to the body
				$body .= img($image_url);
			}

			//if there is no parts array, but there is an inline image, its probably a single image from apple mail
			//with no text, we can do a quick check here to find out and work to display it
#TODO - WE SHOULD PROBABLY TEST THIS - CURRENTLY THERE DOESN'T APPEAR TO BE ANY WAY IN WHICH THESE APPLE ATTACHMENTS ARE GETTING ADDED TO THE ATTACHMENT FILES, SO THE IMAGE LINK MIGHT NOT WORK - MG 2014-04-10
			#if(!$this->property_is_empty('inline_attachment_files') && empty($body)){
				$x_mailer = element('X-Mailer', $this->header_array(), element('x-mailer', $this->header_array())); //find the xmailer value whether or not it's capitalized
				$x_mailer = 'Apple';
				if(!empty($x_mailer) && string_contains('Apple', $x_mailer)){
					$subtype = "";
					if(isset($this->mime->headers->{'Content-Disposition'})) {
						$disp = explode(';',$this->mime->headers->{'Content-Disposition'});
						foreach($disp as $item) {
							$item = trim($item);
							if(strpos('filename=',$item) >= 0) {
								$arr = explode('.',str_replace('filename=','',$item));
								if(count($arr) > 1) {
									$filename = $arr[0] . '.' . $arr[1];
									$subtype = $arr[1];
								}
							}
						}
						//Previously this just replaced the body, but we're now appending it to be a little safer - this is supposed to only apply to empty bodies, so it shouldn't make a difference - MG 2014-04-10
						$body .= '<img src="' . site_url('inbox/imgdisp/' . $this->id . '/' . rawurlencode($filename) . '/' . $subtype) . '" />';
					}
				}
			#}



		}
		return (is_on_local()) ?@purify($body) : purify($body);
	}

	/**
	* True if this message is a draft.
	* Getter for message.draft, with a default of true if the message hasn't been saved yet.
	* @return boolean
	*/
	protected function draft(){
		if(!array_key_exists('draft', $this->_values)){
			//if this isn't set, then we didn't pull this data from the API and this must be a draft
			$this->_set_field_value('draft', true, 0, TRUE); //override validation because this is normally a read-only property
		}
		return $this->_values['draft'];
	}

	protected function folder(){
		$folders = get_instance()->mailbox->folders();

		if($this->archived == MSG_ARCHIVED)
			return $folders['archived'];
		if($this->draft)
			return $folders['draft'];
		if($this->sent)
			return $folders['sent'];
		if(!Folder::formatted_like_an_id($this->folder_id))
			return $folders['inbox'];

		//note - mailbox->folders is cached, so doing it this way means that we don't need to make another API call
		return element($this->folder_id, get_instance()->mailbox->folders());
	}

	function importance(){
		if(isset($this->priority)){
			if($this->priority < 3)
				return 'high';
			if($this->priority > 3)
				return 'low';
			return 'normal';
		}
		if(!isset($this->id)) return 'normal'; //default to normal if priority hasn't been set
	}

	function inline_attachment_files(){
        log_message('debug','Message::inline_attachments_files()');
		if(!isset($this->_inline_attachment_files)){
			//$this->attachment_files(); //make sure we've parse the normal attachment files so that this array is set
			$this->_inline_attachment_files = $this->_attachment_files_of_type('inline');
            log_message('debug','Message::inline_attachment_files() found ' . count($this->_inline_attachment_files) . ' files');
		}
		return $this->_inline_attachment_files;
	}

	function original_sender(){
		if(!isset($this->_values['original_sender']) && !isset($this->id)){
			//$user = User::find_from_session();
			if(User::is_an_entity($this->user)){
				$this->_set_field_value('original_sender', $this->user->username);
			}
		}
		if(isset($this->_values['original_sender'])) return $this->_values['original_sender'];
	}

	function sender_for_display(){
		if(!$this->property_is_empty('headers')){
			$raw_header = json_decode($this->headers, true);
			if(isset($raw_header->From)){
				$from = Mail_RFC822::parseAddressList($raw_header->From);

				if(!empty($from[0]->personal))
					return htmlentities($from[0]->personal);

				if(!empty($from[0]->mailbox) && !empty($from->host))
					return htmlentities($from[0]->mailbox . "@" . $from[0]->host);

				if(!empty($from[0]->mailbox))
					return htmlentities($from[0]->mailbox);

				if(!empty($from[0]->host))
					return htmlentities($from[0]->host);
			}
		}
		return $this->sender;
	}

	function size_for_display(){
		if(isset($this->size)){
			return byte_format($this->size);
		}
	}

	function subject_for_display(){
		$subject = htmlentities(trim(element('subject', $this->_values)));
		if(empty($subject))
			$subject = '(No Subject)';
		return $subject;
	}

	function timestamp_for_display(){

		if(isset($this->timestamp)){
			$display_time = $this->timestamp;
			//adjust for user location
			//$user = User::find_from_session();
			if($this->user->timezone != '' || !is_null($this->user->user_locale)){
				//since a integer timezone kills the PHP DATETIMEZONE object, convert the int to the datetimezone string
				if(is_int($this->user->timezone)){
					$correct_timezone = get_instance()->locale->get_timezone_from_id($this->user->timezone);
					$timezone_object = new DateTimeZone($correct_timezone);
				}else{
					$timezone_object = new DateTimeZone($this->user->timezone);
				}

				$time_offset = $timezone_object->getOffset(new DateTime('now', new DateTimeZone('GMT')));
				$display_time = $this->timestamp + ($time_offset);
			}

			$date = date("n/j/Y",$display_time); //format date from unix date
			if($date == date("n/j/Y")) { //if the message was sent today, display the time
				$date = date("h:i A",$display_time);
			}
			return $date;
		}
	}

	/**
	* A JSON-encoded array of the email addresses of all recipients (to, cc, and bcc) of this message.
	* Getter for message.recipients, with a default determined from to, cc, and bcc values of this message
	* if it has not yet been saved.
	* @return string
	*/
	protected function recipients(){

		if(!isset($this->_values['recipients'])){
			$recipients = array_unique( array_filter( array_merge( explode(',', $this->to), explode(',', $this->cc), explode(',', $this->bcc) ) ) );
			if(!empty($recipients))
				$this->_values['recipients'] = json_encode( $recipients );
		}

		//TEMPORARY MEASURE BECAUSE THE RI ISN'T ENCODING THE RECIPIENTS FIELD CORRECTLY - REMOVE THIS WHEN THE RI IS FIXED
		if($this->is_incoming() && !string_contains('"', $this->_values['recipients'])){
			$this->_values['recipients'] = json_encode(array_map('trim', explode(',', strip_from_beginning('[', strip_from_end(']', $this->_values['recipients'])))));
		}

		return $this->_values['recipients'];
	}

	protected function seen(){
		if($this->is_outgoing()) return true;
		return $this->_values['seen'];
	}

	protected function sent(){
		if(!isset($this->id)) return false; //if there's no id, this is a draft
		return $this->_values['sent'];
	}

	function to_for_display(){
		if(!$this->property_is_empty('headers')){
			$raw_header = json_decode($this->headers);

			//format To addresses, will look for personal names before just displaying addresses
			if(isset($raw_header->To)) {
				$to = Mail_RFC822::parseAddressList($raw_header->To);
				if(!empty($to[0]->personal)){
					return htmlentities($to[0]->personal);
				}
				if(!empty($to[0]->mailbox) && !empty($to->host)){
					return htmlentities($to[0]->mailbox . '@' . $to[0]->host);
				}
				if(!empty($to[0]->mailbox)){
					return htmlentities($to[0]->mailbox);
				}
				if(!empty($to[0]->host)){
					return $to[0]->host;
				}
			}
		}
		return $this->to;
	}

	protected function _attachment_files_of_type($type){
        log_message('debug','Message::_attachment_files_of_type('.$type.')');
		require_library('attachment');

		if(!in_array($type, array('inline', 'attachment'))) {
		    return $this->error->should_be_a_known_type($type);
        }
		$files = array();
		// doest the message have attachments?
		if(isset($this->mime) && $this->property_exists('mime') && is_object($this->mime) && isset($this->mime->parts)){
			foreach($this->mime->parts as $part){
			    //log_message('debug', 'Message Parts:' . "\n" . print_r($part, true));
				if(isset($part->headers) && !empty($part->headers->{'content-transfer-encoding'}))	{
					// decode part based on content transfer encoding, if necessary
                    $encoding = strtolower($part->headers->{'content-transfer-encoding'});
					switch ($encoding) {
						case 'base64':
							$part->body = base64_decode($part->body);
							break;
						case 'quoted-printable':
							$part->body = quoted_printable_decode($part->body);
							break;
						default:
                            // 8bit, 7bit, binary are options we already handle.
                            // x-token would be custom, which we don't handle.
							break;
					}
				}

				if(isset($part->disposition) && $part->disposition == $type){

				    // try to track down the name of this attachment if we can
					$name = '';
                    $attachment_content_type_parts = explode(';',$part->headers->{'content-type'});

                    // attachment file type
                    $mime_type = strtolower(trim($attachment_content_type_parts[0]));

                    // attachment file name
                    if (preg_match('/name="([^\"\']+)"/', trim($attachment_content_type_parts[1]), $match)) {
                        if (!empty($match[1])) {
                            $name = trim($match[1]);
                        }
                    }

                    if (empty($name)) {
                        if(isset($part->ctype_parameters) && !empty($part->ctype_parameters->name)) {
                            $name = $part->ctype_parameters->name;
                        }
                        elseif(isset($part->ctype_parameters) && !empty($part->ctype_parameters->filename)) {
                            $name = $part->ctype_parameters->filename;
                        }
                        elseif(isset($part->d_parameters) && !empty($part->d_parameters->name)) {
                            $name = $part->d_parameters->name;
                        }
                        elseif(isset($part->d_parameters) && !empty($part->d_parameters->filename)) {
                            $name = $part->d_parameters->filename;
                        }
                        else {
                            $name = 'Untitled';
                        }
                    }

					// figure out the extension, in case we need to modify the name at all
					$extension = pathinfo($name, PATHINFO_EXTENSION);
                    $filename = pathinfo($name, PATHINFO_FILENAME);
					if($name == 'Untitled' && empty($extension) && !empty($part->ctype_secondary)) {
					    $extension = $part->ctype_secondary;
                        // This may not actually be a file type (eg "plain" or "x-unknown-content-type").
                        // Could add a check, but risk falsely ruling out file types
                    }

					// if we have duplicate file names, make them unique by adding a number (#) to each one
					for($i = 1; array_key_exists($filename.'.'.$extension, $files); $i++){
						if(preg_match('/\(\d+\)$/', $filename, $matches)) {
                            $filename = trim(strip_from_end(first_element($matches), $filename));
						}
                        $filename = $filename.' ('.$i.')';
					}
                    $filename = $filename.'.'.$extension;

					// size of attachments (bytes)
                    $bytes = 0;
                    if(!empty($part->headers->{'content-length'})) {
                        $bytes += (int)$part->headers->{'content-length'};
                    }

					// create an attachment object for this file
					$attachment = Attachment::create(
					    str_replace('/', '', $filename),
                        $part->body,
                        array(
                            'message_id' => $this->id,
                            'mime_type' => $mime_type,
                            'file_size' => $bytes,
                        )
                    );

					//add some additional properties for backwards compatibility
					if(!empty($part->ctype_secondary)) {
					    $attachment->filetype = $part->ctype_secondary;
                    } // will default to the extension
					if(isset($part->headers->{'content-id'})) {
						if(preg_match('/<(.*?)>/',$part->headers->{'content-id'},$cid)) {
						    $attachment->cid = trim($cid[1]);
                        }
					}

					//check to see if we have any disclosures for this attachment - if so, populate patient data
					if(isset($this->disclosures) && array_key_exists($attachment->hash, $this->disclosures)){
						$attachment->ssn = element('ssn', $this->disclosures[$attachment->hash]);
						$attachment->purpose = element('purpose', $this->disclosures[$attachment->hash]);
					}

#TODO - IF THERE ARE PATIENTS FOR THIS MESSAGE, ITERATE THROUGH THEM AND SUPPLY ADDITIONAL DATA

					$files[$attachment->name] = $attachment;
				}
			}
		}

		return $files;
	}

//////////////////////
// SETTERS
//////////////////////

    protected function set_user($user) {
	    return $this->_user = $user;
    }
	protected function set_bcc($value, $offset=0){
		return $this->_set_recipient_field('bcc', $value, $offset+1);
	}

	protected function set_body($value, $offset=0){
		if($this->body == str_replace('<br>', '<br />', $value)) return true; //we don't need to alter the body if there's no difference except how <br />s are formatted
		if(!is_string($value)) $this->error->property_value_should_be_a_string('body', get_class($this), $value, $offset+1);
		$this->_set_field_value('body', $value, $offset+1);
	}

	protected function set_cc($value, $offset=0){
		return $this->_set_recipient_field('cc', $value, $offset+1);
	}

    /*
    protected function set_pscc($value, $offset=0){
		return $this->_set_field_value('pscc', $value, $offset+1);
	}
	*/

	protected function set_importance($value, $offset=0){
		$valid_values = array('low', 'normal', 'high');
		if(!empty($value) && !in_array($value, $valid_values)) return $this->error->property_value_should_be_an_x('importance', get_class($this), $value, array_to_human_readable_list($valid_values, 'or'), $offset+1);

		if($value == 'low') return $this->_set_field_value('priority', 5, $offset+1);
		if($value == 'high') return $this->_set_field_value('priority', 1, $offset+1);
		return $this->_set_field_value('priority', 3, $offset+1);
	}

	protected function set_to($value, $offset=0){
		return $this->_set_recipient_field('to', $value, $offset+1);
	}

	protected function _set_recipient_field($field, $value, $offset=0){
		if(!in_array($field, array('to', 'cc', 'bcc'))) return $this->error->should_be_a_known_recipient_field($field);
		$original_value = $value;
		if(!is_string($value) && !is_array($value)) return $this->error->property_value_should_be_a_string_or_an_array($field, get_class($this), $value, $offset+1);
		if(is_string($value)) $value = array_filter(array_map('trim', explode(';', $value)), 'strlen');

		if(!empty($value)){
			//replace any distro list aliases
			$value = get_instance()->public_distribution_list_model->substitute_addresses_for_aliases($value);
			$value = get_instance()->private_distribution_list_model->substitute_addresses_for_aliases($value);

			//make sure we now have a list of email addresses
			if(!$this->is->array_of_string_like_an_email_addresses($value))
				return $this->error->property_value_should_be_a_semicolon_separated_list_of_email_addresses_or_distribution_list_aliases($field, get_class($this), $original_value, $offset+1);
		}

		$this->_values['recipients'] = null; //force recipients to be regenerated
		$value = implode(';', $value);
		return $this->_set_field_value($field, $value, $offset+1);
	}


///////////////////////////////
// DATA MANAGEMENT (INSTANCE)
//////////////////////////////

	public function add_existing_attachments_to_cache($user = null){
        log_message('debug','Message::add_existing_attachments_to_cache()');
		if ( is_null($user)) {
		    $user = $this->user;
        }
		if(isset($this->mime) && $this->has_attachments()){
            // parse from mime body
		    $this->attachment_files();
            $this->inline_attachment_files();
            log_message('debug','Message::add_existing_attachments_to_cache() adding ' . count($this->_attachment_files) . ' attachments');
			foreach($this->_attachment_files as $attachment){
                $user->add_attachment_to_cache($attachment->filename, $attachment->binary_string);
			}
            log_message('debug','Message::add_existing_attachments_to_cache() adding ' . count($this->_inline_attachment_files) . ' attachments');
            foreach($this->_inline_attachment_files as $attachment){
                $user->add_attachment_to_cache($attachment->filename, $attachment->binary_string);
			}
		}
	}

	//by default, we want to be able to make changes to the message without overwriting the attachments
	//setting $preserve_existing_attachments to FALSE will ensure that the attachments will be changed to match only what is set in the cache
	public function save( $preserve_existing_attachments = true ){
        log_message('debug','Message::save('.($preserve_existing_attachments?'yes':'no').')');
		if(!is_bool($preserve_existing_attachments)) {
		    return $this->error->should_be_a_boolean($preserve_existing_attachments);
        }
		if($preserve_existing_attachments) {
            log_message('debug','Message::preserving attachments');
		    $this->add_existing_attachments_to_cache();
        }

		return parent::save();
	}

	//overrides parent to make sure that the mailbox for this message is specified
	public function load_field_values_from_db(){
        log_message('debug','Message::load_field_values_from_db()');
		$id = $this->id();
		if(empty($id)) return $this->error->warning("I can't load values from the api for a ".get_class($this).' entity without an id', 1);
		if($this->property_is_empty('sender')) return $this->error->warning("I can't load values from the api for a ".get_class($this).' entity without a sender', 1);

		$conditions = array('id' => $id);

		//if this is a system-wide message, we need to include that in the search conditions as an indicator
		if(isset($this->to) && string_contains(ALL_USERS_MOCK_ADDRESS, $this->to))
			$conditions['to'] = $this->to;

		$entity = static::find_one($conditions);
		if(!static::is_an_entity($entity)) return $this->error->should_be_an_x($id, 'id for a '.get_class($this).' entity', 1);
		$success = $this->_set_field_values($entity->values());
		if($success) $this->_altered_fields = array();

		return $success;
	}

	protected function _run_before_create_and_update(){
		if(!parent::_run_before_create_and_update()) return false;

		if($this->property_is_empty('original_sender'))
			$this->_set_field_value('original_sender', $this->user->username);

		return true;
	}

	protected function _set_field_values($values){
		$success = parent::_set_field_values($values);
		if($success && array_key_exists('mime', $this->_values)){
			$this->_values['mime'] = (object)$this->_values['mime'];
			//reset the mime-derived fields
			$this->_attachment_files = null;
		}
		return $success;
    }

	protected function _values_are_valid_for_create_and_update(){
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		if($this->property_is_empty('sender')) return $this->error->warning(ucfirst($this->describe()).' is missing required field '.$this->error->describe('sender'));
		if($this->draft && $this->property_is_empty('original_sender')) return $this->error->warning(ucfirst($this->describe()).' is missing required field '.$this->error->describe('original_sender'));
		return true;
	}

	protected function _values_for_save(){
		$values = parent::_values_for_save();

		//make sure that these fields are always included in the call
		$required_fields = array('id', 'sender');
		foreach($required_fields as $required_field){
			if(!$this->property_is_empty($required_field)){
				$values[$required_field] = $this->$required_field;
			}
		}

		//find the attachments for this message
		//$user = User::find_from_session();  //uploaded attachments will always be in the logged-in user's cache
		if(User::is_an_entity($this->user)){

			//if this is a system-wide email address, pay attention to the logged-in user instead of the current mailbox (since the user is the one with admin privileges)
			if($this->to == ALL_USERS_MOCK_ADDRESS)
				$values['sender'] = $this->user->email_address();


			$attachments_to_save = $this->user->attachments_from_cache();
			foreach($attachments_to_save as $file => $attachment){
				$attachments_to_save[$file] = $attachment->binary_string;
			}

			//if this message has been saved before and the only thing we're saving is attachments, make sure the attachments are different from how they used to be
			//otherwise, we'll just skip saving
			if(isset($this->id) && count($values) == count($required_fields) && array_keys($values) == $required_fields){
				$current_attachments = array();
				foreach(array_merge($this->attachment_files, $this->inline_attachment_files) as $file){
					$current_attachments[$file->name] = $file->binary_string;
				}

				//no changes have been made to this message, no values need to be saved.  (note that array_diff_assoc is unidirectional and so we need to check both directions.)
				if(empty(array_diff_assoc($attachments_to_save, $current_attachments)) && empty(array_diff_assoc($current_attachments, $attachments_to_save))) return array();
			}

			$CI = get_instance();

			//now that we know attachments need to be saved, add them to the api library
			foreach($attachments_to_save as $filename => $binary_string){
				$CI->api->files_to_send[] = compact('filename', 'binary_string');
			}
			if(!empty($attachments_to_save)) $this->_altered_fields[] = 'attachments'; //make sure that we indicate that there's been changes, since we're not adding anything to the values array
		}

		return $values;
	}

	//note that this only gets run if the save was successful
	protected function _run_after_create_and_update(){
		parent::_run_after_create_and_update();

		//$user = User::find_from_session();
		if(!$this->user->clear_attachment_cache()){
			$remaining_files = get_filenames($this->user->attachment_cache());
			$message = "I couldn't clear the attachment cache for ".$this->user->describe().' after saving '.$this->describe().'. '.
					   'These '.pluralize_if_necessary('attachment', count($remaining_files)).' attachments may show up in the next draft this '.$this->user->describe().' edits: '.
					   array_to_human_readable_list($remaining_files);
			$this->error->warning($message); //don't return false for this - we aren't going to fail the entire action just because we couldn't clear the cache.
		}

		return true;
	}


//////////////////////////////
// STATIC
//////////////////////////////

	/**
	* Checks to make sure that the given file has a valid filetype.
	*
	* @todo It would be better to whitelist these.
	*
	* @param string
	* @param string
	* @return boolean
	*/
	public static function attachment_has_valid_extension($name){
#TODO - WE SHOULD REALLY BE CHECKING THIS AGAINST A WHITELIST, NOT JUST BLACKLISTING EXE
		$extension = pathinfo($name, PATHINFO_EXTENSION);
		return !empty($extension) && strtolower($extension) != 'exe';
	}

	public static function fields(){
		//hokay.  confusingly, we get very different fields back from the API then we give it.
		//ideally, it would be nice to change this, because presumably other users of the API will find this just as difficult as we do.
		//but for now, both sets need to be recognized so that we can access them easily

		$fields_we_send_to_the_api = array( 'id',
											'mailtype',
											'priority',
											'to',
											'cc',
											'bcc',
											'subject',
											'body',
											'sender',
											'original_sender',
											'protected_data',
                                                                                        'non_va_referral'/*,
                                                                                        'pscc'*/
										   );

		//these should all be in the read-only
		$fields_we_get_back_from_the_api = array( 'recipients',
												  'attachments',
												  'plain',
												  'html',
												  'timestamp',

												  'folder_id',
												  'size',
												  'flags',
												  'headers',
												  'seen',
												  'draft',
												  'sent',
												  'archived',
												  'message_id'.
												  'mime',
												  'message_status');

		return array_unique(array_merge($fields_we_send_to_the_api, $fields_we_get_back_from_the_api));
    }

    public static function find_one($id_or_conditions=array()){
		if(!is_array($id_or_conditions))
			$id_or_conditions = array('id' => $id_or_conditions);
		$id_or_conditions['limit'] = 1;
		return parent::find_one($id_or_conditions);
    }


	protected static function _conditions_for_find($id_or_conditions){
		$conditions = parent::_conditions_for_find($id_or_conditions);

		if(array_key_exists('to', $conditions) && string_contains(ALL_USERS_MOCK_ADDRESS, $conditions['to'])){
			$user = User::find_from_session();
			$conditions['mailbox'] = $user->user_name; //if this is a sytem-wide message, use the user's address, since users are the ones with admin permissions
			unset($conditions['sender']);
		}

		//make sure we have a mailbox - assume that it's the current mailbox if not otherwise specified
		if(!array_key_exists('mailbox', $conditions)){
			$mailbox = Mailbox::find_from_session();
			$conditions['mailbox'] = $mailbox->name;

			if(array_key_exists('sender', $conditions))
				unset($conditions['sender']);
		}

		if(!array_key_exists('folder', $conditions)){
			$conditions['folder'] = 'all';
		}

		$filter = array();
		foreach(array('seen','flags','size','sender','to','cc','bcc','subject','plain','html','priority', 'original_folder', 'first_date', 'end_date', 'date', 'has_attachment', 'smallest', 'largest', 'body') as $key){
			if(array_key_exists($key, $conditions)){
				$filter[$key] = $conditions[$key];
				unset($conditions[$key]);
			}
		}

		if(!empty($filter)){
			$conditions['filter'] = base64_encode(get_instance()->json->encode($filter));
		}

		return $conditions;
	}


	protected static function _results_from_api_output(){
		$output = parent::_results_from_api_output();

		if(!is_array($output) || !array_key_exists('mail', $output))
			return array();

		$messages = $output['mail'];
		if(empty($messages)) return array();

		$output_as_object = static::api()->output();
		if(!is_array(first_element($messages))){ //if we were searching for just one id
			if(isset($output_as_object->mail->mime))
				$messages['mime'] = $output_as_object->mail->mime; //for backwards compatibility, make sure the mime is an object
			return array($messages['id'] => $messages);
		}

		//for backwards compatability, make sure mime is an object

		foreach($output_as_object->mail as $message){
			if(isset($message->mime))
			$messages[$message->id]['mime'] = $message->mime;
		}

		return $messages;
	}



}
