<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

require_model('db_entity');

/**
* @author M. Gibbs <gibbs_margaret@bah.com>
* @package direct-project-innovation-initiative
* @subpackage models
*/
class Mailbox extends DB_entity {
	static $table = 'users';
	static $primary_key = 'user_id';

	protected $_folders;

	protected $_property_validation_rules = array( 'user_name' => 'nonempty_string',
													'user_mail' => 'string_like_an_email_address',
													'user_theme' => 'nonzero_unsigned_integer',
													'user_ep' => 'nonempty_string',
													'user_created_by' => 'nonzero_unsigned_integer',
													'user_created_date' => 'unix_timestamp',
													'user_ext_notify_flag' => 'boolean',
													'user_ext_group_notify_flag' => 'boolean',
													'user_deleted_flag' => 'boolean',
													'user_is_group' => 'boolean'
													);


	function name(){ return $this->user_name; }

	//adding this to be consistent with api and so that we can swap this out if needed
	function is_active(){
		return !$this->user_deleted_flag;
	}

	function is_group(){ return $this->user_is_group; }


	function messages($conditions = array(), $key_by=null){
		if(!is_array($conditions)) return $this->error->should_be_an_array($conditions);
		if(!is_null($key_by) && !$this->is->nonempty_string($key_by)) return $this->error->should_be_a_nonempty_string($key_by);

		$conditions['mailbox'] = $this->name;
		return Message::find($conditions, $key_by);
	}

	/**
	* The email address for this mailbox.
	* @return string
	*/
	public function email_address(){
		if($this->property_is_empty('user_name')) return '';
		return $this->user_name.'@'.CLINICAL_DOMAIN;
	}

	public function current_folder(){
		return element(get_instance()->session->mailbox_location(), $this->folders);
	}

	public function folders($conditions = array(), $key_by=null){
		if(!is_array($conditions)) return $this->error->should_be_an_array($conditions);
		if(!is_null($key_by) && !$this->is->nonempty_string($key_by)) return $this->error->should_be_a_nonempty_string($key_by);

		$use_cache = false;
		if(empty($conditions) && empty($key_by)){
			$use_cache = true;
			if(isset($this->_folders)) return $this->_folders;
		}


		$conditions['mailbox'] = $this->name;
		$folders = Folder::find($conditions, $key_by);
		if($use_cache) $this->_folders = $folders;
		return $folders;
	}

	public function system_folders(){
		$folders = array();
		foreach($this->folders_in_hierarchical_order as $id => $folder){
			if(!$folder->is_custom_folder())
				$folders[$id] = $folder;
		}
		return $folders;
	}

	public function custom_folders(){
		$folders = array();
		foreach($this->folders_in_hierarchical_order as $id => $folder){
			if($folder->is_custom_folder())
				$folders[$id] = $folder;
		}
		return $folders;
	}

	public function folders_in_hierarchical_order(){
		$folders = array();

		foreach(Folder::sort_by_name($this->root_folders) as $id => $folder){
			$folders[$id] = $folder;
			$folders = $folders + $folder->descendants_in_hierarchical_order();
		}
		return $folders;
	}

	public function folders_for_move(){ //there has got to be a better name for this
		$folders = $this->folders_in_hierarchical_order();
		if(!is_array($folders)) return $this->error->warning('Unable to determine folders for '.$this->describe()); //something went wrong with the API call

		$CI = get_instance();

		$folders_for_move = array();
		foreach($folders as $id => $folder){
			//don't include our current location in the places where we can move the message to
			if($CI->session->mailbox_location() == $id || $id == 'archived') continue;
			$folders_for_move[$id] = $folder;
		}

		//sent and draft folders are only available options if we're currently in the Archive section
		if($CI->session->mailbox_location() != 'archived'){
			unset($folders_for_move['draft']);
			unset($folders_for_move['sent']);
		}

		return $folders_for_move;
	}

	public function hierarchical_folder_values(){
		$values = array();
		foreach($this->root_folders() as $id => $folder){
			$values[$id] = $folder->hierarchical_values();
		}

		return $values;
	}

	//we cache the folders that we pull from the API for this mailbox so that we don't need to keep calling teh API
	//this will clear that cache so that the folder will be regenerated the next time Mailbox::folders() is called
	public function refresh_folders(){
		$this->_folders = null;
		$_SESSION['mailbox_list'] = get_instance()->mailformat->mailbox_list();
	}

	public function root_folders(){
		$folders = $this->folders;
		foreach($folders as $id => $folder){
			if($folder->has_parent()) unset($folders[$id]);
		}
		return $folders;
	}


/////////////////////
// STATIC FUNCTIONS
/////////////////////

	public static function find_by_email_address($address){
		if(!get_instance()->is->string_like_a_direct_address($address)) return get_instance()->error->should_be_a_direct_address($address);
		$user_name = strip_from_end('@'.CLINICAL_DOMAIN, $address);
		return static::find_one( compact('user_name') );
	}

	public static function find_from_session(){
		$mailbox_group = get_instance()->session->userdata('mailbox_group');
		if(empty($mailbox_group)) $mailbox_group = get_instance()->session->userdata('username');
		if(empty($mailbox_group)) return false;

		$CI = get_instance();
		if(isset($CI->mailbox) && static::is_an_entity($CI->mailbox) && $CI->mailbox->name == $mailbox_group)
			return $CI->mailbox; //skip the db query

		return static::find_one(array('user_name' => $mailbox_group));
	}

	public static function random_password() {
		$chars = array('a','b','c','d','e','f','g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','1','2','3','4','5','6','7','8','9','0','!','@','#','$','%','^','&','*','(',')','{','}','?');
		$pass = '';
		$count = count($chars);
		for($i = 0; $i < 32; $i++) {
			$pass .= $chars[abs(hexdec(bin2hex(openssl_random_pseudo_bytes(6)))%$count)];
		}
		return $pass;
	}

}

