<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

/** */
require_model('api_entity');


/**
* Manages automatic replies for mailboxes using the API.
*
* Values available from the API:
* * id
* * mailbox_id
* * start_at: unix timestamp for when replies should be sent, may be null
* * end_at: unix timestamp for when replies should stop being sent, may be null
* * content: The content to be sent for the reply, may include HTML
*
* Usage:
* <code>
*  //how to find and update
*  $reply = Automatic_reply::find_one( array('mailbox_id' => $this->mailbox->id) ); 
*  echo $reply->content; //all of the values listed above are available as class values
*  $reply->start_at = now(); //you can set the API values as though they were class values
*  $reply->save(); //saving the message will access the API service specified by the $update_resource
*  var_dump($reply->values); //see all the values for this automatic reply taht were sent by the API
* 
* //how to create
*  $reply = Automatic_reply::create( array('content' => $content) );
*  if(!Automatic_reply::is_an_entity($reply))
*     trigger_error('Creation was not successful');
* 
* //how to delete
*  Automatic_reply::delete($reply->id); 
* </code>
*
* @package direct-project-innovation-initiative
* @subpackage models
*/
class Automatic_reply extends API_entity {

	protected $_property_validation_rules = array('start_at' => 'null|unix_timestamp',
												  'end_at' => 'null|unix_timestamp');
	

	static $create_resource = '/direct/automatic_reply/create';
	static $update_resource = '/direct/automatic_reply/update';	
	static $delete_resource = '/direct/automatic_reply/delete';
	static $find_resource = 'direct/automatic_reply/find';
	static $find_one_resource = 'direct/automatic_reply/find';


	//true if the automatic reply is currently turned on (may only be enabled for a given time period)
	public function is_active(){
		if(!isset($this->id)) return false; //if this hasn't been saved, it's not yet in use
		if($this->start_at > now()) return false;
		if(!$this->property_is_empty('end_at') && $this->end_at < now()) return false;
		return true;
	}
	
////////////////////
//  DATA MANAGEMENT
////////////////////

	//overrides parent just to change where we're getting the new create valeus from - should probably just add a hook to the parent for this
	protected function _create(){        
		$error_message = 'Unable to create new '.$this->model_alias;
		if(empty(static::$create_resource))
			return $this->error->warning($error_message.': No resource has been specified for creation');
		
		if(!$this->_run_before_create() || !$this->_run_before_create_and_update()) 
			return $this->error->warning($error_message, 2); //run any specific actions for the child class; quit if actions fail.
		if(!$this->_values_are_valid_for_create() || !$this->_values_are_valid_for_create_and_update()){
			return $this->error->warning($error_message, 2);
		}
		
		//for create, save all writeable values - they may have defaults specified, even if we haven't altered anything
		static::api()->clear();
		$success = static::api()->call(static::$create_resource, $this->_values_for_save(), 'POST');
		if(!$success){
			return $this->error->warning($error_message.' -- API says "'.$this->api->message().'"');
			return false;
		}
		
		$id = element('id', first_element(static::_results_from_api_output()));
		if(!static::formatted_like_an_id($id)) return $this->error->warning($error_message, 2);
		$this->_values[static::$primary_key] = $id; //usually you should use _set_field_value - _create() is the only place where we set this manually
		
		//reload the values in case the database automatically generates any values (timestamps, etc.)    
		$this->load_field_values_from_db();
		
		if(!$this->_run_after_create()) return false; //if any cleanup (deletions) or error messages need to happen on failure, run_after_create should supply them
		if(!$this->_run_after_create_and_update()) return false;
		return true;
	}
		
	protected function _values_for_save(){
		$values = parent::_values_for_save();
		
		//set the mailbox value to the ueer's person mailbox in case they're currently logged into a group mailbox
		//note that future functionality may require changing this if we eventually allow setting automatic replies for groups
		$values['mailbox'] = get_instance()->user->username;
		
		//if this hasn't yet been saved, don't save date values if they're empty, the API doesn't like it
		if($this->property_is_empty('id')){
			if(array_key_exists('start_at', $values) && empty($values['start_at']))
				unset($values['start_at']);
			if(array_key_exists('end_at', $values) && empty($values['end_at']))
				unset($values['end_at']);
		}

		return $values;
	}
	
	protected function _run_after_create_and_update(){ 
		if($this->is_active()) 
			return Automatic_reply::enable_banner();
		else
			return get_instance()->session->dismiss_banner('automatic_replies');	
	}  
	
	protected static function _run_after_delete($entity){
		return get_instance()->session->dismiss_banner('automatic_replies');
	} 	

/////////////////////////
// STATIC
/////////////////////////	


	
	//overrides parent because we don't have a count resource for folders	
    public static function count($id_or_conditions = array()){
		return count(static::find($id_or_conditions));
    }
	
	public static function enable_banner(){
		$CI = get_instance();
		$content = '<strong>Automatic Replies:</strong> Automatic Replies are being sent for this account. '.
				   '<a href="'.site_url('settings/automatic_replies').'" target="_blank">Turn off automatic replies</a>';
		
		return $CI->session->display_banner('automatic_replies', $content);
	}
	
	public static function fields(){
		return array('id', 'mailbox_id', 'start_at', 'end_at', 'content');  
    }
	
	protected static function _results_from_api_output(){
		$output = parent::_results_from_api_output();
		
		if(!is_array($output) || !array_key_exists('automatic_reply', $output))
			return array();
			
		return array($output['automatic_reply']['id'] => $output['automatic_reply']);
	}	
	
	protected static function _conditions_for_find($id_or_conditions){
		$conditions = parent::_conditions_for_find($id_or_conditions);
		$CI = get_instance();
		
		//make sure we have a mailbox - assume that it's the current mailbox if not otherwise specified
		if(!array_key_exists('mailbox', $conditions)){
			//set the mailbox value to the ueer's person mailbox in case they're currently logged into a group mailbox
			//note that future functionality may require changing this if we eventually allow setting automatic replies for groups
			$conditions['mailbox'] = get_instance()->user->username;
		}
		return $conditions;	
	}			
		
	protected static function _delete($entity){ 
		static::api()->clear();
		
		$values = array();
		
		//set the mailbox value to the ueer's person mailbox in case they're currently logged into a group mailbox
		//note that future functionality may require changing this if we eventually allow setting automatic replies for groups
		$values['mailbox'] = get_instance()->user->username;
			
		return static::api()->call(static::$delete_resource, $values, 'POST');
	}
			
		
		
}
