<?php
/**
* @package direct-project-innovation-initiative
* @subpackage libraries
*/ /** */

require_library('form_markup_generator');

/**
* Extension of the Form Markup Generator, specifically for the Patient Disclosure Information form.
*
* In general, it's easiest just to configure an instance of the standard form markup generator, but this form is complex enough that it's easier to give it its own class. 
*
* The Patient Disclosure Information Form is shown every time the user attempts to send a message.  When patient information is shared in a message, we need to collect 
* identifying information about each patient in the message so that we can record the disclosure in accordance with VA regulations.  The following rules are enforced:
*
* * If parsable patient documents (C32s, CCDAs, etc) are attached to the message, the user is required to supply the patient's social security number and the reason for the disclosure.
* * If other files are attached to the message, the user is given the opportunity to provide patient first name, last name, social security number, and the reason for the disclosure.  Note that
*   we cannot accept partial information - if they fill out the last name, they must also fill out all other fields available for that patient.
* * The user will be asked whether or not any patient information was disclosed in the body of the message.  If it was, they are required to fill out disclosure information for at least one patient.
*
* @package direct-project-innovation-initiative
* @subpackage libraries
*/
class Disclosure_form_markup_generator extends Form_markup_generator{
	
	protected $_message; //the message that the user is attempting to send	
	protected $_attachment_disclosures;
	protected $_message_body_disclosures;
	protected $_patient_document_disclosures;
	//protected $_file_transfer_disclosures;


	function __construct($message, $fields = array()){
		parent::__construct($fields);
		$this->set_message($message);
	}

	function validates(){
		//make sure that a message is set - if not, trigger a warning for the developer (no messages needed for users - they should never get to this point)
		if(!Message::is_an_entity($this->message))
			return $this->error->warning("I can't validate the disclosure_form_markup_generator form fields until a message has been supplied");
		
		//make sure that if the user has indicated that there is patient information in the message body, we require at least one patient field	
		return parent::validates() && $this->patients_have_been_entered_for_message_body_if_required();
	}	
	
	
	function validation_messages(){
		$messages = parent::validation_messages();
		
		//if form failed validation because there were no entries for the message body, add that message here
		if(!$this->patients_have_been_entered_for_message_body_if_required())
			$messages[] = 'You have indicated that patient information was disclosed in the message body.  Please enter message body disclosure information for at least one patient.';
		
		return $messages;
	}
	
	/** @return boolean */
	protected function patients_have_been_entered_for_message_body_if_required(){
		
		//make sure we have a message supplied, otherwise we don't have the necessary form fields
		if(!Message::is_an_entity($this->message))
			return $this->error->warning("I can't validate patient disclosure information until a message has been supplied");
		
		if($this->field('disclosures[message_body_has_patient_data]')->value != 'yes')
			return true; //if we selected no, it's not required; if they left it blank, they're going to have a message telling them that it's required
		
		return !$this->property_is_empty('message_body_disclosures');
	}
		
///////////////////////////
// GETTERS FOR CLASS VARS
////////////////////////////
	
	public function disclosures(){
		return array_merge($this->patient_document_disclosures, $this->attachment_disclosures, $this->message_body_disclosures);
	}


	//note to self - it's not a problem that we're caching disclosure information, because it's generated from $_POST and won't change after the page load
	public function attachment_disclosures(){
		if(!isset($this->_attachment_disclosures)){
			$disclosures = array();
			
			foreach(array_diff_key($this->message->attachment_files, $this->message->attached_patient_documents) as $filename => $attachment){
				if($this->field('disclosures[attachments]['.$attachment->html_friendly_name.']')->has_value())
					$disclosures[] = array_merge(array('hash' => $attachment->hash), $this->field('disclosures[attachments]['.$attachment->html_friendly_name.']')->values_for_api());
			}
			
			$this->_attachment_disclosures = $disclosures;
		}
		return $this->_attachment_disclosures;
	}
	
	
	public function message_body_disclosures(){
		if(!isset($this->_message_body_disclosures)){
			$disclosures = array();
			if($this->field('disclosures[message_body_has_patient_data]')->value == 'yes'){			
				foreach($_POST['disclosures']['message'] as $key => $values){		
							
					//the $key should always be formatted "patient-1" - if it's not, assume the $_POST has been tampered with and silently continue to avoid the possibility of log forgery.  See VAD-1723.
					$row_number = substr($key, strpos($key, '-') + 1);
					if(empty($row_number) || !$this->is->integer($row_number))
						continue; 
					
					if($this->field('disclosures[message]['.$key.']')->has_value())
						$disclosures[] = $this->field('disclosures[message]['.$key.']')->values_for_api();																		
				}
			}
			$this->_message_body_disclosures = $disclosures;
		}
		return $this->_message_body_disclosures;
	}
	

	public function patient_document_disclosures(){
		if(!isset($this->_patient_document_disclosures)){
			$disclosures = array();		
										
			//save the disclosures to the API
			foreach($this->message->attached_patient_documents() as $filename => $attachment){
				//make sure that the disclosure has changed - otherwise, no need to save
				$new_ssn = $this->field('disclosures[patient_documents]['.$attachment->html_friendly_name.'][ssn]')->value;
				$new_purpose = $this->field('disclosures[patient_documents]['.$attachment->html_friendly_name.'][purpose_of_disclosure]')->value;					
				if($attachment->ssn == $new_ssn && $attachment->purpose == $new_purpose) continue;
				
				$disclosures[] = array('hash' => $attachment->hash, 'ssn' => str_replace('-', '', $new_ssn), 'purpose' => $new_purpose);
			}
			
			$this->_patient_document_disclosures = $disclosures;
		}
		return $this->_patient_document_disclosures;		
	}

/**
	public function file_transfer_disclosures(){
		if(!isset($this->_file_transfer_disclosures)){
			$disclosures = array();

			foreach($this->message->file_transfers as $file_transfer){
				prp($file_transfer, 'file_transfer');
				$html_friendly_name = make_string_css_friendly($file_transfer['name']);
				if($this->field('disclosures[file_transfers]['.$html_friendly_name.']')->has_value())
					$disclosures[] = array_merge(array('hash' => $file_transfer['hash']), $this->field('disclosures[file_transfers]['.$html_friendly_name.']')->values_for_api());
			}

			$this->_file_transfer_disclosures = $disclosures;
		}
		return $this->_file_transfer_disclosures;
	}
**/



		
///////////////////////////
// SETTERS FOR CLASS VARS
///////////////////////////
	
	protected function set_message($message){
		if(!Message::is_an_entity($message)) return $this->error->property_value_should_be_a_message_entity('message', $this, $message);
		$this->_message = $message;		
		
		//SET UP FIELDS BASED ON THE MESSAGE
				
		//set up the fields for patient documents
		foreach($message->attached_patient_documents() as $filename => $attachment){
			$this->set_field('disclosures[patient_documents]['.$attachment->html_friendly_name.'][ssn]', array(	'type' => 'social_security_number', 
																												'label_text' => $attachment->name.' Social Security Number', 
																												'required' => true));
			$this->set_field('disclosures[patient_documents]['.$attachment->html_friendly_name.'][purpose_of_disclosure]', array('type' => 'dropdown', 
																																 'label_text' => $attachment->name.' Purpose of Disclosure', 
																																 'options' => array('TREATMENT' => 'Medical Treatment'), 
																																 'required' => true));
		}
			
		
		//set up the fields for other attachments
		foreach(array_diff_key($message->attachment_files, $message->attached_patient_documents) as $filename => $attachment){
			$this->set_field('disclosures[attachments]['.$attachment->html_friendly_name.']', array('type' => 'patient_disclosure', 'label_text' => $attachment->name));
		}
	

		//set up fields for file_transfers
		/**
		foreach($message->file_transfers as $file_transfer){
			$this->set_field('disclosures[file_transfers]['.make_string_css_friendly($file_transfer['name']).']', array('type' => 'patient_disclosure', 'label_text' => $file_transfer['name']));
		}
		**/
		
		
		//default to "No" by client request - this will get overriden if a different value is specified in POST - MG 2014-03-11
		$this->set_field('disclosures[message_body_has_patient_data]', array('type' => 'radio_button_group', 
																			 'label_text' => 'Does the body of this message disclose any patient information?',
																			 'referral_label' => 'Message Body',
																			 'options' => array('yes' => 'Yes', 'no' => 'No'),
																			 'required' => true,
																			 'value' => 'no')); 
		
		//set up disclosure fields for the message body. we should always have one, but we may have more than one if the user has added more than one
		for($i=1; $i == 1 || (!empty($_POST['disclosures']) && !empty($_POST['disclosures']['message']) && !empty($_POST['disclosures']['message']['patient-'.$i])); $i++){
			$this->set_field('disclosures[message][patient-'.$i.']', array('type' => 'patient_disclosure', 'label_text' => 'Patient #'.$i));	
		}
		
		
		//since we get this form after submitting the compose form, we need to preserve the compose form values as hidden fields			
		foreach($_POST as $key => $value){
			if($key == 'disclosures') continue; //all of our fields for this form will be under the 'disclosures' key
			/* if the message id was not passed in the POST, but we have a value for the message id, it means we saved a draft of the previously unsaved message
			   in preparation for sending it. Since we've done this, we need to preserve that value as if it was POSTed to us originally so that we get rid
			   of the draft we produced if the message is sent. */
			if($key == 'msg_id') {
				if(empty($value) && !$message->property_is_empty('id')) { $value = $message->id; }
			}

			$this->set_field($key, array('type' => 'hidden', 'value' => $value));
		}
		
		
		return true;
	}

	
}