<?php
/**
* @package vler
* @subpackage libraries
*/

/**
* A parent class for collections of Field markup generators.
*
* Some fields just tend to go together.  If you're asking a user to input a time, you'll probably want to group the hour and minute together visually, and 
* their values are probably destined for the same database column.  If you're asking a user to input an address, you're probably grouping the fields together
* in a standard way each time.  This class gives you a way to create a markup generator that contains a subset of other markup generators that you can treat 
* as a single field in your form markup generator.
*
* This class is intended to be treated as abstract.  When creating extensions to this class, you'll need to set up the {@link _field_templates} array as well as the {@link _type}.  
*
* @uses _field_group.php
* @uses _field_group_field.php
* @uses _label.php
*
* @package vler
* @subpackage libraries
*/
class Field_group_markup_generator extends Field_markup_generator{
	
	protected $_type = 'field_group';
	
	/**
	* The names and types of the field markup generators that should be included in this field group.
	* This array should be formatted in the same way that you'd set up an array of fields to be passed to the {@link Form_markup_generator}, array($field_name => $type),
	* or for fields that you need to configure, array($field_name => array($field_property => $value)).
	* These fields will be generated by {@link load_field_generators}
	* @var array
	*/
	protected $_field_templates = array(); //overload in child classes.  this should be name => type or name => values, same as when you're loading the form mg
	
	/**
	* The field markup generators for this field group.
	* This will be populated by {@link load_field_generators}, and formatted array($field_name => $field_markup_generator_object).  You can access these 
	* {@link Field_markup_generator}s using {@link field()}.
	* @var array
	*/	
	protected $_fields = array();
	
	/**
	* A blanket error message for this field group that will be used before listing multiple issues with the child fields.
	* @see feedback_problems_with_child_fields
	* @var string
	*/
	protected $_feedback_problems_with_child_fields = "Some corrections need to be made to the %s field."; //child classes should implement a specific description of the value


	/**
	* Extends the parent constructor to create the field generators for this field group.
	*/
	//NOTE -- you should always specify values for $name and $values.  We're only allowing empty values so that we don't break get_default_parent_value in Object
	function __construct($name=NULL, $values = array()){
		parent::__construct($name, $values);
		$this->is = $this->validator;
		if(!is_null($name) && !empty($values))
			$this->load_field_generators();
	}
	
	/**
	* Instantiates the field markup generators for this field group.
	* {@link Field_markup_generator} objects will be stored in {@link _fields}, and can be accessed using {@link field()} or {@link fields()}.  Configuration of
	* each markup generator is determined by {@link _field_templates}.
	*/	
	protected function load_field_generators(){
		if($this->property_is_empty('field_templates')) return $this->error->warning("I can't generate fields for a field group when no field templates are specified");
	
		foreach($this->field_templates as $field_name => $values_or_type){
			$field_type = $this->field_type($field_name);

			require_library('form_markup_generator/field_markup_generators/'.$field_type);
			$class = ucfirst($field_type).'_markup_generator';
			$this->_fields[$field_name] = new $class($field_name, $this->field_template($field_name));
		}	
	}
	
	protected function field_type($field_name){
		if(!array_key_exists($field_name, $this->field_templates)) return $this->error->should_be_the_name_of_a_child_field($field_name);
		if(is_string($this->field_templates[$field_name])) return $this->field_templates[$field_name];
		return element('type', $this->field_templates[$field_name]);
	}
	
	protected function field_template($field_name){
		if(!array_key_exists($field_name, $this->field_templates)) return $this->error->should_be_the_name_of_a_child_field($field_name);
		$default_field_values = $this->default_values_for_child_fields();
		$default_field_values['name_for_markup'] = $this->field_name_for_markup($field_name);
		
		if(is_string($this->field_templates[$field_name])){
			$field_values = array();
		}else{
			$field_values = $this->field_templates[$field_name];
			unset($field_values['type']);
		}
		
		return array_merge($default_field_values, $field_values);
	}
	
	protected function default_values_for_child_fields(){
		$default_field_values = array('parent_field_group' => &$this);
		foreach(array('extension_directory', 'partial_directory', 'model_class') as $property){
			if(isset($this->$property)) $default_field_values[$property] = $this->$property;
		}
		return $default_field_values;
	}
	
	/**
	* Generates the default name_for_markup value for the given child field generator.
	* We want the values of this field group to appear in $_POST as an array using the name of this field group as the key, so the {@link _name_for_markup} of the 
	* child field generator should be $this->name_for_markup[$child_name].
	* 
	* @param string $field_name The name of the child field generator, as specified in {@link _fields}
	* return string
	*/
	protected function field_name_for_markup($field_name){
		if(!$this->is->nonempty_string_with_no_whitespace($field_name)) return $this->error->should_be_a_nonempty_string_with_no_whitespace($field_name);
#		if(!$this->property_is_empty('model_class')) return $this->model_class.'['.$this->name.']['.$field_name.']';
		return $this->name_for_markup.'['.$field_name.']';
	}



/////////////////////////////
// ACCESSING FIELDS	
/////////////////////////////	
	
	/**
	* Gives you direct access to one of the fields in this field group.
	* @param string The name of the field, as specified in {@link _fields}
	* @return Field_markup_generator A reference to the field markup generator office
	*/
	function &field($field_name){
		if(!$this->has_field($field_name)) return $this->error->should_be_the_name_of_a_field_in_this_field_group($field_name);
		return $this->_fields[$field_name];
	}
	
	/**
	* True if there's a field by the given name in this group.
	* @param string 
	* @return boolean
	*/
	function has_field($field_name){
		if(!$this->is->nonempty_string($field_name)) return $this->error->should_be_a_nonempty_string($field_name);
		return array_key_exists($field_name, $this->_fields);
	}
	
	/**
	* The names of all the fields in this field group.
	* Format is array('name1', 'name2', 'name3').
	* @return array 
	*/
	function field_names(){
		return array_keys($this->_fields);
	}
	
	/**
	* Collects an array of the value of the given property for each of the fields in this field group.
	* The array will be formatted array('field_name' => 'value')
	* @param string Name of a class variable shared by all of the fields in this field group.
	* @return array
	*/
	function field_values_for_property($property){
		$values = array();
		foreach($this->field_names() as $field_name) $values[$field_name] = $this->_fields[$field_name]->$property;
		return $values;
	}
	
	/**
	* Collects an array of the value of the given method for each of the fields in this field group.
	* The array will be formatted array('field_name' => 'value')
	* @param string Name of a method shared by all of the fields in this field group
	* @param array Arguments to be passed to the method, formatted array(first_param_value, second_param_value, third_param_value)
	* @return array
	*/
	function field_values_for_method($method, $method_arguments = null){
		$values = array();
		if(is_null($method_arguments))
			foreach($this->field_names() as $field_name) $values[$field_name] = $this->_fields[$field_name]->$method();
		else
			foreach($this->field_names() as $field_name) $values[$field_name] = call_user_func_array(array($this->_fields[$field_name], $method), $method_arguments);
		
		return $values;		
	}
		
	
/////////////////////////////
// DEALING WITH FIELD VALUES
/////////////////////////////

	/**
	* The value of this field group.
	* By default, this is the same as {@link values}, but in child classes, you could parse the values of all the fields into a single value.
	* For example, in a class to do with dates/times, it might be useful to compile a unix timestamp here from the child fields.
	* @return array
	*/
	function value(){	
		return $this->values();
	}

	/**
	* An array of the values of all the fields belonging to this field group.
	* Format: array('field_name' => 'value')
	* @return array
	*/	
	function values(){
		return $this->field_values_for_property('value');
	}
	
	/**
	* Set the value of this field group.
	* By default, this is the same as {@link values}, but if you've overridden {@link value} to compile the child field values into a single value,
	* it may be useful to un-compile them here before passing them to the child field groups.
	* @param array
	*/	
	function set_value($value){
		return $this->set_values($value);
	}
	
	/**
	* Set the values for each of the fields belonging to this field group.
	* @param array Formated array(field_name => value)
	*/
	function set_values($values){
		if(!is_array($values)) return $this->error->should_be_an_associative_array($values);
		//if(!$this->is->associative_array($values)) return $this->error->should_be_an_associative_array($values);
		foreach($values as $field_name => $field_value){
			if(!$this->has_field($field_name)) return $this->error->should_be_the_name_of_a_field_in_this_field_group($field_name);
			$this->_fields[$field_name]->value = $field_value;	
		}
	}
	
	function validates(){
		$validates = parent::validates();
		if(!$validates)
			$this->has_error = true;
		return $validates;
	}	
	
	/**
	* Overrides the parent method to take into account each of the child field generators.
	* @return boolean True if value_does_not_exceed_max_length() is true for each of the child field generators.
	*/
	protected function value_does_not_exceed_max_length(){        
		$validities = array_unique($this->field_values_for_method('value_does_not_exceed_max_length'));
		return (count($validities) == 1 && first_element($validities) === true);
    }
	
	protected function has_value(){
		$validities = array_unique($this->field_values_for_method('has_value'));
		return (count($validities) == 1 && first_element($validities) === true);		
	}
	
	/**
	* Overrides the parent method to take into account each of the child field generators.
	* @return boolean True if has_value_if_required() is true for each of the child field generators.
	*/
	protected function has_value_if_required(){
		$validities = array_unique($this->field_values_for_method('has_value_if_required'));
		return (count($validities) == 1 && first_element($validities) === true);
	}
	
	/**
	* Overrides the parent method to take into account each of the child field generators.
	* @return boolean True if has_value_if_any is true for each of the child field generators
	*/
	protected function has_valid_value_if_any(){		
		$validities = array_unique($this->field_values_for_method('has_valid_value_if_any'));
		return (count($validities) == 1 && first_element($validities) === true);
	}

	/**
	* Overrides the parent method to take into account each of the child field generators.
	* @return array Sanitized values for each of the child fields, formatted array(field_name => sanitized_value)
	* @todo is this used?
	*/
	protected function sanitize_value($value){
		if(!is_array($value)) return parent::sanitize_value($value);
		$sanitized_values = array();
		foreach($value as $field_name => $field_value){
			$sanitized_values[$field_name] = $this->_fields[$field_name]->sanitize_value($field_value);
		}
		return $sanitized_values;
	}

	/**
	* Overrides the parent method to take into account each of the child field generators.
	* If this field group is a child of another field group, we'll pass to its parent an array of all the validation messages from this field group's children.
	* If this is the topmost in the hierarchy, we'll return a string.
	* @return string|array
	*/
	function validation_messages(){		
	
		//GOAL -- If there's only one error, just return a single string to be included in our list of error
		//If there are multiple errors to be included and this is the topmost in the hierarchy, return a string that contains a flattened list of the errors
		//If there are multiple errors and this _isn't_ the topmost in the hiearchy, we'll pass the errors as an array to this object's parent.
	
		$messages = array();
	
		//if no values were entered for any child fields and this field group is required, we can just return one required message for the entire field group
		$show_single_required_message = false;	
		if(!$this->has_value_if_required()){			
			$show_single_required_message = true;
			foreach($this->field_names as $field_name){		
				$show_single_required_message = $show_single_required_message && $this->_fields[$field_name]->required && !$this->_fields[$field_name]->has_value_if_required;
			}
			
			if($show_single_required_message){
				$messages[] = $this->feedback_required_field;
			}	
		}
		
		//if we didn't go for the single required message, get all of the validation methods for the child fields
		if(!$show_single_required_message )
			$messages = flatten_array($this->field_values_for_method('validation_messages'));

			
		//remove empty messages (this is probably unnecessary, but keeping in harmless legacy code just in case -- MG 2014-12-30)
		foreach($messages as $key => $value){
			if(empty($value)) 
				unset($messages[$key]);
		}
		
		if(count($messages) == 1) return first_element($messages);
				
		if(count($messages) > 1){
			if($this->is_part_of_field_group)
				return $messages;
			else
				return $this->feedback_problems_with_child_fields.ul($messages);
		}
		
		return parent::validation_messages();		
	}	
	
	///////////////////////////
	// GETTERS
	///////////////////////////
	
	/**
	* @see _feedback_problems_with_child_fields
	* @return Value of {@link _feedback_problems_with_child_fields} with the {@link referral_label} added.
	*/
	function feedback_problems_with_child_fields(){
		return str_replace("%s", $this->link_to_field(), $this->_feedback_problems_with_child_fields);
	}
	
	//////////////////////////
	// SETTERS
	////////////////////////////
	
	/**
	* Overrides the parent method to allow attributes to be passed to the child fields.
	* To do so, include the field name in the attributes array: array(attribute1=>value1, field_name => array(attribute1 => value1))
	* @param array
	*/
	function set_attributes($new_attributes){
		if(!is_array($new_attributes)) return $this->error->should_be_an_array($new_attributes);
		foreach($this->field_names() as $field_name){
			if(array_key_exists($field_name, $new_attributes)){
				$this->_fields[$field_name]->set_attributes($new_attributes[$field_name]);
				unset($new_attributes[$field_name]);
			}
		}
		return parent::set_attributes($new_attributes);
	}

}