<?PHP  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Class Zip_attachment
 * Any changes to this library need to be made in both Webmail and the API.
 * @package vler
 * @subpackage libraries
 */
class Zip_attachment extends Attachment{

	protected $_children; //an array of Attachment objects for the files inside this Zip archive
	protected $_child_patient_documents; //a subset of just the Patient_document_attachments stored in $_children

    /**
     * @return array
     */
    function children(){
		log_message('debug','Who called this ZIP(children) ' . debug_backtrace()[1]['class'].':'.debug_backtrace()[1]['function']);
		if(!isset($this->_children)){
			$children = array();
			$values_to_pass = array('parent' => $this);
			if(isset($this->message_id)) {
                $values_to_pass['message_id'] = $this->message_id;
            }
			$tmp_file = tmpfile();
			fwrite($tmp_file, $this->_binary_string);
			$zip = new ZipArchive;
			if($zip->open(element('uri', stream_get_meta_data($tmp_file))) === TRUE) {
				for($i = 0; $i < $zip->numFiles; $i++){
					$path_in_parent = element('name', $zip->statIndex($i));
					//make sure that this item isn't a directory
					if ( preg_match('/\/$/', $path_in_parent) ) {
						continue;
					}
					
					$binary_string = $zip->getFromIndex($i);
					
					$name = basename($path_in_parent);
					$path_in_parent = strip_from_end($name, $path_in_parent);
					
					if(!empty($path_in_parent))
						$values_to_pass['path_in_parent'] = $path_in_parent;

					$values_to_pass['index_in_parent'] = $i;
					log_message('debug','File ' . $name . ' found in ZIP file ' . $this->name);
					$children[$i] = Attachment::create($name, $binary_string, $values_to_pass);
				}
			}
			fclose($tmp_file);
			$this->_children = $children;
		}
		return $this->_children;
	}

    /**
     * @return array
     */
	public function child_patient_documents(){
		if(!isset($this->_child_patient_documents)){
			$child_patient_documents = array();
			foreach($this->children as $index => $child){
				if(is_a($child, 'Patient_document_attachment')) {
					$child_patient_documents[$index] = $child;
				}
			}

			if (isset($this->is_xdm)) {
			    if ( $this->is_xdm == XDM_SCORE_VALID && count($child_patient_documents) == 0) {
                    // if we think this is a valid XDM, but there are no valid doc, then flag this as an invalid XDM
                    $this->set_is_xdm(XDM_SCORE_INVALID);
                } else if ( $this->is_xdm == XDM_SCORE_INVALID && count($child_patient_documents) > 0) {
                    // if we think this is an invalid XDM, but we do end up finding docs, then flag as a valid XDM
                    $this->set_is_xdm(XDM_SCORE_VALID);
                }
            } else {
			    // same concept as above. but, we are setting the property for the first time
			    if (count($child_patient_documents) == 0) {
                    $this->set_is_xdm(XDM_SCORE_NOT_XDM);
                } else {
                    $this->set_is_xdm(XDM_SCORE_VALID);
                }
            }

			$this->_child_patient_documents = $child_patient_documents;
		}
		return $this->_child_patient_documents;
	}

    /**
     * @param null $zip_index
     * @return false
     */
	public function download($zip_index = null){
	 	if(is_null($zip_index)) {
	 	    return parent::download();
        }
		if(!array_key_exists($zip_index, $this->children)) {
	 	    return $this->error->should_be_an_x('valid index for a child file of '.$this->describe(), $zip_index);
        }

		$this->children[$zip_index]->download();
	}

    /**
     * @param null $zip_index
     * @return false
     */
	public function download_as_pdf($zip_index = null){
	 	if(is_null($zip_index)) {
	 	    return parent::download_as_pdf();
        }
		if(!array_key_exists($zip_index, $this->children)) {
	 	    return $this->error->should_be_an_x('valid index for a child file of '.$this->describe(), $zip_index);
        }

		$this->children[$zip_index]->download_as_pdf();
	}

    /**
     * @return array
     */
	function metadata(){
		$metadata = parent::metadata();
		$metadata['children'] = collect('metadata', $this->children);
		$metadata['child_patient_documents'] = collect('metadata', $this->child_patient_documents());
		return $metadata;
	}

	/**
	* Checks the content of a file and casts it as the appropriate child class.
	* @param string Name of the file
	* @param string Content of the file
	* @param array Any additional metadata, formatted array('message_id' => $message_id, 'directory' => $directory)
	* @return Zip_attachment
	*/
	public static function create($name, $binary_string, $more_values = array()){
		log_message('debug','Who called this ZIP(create) ' . debug_backtrace()[1]['class'].':'.debug_backtrace()[1]['function']);
	    if(!static::matches_file($name, $binary_string)){
		    return get_instance()->error->should_be_an_x('name of a '.humanize(get_called_class()).' file', $name);
        }

        // if we have not validated this has an XDM yet, please do so.
        if ( !isset($more_values['is_xdm']) ) {
            require_library('attachments/XDM_attachment');
            $xdm_class = class_name_for_library('attachments/XDM_attachment');
			$more_values['is_xdm'] = $xdm_class::matches_file($name, $binary_string);
            if ($more_values['is_xdm'] == XDM_SCORE_VALID) {
                return $xdm_class::create($name, $binary_string, $more_values);
            }
        }
        $values = array(
            'name' => $name,
            'binary_string' => $binary_string,
        );
        $values = array_merge($values, $more_values);
		return new Zip_attachment($values);
	}

	/**
	* Updated this to check mime type so we can avoid having to use tmpfile() which is failing occasionally. DS 2018-06-18
	* True if the given file can be cast as an instance of this class.
	* @param string Name of the file
	* @param string Content of the file
	* @return boolean
	*/
	public static function matches_file($name, $binary_string){
		log_message('debug','Who called this ZIP(matches_file) ' . debug_backtrace()[1]['class'].':'.debug_backtrace()[1]['function']);
		if(!parent::matches_file($name, $binary_string)) return false;

		// Determine if this is a zip file by checking it's mime type.
        // Added to prevent using tmpfile() which is failing occasionally
		$finfo = new finfo(FILEINFO_MIME_TYPE);
		if ($finfo) {
			$mime_type = $finfo->buffer($binary_string);
			return $mime_type === 'application/zip';
		}

		// If finfo doesn't work for some reason, use the old way of determining if it's a zip file
		//we can also try to open the file as a zip to see if it's a valid file, even if the extension is different
		$valid_zip = false;
		//do this by writing temp file and attempting to open as zip
		$tmp_file = tmpfile();
		$metadata = stream_get_meta_data($tmp_file);
		$tmp_filename = $metadata['uri'];
		fwrite($tmp_file, $binary_string);
		$zip = new ZipArchive;
		if($zip->open($tmp_filename) === TRUE) { $zip->close(); $valid_zip = true; }
		fclose($tmp_file);

		return $valid_zip;
	}

}
