<?php defined('BASEPATH') OR exit('No direct script access allowed');

/**
* Extension to the CI session that ensures that our cached files are destroyed via session_destroy() and garbage_collection.
*
* Storing in the session has the advantage of assuring that they'll be stored in a non-web-accessible location and that they'll 
* be automatically cleaned up, either when the session is destroyed using session_destroy() or when garbage collection is run
* at a later date.  However, items stored in the usual session will be loaded into the global $_SESSION variable, which can cause
* performance issues if we need to store anything large (such as cached files).  We extended the session library to allow us
* to cache files in the same location as the session and access them via the session.  This extension of the session driver ensures
* that the cached files will be cleaned up as part of the usual session clean up activities.
*
*
* @author M. Gibbs <gibbs_margaret@bah.com>
* @package vler
* @subpackage libraries
*/ 

/**
* @package vler
* @subpackage libraries
*/ 
class VLER_Session_files_driver extends CI_Session_files_driver {

	//when the session is destroyed, destroy our cached files as well	
	public function destroy($session_id){
		$success = parent::destroy($session_id);
		$this->remove_directory(SESSION_CACHE_ROOT.'cache_'.$this->_config['cookie_name'].$session_id);
		return $success;
	}

	//when garbage collection is run, destroy any expired cached files as well
	public function gc($maxlifetime){
		parent::gc($maxlifetime);
		
		//after doing normal collection, clean up the sessions that we've stored in the cache		
		//todo - possible alternate criteria: delete if there is no file session with this name in tmp
		if ( ! is_dir(SESSION_CACHE_ROOT) OR ($directory = opendir(SESSION_CACHE_ROOT)) === FALSE)
		{
			log_message('debug', "Session: Garbage collector couldn't list files under directory '".SESSION_CACHE_ROOT."'.");
			return FALSE;
		}

		$ts = time() - ($maxlifetime * 3); //for our custom session cache, assume a longer session, since we set some higher timeouts for large file transfers

		$pattern = sprintf(
			'/^%s[0-9a-f]{%d}$/',
			preg_quote('cache_'.$this->_config['cookie_name'], '/'),
			($this->_config['match_ip'] === TRUE ? 72 : 40)
		);

		while (($subdir = readdir($directory)) !== FALSE){
			$mtime = filemtime(SESSION_CACHE_ROOT.$subdir); 
			
			//look for PHP tmp files - these theoretically should get cleaned up some other way, but since we can't figure out why they're not being cleaned up, do it here
			//could be something the CI session handler isn't handling properly, could be a problem with Windows and PHP, could be us misunderstanding something ...
			if(preg_match('/php[A-Z0-9]{2,6}.tmp/', $subdir) && is_file(SESSION_CACHE_ROOT.$subdir) && $mtime && $mtime < $ts){
				unlink(SESSION_CACHE_ROOT.$subdir);
				continue;
			}
			
			// If the filename doesn't match this pattern, it's either not a session file or is not ours
			if ( ! preg_match($pattern, $subdir)
				OR ! is_dir(SESSION_CACHE_ROOT.$subdir)
				OR ($mtime = filemtime(SESSION_CACHE_ROOT.$subdir)) === FALSE
				OR $mtime > $ts)
			{
				continue;
			}

			$this->remove_directory(SESSION_CACHE_ROOT.$subdir);
		}
		
		closedir($directory);
		return TRUE;			
	}
	
	//helper function to ensure that we remove the session cache the same way each time
	protected function remove_directory($directory){
		if(!directory_exists($directory)) return true;
		
		delete_files($directory, TRUE);
		rmdir($directory);
		
		if(directory_exists($directory)){
			trigger_error('Unable to remove directory '.$directory, E_USER_NOTICE);
			return false;
		}
			
		return true;
	}

}
