<?php
/**
* @package vler
* @subpackage libraries
*/


//Suggested that you read the documentation for Validation and Error before this class.

	//class variable accessors -- see Ruby on Rails discussion.
	// getters + setters as needed; can be changed and altered without needing to change the code that uses those variables.

		// public $value;
		// $object->value = -3.1456925; //at first, you don't care what the value is; could be anything.

		//but later on, requirements change, and it becomes important that this value should always be an integer greater than 0.  Or maybe you just hadn't cared about enforcing that rule earlier on, but you're running into bugs that you think you could identify more quickly if you were checking for data integrity.  Either way, you can easily add a check to this variable without changing the way that you interact with it.

			// protected $_value;
		  // protected $_property_validation_rules = array( 'value' => 'nonzero_unsigned_integer' )

			// You still should be referring to this variable as $object->value everywhere except any getter or setter methods that you write, but
			// now that you've added the underscore to the variable name, __set() will be called when you set $object->value.  __set() will confirm that there's a
			// corresponding class variable with a property validation rule, and check the value against the rule before setting it.
			//$object->value =  -3.1456925; //this will trigger a warning stating that Object::value should be a nonzero unsigned integer.  $object->_value is null.
			//$object->value = 3; //this will be successful, since 3 is a nonzero unsigned integer;  $object->_value is now 3.

			//a little later on, you find that there are additional actions that you want to take care of whenever you set this variable.  Maybe the value
			//of this variable affects another variable, and you want to make sure it gets adjusted whenever


		// Speaking of getter methods, let's say you find that you want to process the value somehow consistently every time you use it.  Or
		// maybe you want the variable to have a default value that can't be set in the variable declaration because it's generated by another method.

			//prote




/**
* Parent class that helps facilitate type-checking and value validation for class properties
*
* Variable naming conventions for this class and its children:
* <ul>
*	<li>public $var -- public variable that can be freely accessed both within and without the code.</li>
*	<li>protected $var -- readonly when being accessed from outside this class (protects from being written, but no real reason not to let devs access this var).</li>
* <li>protected $_var -- should not be accessed directly from either within or without class; has either a custom getter or setter.</li>
* </ul>
*
* @package vler
* @subpackage libraries
* @author M. Gibbs <gibbs_margaret@bah.com>
*/

//discussion: one of the main purposes of this class is to allow simple validation/enforcement of variable types.  (If you're expecting a variable to be an array,
//you should be getting a warning if it isn't.)  How to do this -- make it inaccessible by adding an underscore, then either use a property validation rule for simple cases
//or use a setter method for more complicted validation rules.

//note -- the normal (empty) function does not work for these properties; use {@link property_is_empty}.  isset() does.

//unsetting properties:  most of the time, you don't want to do this, because unsetting will wipe the value AND the var declaration.
//setting the var to null will preserve the variable declaration, but will make isset($var) evaluate to false.

class Object{

	/** For type-checking and validating values.  @var Validator */
	protected $validator;
	protected $is;

	/** For conveniently generating error messages. @var Error */
	protected $error;

	/**
	* Validation rules to run when setting inaccessible class variables.
	* For any of the protected variables name with the convention "_{var_name}" that require very basic type checking before being set,
	* you can add an entry here instead of creating a setter method.  The format is $var_name => $validation_rule, where $var_name is any class variable
	* prepended with an underscore, and $validation_rule is the name of any method on the {@link validator} object, or the name of the method with "is_a" removed.
	* (That is, the validator has a method name 'is_a_nonzero_unsigned_integer', but you can just write 'nonzero_unsigned_integer' in this array).
	*
	* For more sophisticated behavior for setters, you should write a setter method named "set_{var_name}".
	*
	* In the following example, we're assuming that there's a class variable called $_weasley that needs to be an array.
	* <code>
	* protected $_property_validation_rules = array('weasley' => 'array');
	* $this->weasley = 'fred'; //will trigger a warning and not set the variable because 'fred' is not an array
	* $this->weasley = array('bill', 'charlie', 'percy'); //will work
	* </code>
	*
	* @var array
	*/
	protected $_property_validation_rules = array('property_validation_rules' => 'array');

	/** Instantiates the {@link Error} and {@link Validator} instances */
	function __construct($values = array()){

		if(function_exists('get_instance')){
			$CI = get_instance();
			$this->error = $CI->error;
			$this->is = $this->validator = $CI->is;
		}else{
			$this->error = new Error();
			$this->is = $this->validator = new Validator();
		}

		foreach($values as $property => $value){
		    if ($this->property_exists($property)) {
		        $this->$property = $value;
            }
		}
	}

///////////////////////////
// GETTERS
///////////////////////////

	/**
	* Array containing the validation rules for the class variables of this object.
	* This is a getter for {@link _property_validation_rules}.
	*
	* @uses _property_validation_rules
	* @uses _merge_with_parent_array
	*
	* @return array
	*/
	public function property_validation_rules(){
		return $this->_merge_with_parent_array('property_validation_rules');
	}

//////////////////////////////////////////
// DEALING WITH PROPERTIES / CLASS VARS
//////////////////////////////////////////

	/**
	* True if this class has a variable of this name.
	* This method will check both for a variable named $property and for a protected variable named "_{$property}".
	*
	* @param string Name of the class variable
	* @param mixed (Optional) Either an object or the name of a class.  Defaults to the current class.
	* @return boolean
	*/
	function property_exists($property, $object_or_class_name = ''){
		if($object_or_class_name === '')
			$object_or_class_name = get_class($this);

		return (property_exists($object_or_class_name, $property) || property_exists($object_or_class_name, '_'.$property));
	}

	/**
	* True if this class variable is set.
	* This method will check both for a variable named $property and for a protected variable named "_{$property}".
	*
	*	DEPRECATED --- is now just a wrapper for isset().
	* @param string Name of the class variable
	* @param mixed (Optional) Either an object or the name of a class.  Defaults to the current class.
	* @return boolean
	*/
	function property_is_set($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property);

		return isset($this->$property);
	}

	//unsets for the purposes of the class by setting to null, but doesn't actually call unset() so that the class still knows the property exists.  (exists -- is not set.)
	function unset_property($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property);
		$property_name_possibilities = array('_'.$property, $property);
		foreach($property_name_possibilities as $possibility){
			if(property_exists(get_class($this), $possibility))
				return $this->$possibility = null;
		}
		return $this->error->warning("I was unable to unset ".get_class($this).'::'.$property);
	}

	/**
	* Returns false if the class var has a non-empty and non-zero value.
	* This method operates based on the same rules as PHP's {@link empty()} function.  Due to the custom __get and __set methods used in this class,
	* empty($this->property) will not always be reliable.  This method takes into account the special naming rules (property vs _property).
	*
	* Additionally, sometimes variables tht have a custom getter method will have a default value in the getter method rather than in the variable declaration.
	* This method takes that into account.
	*
	* @param string Name of the class variable
	* @return boolean True if it has an empty or zero value
	*/
	function property_is_empty($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property, get_class($this));
		#if(!isset($this->$property)) return true; //tempting, but sometimes we put the default value into the method, not into the class var.
		if(property_exists(get_class($this), $property)) return empty($this->$property);
		if(method_exists(get_class($this), $property)){
			$value = $this->$property();
			return empty($value);
		}
		$internal_property = '_'.$property;
		if(property_exists(get_class($this), $internal_property))
			return empty($this->$internal_property);

		$value = $this->$property;
		return empty($value);
	}

	/**
	* Returns true if a property validation rule is set up for this property.
	*
	* @uses property_validation_rules
	*
	* @param string Name of a class var
	* @return boolean
	*/
	public function property_has_validation($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property, get_class($this));

		return array_key_exists($property, $this->property_validation_rules);
	}

	/**
	* Checks a potential value for property against the validation rule for that property.
	*
	* @uses _property_validation_rules
	*
	* @param string Name of a class var
	* @param mixed Value for the class var
	* @return boolean
	*/
	public function value_is_valid_for_property($property, $value){
		if(!$this->property_has_validation($property)) return $this->error->should_be_a_property_with_a_validation_rule($property);

		$validation_rule = $this->property_validation_rules[$property];
		return $this->validator->$validation_rule($value);
	}

	/**
	* Intended to be used for configuration arrays -- a way to extend a parent array rather than completely override it.
	*
	* Normally,
	*
	*
	* @param string Name of a class var.
	* @return array
	*/
	protected function _merge_with_parent_array($property){
		if(!$this->property_exists($property))
			return $this->error->property_does_not_exist($property, get_class($this));

		$internal_property = $property;
		if(property_exists($this, '_'.$property))
			$internal_property = '_'.$property;

		if(isset($this->$internal_property) && !is_array($this->$internal_property))
			return $this->error->should_be_an_array($this->$internal_property);

		$parent_value = $this->get_parent_default_value($property);

		if(!empty($parent_value) && $parent_value !== $this->$internal_property){
			if(!is_array($parent_value)){
				$message = "I can't merge ".get_class($this)."->".$property." with its parent value because parent::".$property." is not an array";
				trigger_error($message, $this->level('should_be_an_array'));
			}
			elseif(empty($this->$internal_property))
				return $parent_value;
			else
				return array_merge($parent_value, $this->$internal_property);
		}

		return $this->$internal_property;
	}

	/**
	* Get the default value of a class variable in the parent class.
	* This function will ignore the value of the variable in the current class and get the default value of it for the parent class.
	*
	* <code>
	* class Mother extends Object{
	* 	var	$name = 'Molly';
	* }
	*
	* class Daughter extends Mother{
	* 	var $name = 'Ginny';
	*	}
	*
	* $daughter = new Daughter();
	* echo $daughter->name; //echoes 'Ginny';
	* echo $daughter->get_parent_default_value('name'); //echoes 'Molly'
	*
	* $mother = new Mother();
	* echo $mother->name; //echoes 'Molly';
	* echo $mother->get_parent_default_value('name'); //void, since Object doesn't have a property called 'name'
	* </code>
	*
	* Like most of the methods in this class, this method will check for both a variable named $property and for a protected variable named "_{$property}".
	*
	* @param string Name of the class variable
	* @return mixed
	*/
	protected function get_parent_default_value($property, $object_or_class_name = null){
		if(is_null($object_or_class_name))
			$object_or_class_name = get_class($this);//note -- get_parent_class($this) would always consider $this to be the Object class, not any child classes

		$parent_class = get_parent_class($object_or_class_name);
		if(!empty($parent_class) && $this->property_exists($property, $parent_class)){
			$parent = new $parent_class();
			return $parent->$property;
		}
	}

	//uses func_get_args to allow a variable number of parameters
	function method_return_value_is_empty($method){
		if(!$this->validator->is_a_nonempty_string($method) || !method_exists(get_class($this), $method))
			return $this->error->method_does_not_exist($method, get_class($this));

		$args = func_get_args();
		unset($args[0]); //remove the method name from the function args

		$return_value = call_user_func_array(array($this, $method), $args);
		return empty($return_value);
	}

	static function default_value_for_property($property, $class){
		$instance = new $class();
		if(!$instance->property_exists($property)) return $instance->error->property_does_not_exist($property,  $class);
		return $instance->$property;
	}

	/**
	* Gives read-only access to the class vars without having to write a getter for each one.
	*
	* Note that this is a magical method from PHP.  Do not access this method directly -- this method will be run anytime that you try to access a variable that
	* isn't declared in the class.
	*
	* <ul>
	* <li>public $var - can access without hitting this method</li>
	* <li>protected $var - will hit this method if we're coming from outside the inheritance chain, but we'll grant read-only access</li>
	* <li>protected $_var - will hit this method if we follow convention and access as $this->var.  We'll either use the getter method or grant read-only access</li>
	* </ul>
	*/
	function __get($property){
		$indirect_property = '_'.$property;

		if(method_exists(get_class($this), $property))
			return $this->$property();
		elseif(property_exists(get_class($this), $indirect_property))
			return $this->$indirect_property;
		elseif(property_exists(get_class($this), $property))
			return $this->$property; //we're hitting this because we're out of scope, but we don't mind giving read-only access

		$this->error->property_does_not_exist($property, get_class($this), 1);
		return null;
	}

	function __isset($property){
		if(property_exists(get_class($this), $property))
			return isset($this->$property);

		$indirect_property = '_'.$property;
		if(property_exists(get_class($this), $indirect_property))
			return isset($this->$indirect_property);

		return false;
	}

	/**
	* Gives read-only access to the class vars without having to write a getter for each one.
	*
	* Note that this is a magical method from PHP.  Do not access this method directly -- this method will be run anytime that you try to access a variable that
	* isn't declared in the class.
	*
	* <ul>
	* <li>public $var - can access without hitting this method</li>
	* <li>protected $var - will hit this method if we're coming from outside the inheritance chain.  We won't allow write access unless there's a validator or method set up/li>
	* <li>protected $_var - will hit this method if we follow convention and access as $this->var</li>
	* </ul>
	*/
	function __set($property, $value){
		//if there's a set_property method set up for this property, use it
		$setter_method = 'set_'.$property;
		if(method_exists($this, $setter_method)){
			//ok to have a setter in the current class && a validator in the parent class, but not defined in the same class.
			if(method_exists($this, $setter_method) && $this->property_has_validation($property) && in_array($property, $this->_property_validation_rules)){
				$this->error->notice('You have both a method called set_'.$property.' and a validation entry for '.get_class($this).'->'.$property.'. Only the method will be used');
			}
			return $this->$setter_method($value);
		}

		//if there's a property validation rule set up, use it
		if($this->property_has_validation($property)){
			if($this->value_is_valid_for_property($property, $value)){
				$internal_property = $property;
				if(!property_exists(get_class($this), $property))
					$internal_property = '_'.$property;
				return $this->$internal_property = $value;
			}
			else{
				$property_should_be = 'property_value_should_be_a_'.str_replace('|', '_or_', element($property, $this->property_validation_rules()));
				return $this->error->$property_should_be($property, get_class($this), $value, 1);
			}

		}

		//if there's no such property (public OR private OR protected, etc), let the developer know that they've made a mistake
		if(!$this->property_exists($property)){
			return $this->error->property_does_not_exist($property, get_class($this), 1);
		}

		//if the property exists, then it must not be in scope (otherwise we wouldn't have gotten to this function
		return $this->error->warning('The property "'.$property.'" exists for the '.get_class($this).' class, but it is not in scope');
	}

}
