<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package vler
* @subpackage core
*//** */


/**
* Allows the application to customize the database drivers.
*
* Database drivers may be overriden or extended by placing a file named MY_driver_name.php in APPPATH/database (replace MY_ with your value for $config['subclass_prefix']).
*
* @package vler
* @subpackage core
*/
class VLER_Loader extends CI_Loader{
	
	var $_default_ci_view_paths;
	var $_filepaths = array(); //type (library/model/core/etc) + subdirectory + name => filepath; used to avoid having to determine the filepath for something we've already loaded
	
	function __construct(){
		parent::__construct();
		
		//the parent loader looks at the application first and the CI system directory to find libraries/helpers/models
		//we're going to look at the application first, then our shared libraries, and then the CI system directory
		$this->_ci_core_paths = array(APPPATH, VLERPATH, BASEPATH);
		$this->_ci_library_paths = array(APPPATH, VLERPATH, BASEPATH);
		$this->_ci_helper_paths = array(APPPATH, VLERPATH, BASEPATH);
		$this->_ci_model_paths = array(APPPATH, VLERPATH);
		
		$this->helper('loader'); //make the loader helper available as soon as the loader is available
	}	
	
	/**
	* Overrides parent so that the search of {@link _ci_classes} is case-insensitive.
	* @todo Should probably attempt to make sure that our extensions get added to _ci_classes using the same case as CI in case we run into further issues.
	* @param string
	* @return string|null
	*/
	public function is_loaded($class){
		$class = strtolower($class);
		$classes = array_map('strtolower', $this->_ci_classes);

		//todo - probably a better performance way to do this, but for now, we'll go with this
		if(array_key_exists($class, $classes) && $classes[$class] == $class)
			return $class;
	}	
	
	
	/**
	* Returns the actual class name for a library or model, even if it is an extension.
	*
	* If you've extended the Attachment library, you may need its actual class name if you need to call on its static methods (in case Attachment::create() gives
	* different results than MY_Attachment::create()).  This method will determine the name of the class actually being used.
	*
	* @todo This currently relies on the fact that file names/class names are case-insensitive in Windows.  We'd need to add a few more complicated cases if we need to make sure that we 
	* get the cases of the files exactly right.  At the moment, I'm not going to implement that because I'm pretty sure that Codeigniter 3 enforces cases more strictly and so we won't have
	* to allow for both situations even if this ever moves off of Windows.  -- MG 2015-05-07
	*
	* @param string The type of the file within CodeIgniter - core, helper, library, or model
	* @param string The name of the model/library/etc.  May optionally include a subdirectory (e.g., 'attachment' or 'attachments/word_attachment' are both valid
	* @return string
	*/
	public function class_name_for($type, $name){
		$filepath = $this->require_file($type, $name);
		$file = substr($filepath, strrpos($filepath, '/') + 1);
		$class = substr($file, 0, strrpos(strtolower($file), EXT));
		return $class;	
	}	

	/**
     * Database Loader
	 *
 	 * This method was only slightly modified from the method suggested by Ellis Labs: {@link https://github.com/EllisLab/CodeIgniter/wiki/Extending-Database-Drivers Extending Database Drivers}.
     *
     * @param    string    the DB credentials
     * @param    bool    whether to return the DB object
     * @param    bool    whether to enable active record (this allows us to override the config setting)
	 * @return   CI_DB
     */
    public function database($params = '', $return = FALSE, $query_builder = NULL)
    {
        // Grab the super object
        $CI =& get_instance();
 
        // Do we even need to load the database class?
		if ($return === FALSE && $query_builder === NULL && isset($CI->db) && is_object($CI->db) && ! empty($CI->db->conn_id)){
			return FALSE;
		}

		require_once(BASEPATH.'database/DB.php');
		$db = &DB($params, $query_builder);
		
		$custom_driver_possibilities = array( APPPATH => config_item('subclass_prefix').'DB_'.$db->dbdriver.'_driver',
											  VLERPATH => 'VLER_DB_'.$db->dbdriver.'_driver');
											  									  

		foreach($custom_driver_possibilities as $directory =>$custom_driver){											  
			$custom_driver_file = $directory.'database/'.$custom_driver.EXT;
	
			
			if (file_exists($custom_driver_file)){
				require_once($custom_driver_file);
				//Ellis Labs suggest you do this to get the configuration that DB() provides, BUT this will wipe out any class variable overrides that you've set in your extesnion
				//As far as I can tell at the moment, the best way to override the default class variables is to define them in the database config file
				//$params[$database_name][$class_var_name] = $class_var_value will do the trick -- MG 2015-08-16
				$default_values = get_object_vars($db); 
				$db = new $custom_driver($default_values);
				break;
			}
		}
		
		if ($return === TRUE) return $db;
 
        // Initialize the db variable.  Needed to prevent
        // reference errors with some configurations
		$CI->db = '';

		// Load the DB class
		$CI->db =& $db;
		return $this;
    }

	
	/** 
	* Extends parent to include helpers and extension in vler-libraries.
	* See {@link require_file} for more information on how inheritanc eis handled.
	* @param string|array Either the name of the helper or an array of the helper names
	*/
	public function helper($helpers = array()){
		foreach ($this->_ci_prep_filename($helpers, '_helper') as $helper){
			if (isset($this->_ci_helpers[$helper])){
				continue;
			}
			
			$path = $this->require_file('helper', $helper);
			if(empty($path))
				show_error('Unable to load the requested file: helpers/'.$helper.EXT);

			$this->_ci_helpers[$helper] = TRUE;
			log_message('info', 'Helper loaded: '.$helper);
			
			// unable to load the helper
			if ( ! isset($this->_ci_helpers[$helper])){
				show_error('Unable to load the requested file: helpers/'.$helper.EXT);
			}
		}
		
		return $this;
	}		
		
	/** 
	* Extends parent to ensure that Entity models can't be loaded using CI's standard singleton pattern.
	* Use {@link require_model} to simply include the file without instantiating anything, or {@link new_instance_of} to create a new instance
	* of the model without storing it on the controller.
	*/
	public function model($model, $name = '', $db_conn = FALSE){
		if(empty($model) || is_array($model)) return parent::model($model, $name, $db_conn);
		
		$path = '';

        // Is the model in a sub-folder? If so, parse out the filename and path.
        if (($last_slash = mb_strrpos($model, '/')) !== FALSE){
            $path = mb_substr($model, 0, $last_slash + 1); // The path is in front of the last slash
            $model = mb_substr($model, $last_slash + 1); // And the model name behind it
        }
		
		//check to see if this is an entity - if so, it shouldn't be loaded.  
		//entity models follow a different design pattern - instances are meant to represent a record in the database.  any methods that deal with more than one record should be static.
		$this->require_model($model); //make sure that we get all of the parent files included without the models having to include them.
		if(class_exists($model) && is_subclass_of($model, 'Entity')){
			return get_instance()->error->warning( $model.' is an entity model and should not be loaded using the CI loader.  To bring the class into scope, use the require_model() method or the standard PHP require/include functions');
		}		
		
		return parent::model($model, $name, $db_conn);
    }		
	
	/**
	* Returns true if there are file paths that match this model name.
	* @param string
	* @return boolean 
	*/
	function model_exists($model){
		return !empty($this->file_paths('model', $model));		
	}
		
	/**
	* Instantiates a new instance of a library or model and returns it.
	*
	* The standard loader methods always attempt to treat librarys and classes as though they should be singleton instances, but sometimes you just
	* need to instantiate an instance of a library and *not* store it on the controller.  This method will allow you to do so without having to know
	* the class name of any extensions of a library that you may have created.
	*
	* See {@link class_name_for} for caveats about case-insensitivity.
	*
	* @param string The type of the file within CodeIgniter - core, helper, library, or model
	* @param string The name of the model/library/etc.  May optionally include a subdirectory (e.g., 'attachment' or 'attachments/word_attachment' are both valid
	* @param array Any parameters that should be passed to the constructor of the model/library
	* @return object
	*/
	public function new_instance_of($type, $name, $parameters){
		//VALIDATE PARAMETERS.  Note that we shouldn't use libraries within the loader class, so we need to trigger errors the old-fashioned way.
		if(!in_array($type, array('library', 'model'))){
			trigger_error('I expected either "library" or "model", but you gave me '.$type, E_USER_WARNING);
			return false;
		}
		
		if(!is_string($name) || empty($name)){
			trigger_error('$name should be a nonempty string, but you gave me '.$name, E_USER_WARNING);
			return false;
		}
		
		if(!is_array($parameters)){
			trigger_error('$name should be an array, but you gave me '.$parameters, E_USER_WARNING);
			return false;			
		}

		//note - if we run into issues with this, check the comments on the newInstanceArgs methods - someone claimed that there were sometimes issues
		//running this on classes that don't have a constructor.  		
		$class = new ReflectionClass($this->class_name_for($type, $name));  //note that class_name_for will require the file, so we don't need to worry about it
		return $class->newInstanceArgs($parameters);
	}
	
	/**
	* Makes sure that a file path has the appropriate extension (usually .php) when you're not sure whether or not it's been included.
	* @param string
	* @return string
	*/
	function path_with_extension($path){
		$_ci_ext = pathinfo($path, PATHINFO_EXTENSION);
		return ($_ci_ext == '') ? $path.EXT : $path;
	}
	
	/**
	* @see require_file
	* @param string Name of the library.  This may include a subdirectory: e.g. 'xml_document' or 'xml_documents/ccda'
	* @param string|null (Optional) Name of the subdirectory, if not included in the $library parameter
	*/
	public function require_library($library, $subdirectory=null){		
		return $this->require_file('library', $library, $subdirectory);
	}
	
	/**
	* @see require_file
	* @param string Name of the model
	* @param string|null (Optional) Name of the subdirectory, if not included in the $model parameter
	*/
	public function require_model($model, $subdirectory=null){	
		if(!class_exists('CI_Model'))
			$this->require_file('core', 'model');	
		return $this->require_file('model', $model, $subdirectory);	
	}	
	

	/**
	* @param string
	* @return boolean
	*/
	public function view_exists($path){		
		foreach($this->_ci_view_paths as $possibility => $number_that_is_here_for_unknown_ci_reasons){
			if(file_exists($possibility.$this->path_with_extension($path)))
				return true;
		}
		
		return false;
	}	
	
///////////////////////////////////////////////
// PROTECTED METHODS
///////////////////////////////////////////////	

	/**
	* Finds the file paths for a helper, model, library, or core library, as well the paths for its parents.
	*
	* So, for an application that has extended this Loader class, the return values would be:
	* 
	* <code>
	*   array(APPPATH.'core/MY_Loader',
	*         VLERPATH.'core/VLER_LOADER',
	* 		  BASEPATH.'core/Loader');
	* </code>
	*
	* The paths wil be returned in order of inheritance: application ptahs first, vler-libraries paths next, CI paths last. The return value will be limited to paths that
	* (a) exist on the file system and (b) are included in the inheritance path - ie, if an application has replaced the string helper entirely instead of extending it, we'll
	* return only the path for the application's version of the string helper.
	*
	* @todo This currently relies on the fact that file names/class names are case-insensitive in Windows.  We'd need to add a few more complicated cases if we need to make sure that we 
	* get the cases of the files exactly right.  At the moment, I'm not going to implement that because I'm pretty sure that Codeigniter 3 enforces cases more strictly and so we won't have
	* to allow for both situations even if this ever moves off of Windows.  -- MG 2015-05-07
	*
	* @param string The type of the file within CodeIgniter - core, helper, library, or model
	* @param string The name of the model/library/etc.  May optionally include a subdirectory (e.g., 'attachment' or 'attachments/word_attachment' are both valid
	* @parm string|null (Optional) The name of a subdirectory that this library/model/etc is in.  For example, for a library in /libraries/attachments/, you would supply 'attachments'
	* @return array The path for this model/library/etc and the paths for its parent classes
	*/
	protected function file_paths($type, $name, $subdirectory=null){
		
		//VALIDATE THE PARAMETERS (without using the Validator class, since that will be loaded using this class)
		if(!in_array($type, array('core', 'helper', 'model', 'library'))){
			trigger_error('$type must be a known type (core, helper, model, or library), but you gave me '.$type);
			return false;
		}
	
		if(!is_string($name) || empty($name)){
			trigger_error('$name should be a nonempty string, but you gave me '.$name);
			return false;
		}
		
		if(!empty($subdirectory) && !is_string($subdirectory)){
			trigger_error('$subdirectory should be null or a nonempty string, but you gave me '.$subdirectory);
			return false;
		}
		
		//check to see if the subdirectory was included in the name (eg attachments/word_attachment), and parse as needed
		if(empty($subdirectory) && !empty(strpos($name, '/'))){
			$subdirectory = substr($name, 0, strpos($name, '/')+1);
			$name = substr($name, strpos($name, '/') + 1);
		}
		
		if(empty($subdirectory)){
			$subdirectory = ''; //since we're doing the fuzzy empty check, make sure this isn't set to 0 or something
		}
		
		if(!empty($subdirectory) && substr($subdirectory, strlen($subdirectory)-1, 1) != '/'){
			$subdirectory .= '/'; //make sure that $subdirectory has a trailing slash
		}
	
		//SET UP ALL POSSIBLE FILE PATHS
		$directory = $type.'s/';
		if($directory == 'librarys/') $directory = 'libraries/';
		if($directory == 'cores/') $directory = 'core/';
			
		$possibilities = array(APPPATH.$directory.$subdirectory.config_item('subclass_prefix').$name.EXT,
							   APPPATH.$directory.$subdirectory.$name.EXT, //if the class is trying to replace an entire helper, we're not going to stop them
						 	   VLERPATH.$directory.$subdirectory.'VLER_'.$name.EXT,
							   VLERPATH.$directory.$subdirectory.$name.EXT);
		
		//theoretically, developers can specify additional paths.  while we're privileging our extension possibilities, allow the others paths to be included.
		$ci_paths_var = '_ci_'.$type.'_paths';	
		if(isset($this->$ci_paths_var)){			   
			foreach($this->$ci_paths_var as $path){
				$possibility =  $path.$directory.$subdirectory.$name.EXT;
				if(!in_array($possibility, $possibilities))
					$possibilities[] = $possibility;
			}
		}		
		
		//REMOVE ANY FILE PATHS THAT DON'T EXIST
		foreach($possibilities as $key => $possibility){
			//make sure the file exists
			if(!file_exists($possibility)){
				unset($possibilities[$key]);
			}
		}
		
		//LOOK FOR REPLACEMENTS OF PARENT LIBRARIES
		//If the application has completely overridden the CI or VLER library by just replacing it in the application, we don't want to include the files they're replacing 
		foreach($possibilities as $key => $possibility){	
			//if this is a replacement of the library, we stop here and return the results so far
			$replacement = $directory.$name.EXT;
			if(mb_strrpos($possibility, $replacement) === (mb_strlen($possibility) - mb_strlen($replacement))){
				return array_slice($possibilities, 0, $key + 1);
			}
		}
		
		if(empty($possibilities) && empty($subdirectory)){
			return $this->file_paths($type, $name, $name); //sometimes libraries/models/etc are in a subdirectory named the same thing as them
		}
		
		return $possibilities;
	}
	
	/**
	* Requires the files for a helper, model, library, or core library instead of loading it.
	*
	* For classes, the CI concept of loading involves instantiating the class and storing a singleton instance of it on the controller.  This is
	* not always desired behavior, so this method allows us to simply require it instead.  This method still follows the CI inheritance principles
	* to allow CI or VLER libraries to overloaded or overridden - see {@link file_paths()} to see how the file paths are determined.  
	* 
	* @param string The type of the file within CodeIgniter - core, helper, library, or model
	* @param string The name of the model/library/etc.  May optionally include a subdirectory (e.g., 'attachment' or 'attachments/word_attachment' are both valid
	* @parm string|null (Optional) The name of a subdirectory that this library/model/etc is in.  For example, for a library in /libraries/attachments/, you would supply 'attachments'
	* @return string The path for the model/library/etc being called - this is the full file path, e.g. APPPATH/core/VLER_Loader.php.
	*/
	public function require_file($type, $name, $subdirectory=null, $vars_in_scope = array()){
		//VALIDATE THE PARAMETERS
		if(!in_array($type, array('core', 'helper', 'model', 'library'))){
			trigger_error('$type must be a known type (core, helper, model, or library), but you gave me '.$type);
			return false;
		}
	
		if(!is_string($name) || empty($name)){
			trigger_error('$name should be a nonempty string, but you gave me '.$name);
			return false;
		}
		
		if(!empty($subdirectory) && !is_string($subdirectory)){
			trigger_error('$subdirectory should be null or a nonempty string, but you gave me '.$subdirectory);
			return false;
		}
	
		if(empty($subdirectory) && !empty(strpos($name, '/'))){
			$subdirectory = substr($name, 0, strpos($name, '/')+1);
			$name = substr($name, strpos($name, '/') + 1);
		}
		
		if(empty($subdirectory)){
			$subdirectory = ''; //since we're doing the fuzzy empty check, make sure this isn't set to 0 or something
		}
		
		//tracing through all possible file paths is expensive enough that we only want to do it once - return the file path for this library/helper/model if we've found it already
		$filepath_key = $type.'_'.strtolower($name);
		if(array_key_exists($filepath_key, $this->_filepaths))
			return $this->_filepaths[$filepath_key];
			
		
		$paths = $this->file_paths($type, $name, $subdirectory);
		
		if(empty($paths)) show_error('Unable to load the requested '.$type.': '.$subdirectory.$name);
		if($type != 'helper') $paths = array_reverse($paths); //classes need their parent to be included first; functional code needs the child included first.
		
		//starting with the root of the inheritance chain, include the files for this library and its ancestors
		foreach($paths as $path){
			if(!file_exists($path)) show_error('Unable to load the requested '.$type.': '.$subdirectory.$name);
			
			//include once will enforce that we don't load files more than once; we don't need to track that we've loaded the class already because we're not using the singleton functionality
			$this->_require_file($path, $vars_in_scope);
		}
		
		if($type != 'helper' && !class_exists($name)){
			if(empty($paths)) show_error('Unable to load the requested  '.$type.': '.$subdirectory.$name);
		}
		
		$this->_filepaths[$filepath_key] = $path;
		
		return $path; 
	}	
	
	//since we're allowing variables to be in scope when requiring the file, we need this in a separate helper method so that there are as few potential var name conflicts as possible
	protected function _require_file($__require_file_path, $vars_in_scope){
		if(array_key_exists('__require_file_path', $vars_in_scope)){
			trigger_error('$vars_in_scope cannot have a key named $__require_file_path', E_USER_WARNING);
			return false;
		}
		
		foreach($GLOBALS as $global_name => $value){
			if(array_key_exists($global_name, $vars_in_scope)){
				if($value !== $vars_in_scope[$global_name]){
					trigger_error('$vars_in_scope cannot include a key named \"$global_name\" - this would overwrite a global variable');
				}
				unset($vars_in_scope[$global_name]);
			}
		}
		
		extract($vars_in_scope, EXTR_SKIP);
		include_once($__require_file_path);
	}
	
	
	/**
	* Overloads parent to allow entities to be auto-required.
	* Entities should not be loaded, since loading involves instantiating the class and storing a singleton instance on the $CI controller, but it's still nice to be able to auto-require
	* entity files.
	* @todo Consider addding equivalent for libraries
	*/
	protected function _ci_autoloader(){	
		$BM =& load_class('Benchmark', 'core'); 
		$BM->mark('autoload_start');
		parent::_ci_autoloader();
		
		if (file_exists(APPPATH.'config/autoload.php')){
			include(APPPATH.'config/autoload.php');
		}

		if (file_exists(APPPATH.'config/'.ENVIRONMENT.'/autoload.php')){
			include(APPPATH.'config/'.ENVIRONMENT.'/autoload.php');
		}

		if ( ! isset($autoload)){
			return;
		}
				
		//Entity models can't be "loaded" in that they're not meant to be a single instance saved as a class var on the $CI object
		//but sometimes we want to automatically require their files so that we don't have to worry whether or not they're in scope
		if(isset($autoload['entity_models'])){
			foreach( $autoload['entity_models'] as $model){
				$this->require_model($model);
			}
		}
		$BM->mark('autoload_end');
	}
	
	//overrides parent so that we can divide the code into smaller, more extensible portions
	//also, to automatically look for vler libraries
	//note two changes from parent:
		//parent manually did case-insensitive search when trying to find filenames.  This isn't necessary on Windows, so we've been too lazy to implemment it.  (CI v3 will change this anyway)
		//if you forgot to specify a subdirectory & loader couldn't find the library, parent would assume that it was in a subdirectory named $class.  This is unnecessary for us and encourages sloppy development, so we've skipped it for now.
	protected function _ci_load_library($library, $params = NULL, $object_name = NULL)	{		
		// Get the class name, and while we're at it trim any slashes. The directory path can be included as part of the class name, but we don't want a leading slash
		$library = str_replace(EXT, '', trim($library, '/'));
		$subdir = '';		
		if(!empty(strpos($library, '/'))){ //if there's no slash (or if it's the first char), we're done, there's no subdirectory
			// Was the path included with the class name? We look for a slash to determine this
			if (($last_slash = strrpos($library, '/')) !== FALSE){
				$subdir = substr($library, 0, ++$last_slash);
				$library = substr($library, $last_slash); // Get the filename from the path
			}
		}
		
		$filepath = $this->require_library($library, $subdir);
		if(empty($filepath)){
			// One last attempt. Maybe the library is in a subdirectory, but it wasn't specified?
			if(empty($subdir)){
				return $this->_ci_load_library($library.'/'.$library, $params, $object_name);
			}		
		
			show_error("Unable to load the requested library: ".$library);
		}
		
		$file = substr($filepath, strrpos($filepath, '/') + 1);		
		$prefix = substr($file, 0, strrpos(strtolower($file), strtolower($library).EXT)); 
		
		if($this->is_loaded($library)){
			// Before we deem this to be a duplicate request, let's see if a custom object name is being supplied.  If so, we'll return a new instance of the object
			if(!empty($object_name)){
				$CI =& get_instance();
				if(!isset($CI->$object_name)){
					return $this->_ci_init_library($library, $prefix, $params, $object_name);
				}
			}

			return;
		}

		return $this->_ci_init_library($library, $prefix, $params, $object_name);
	}

	protected function _ci_init_library($class, $prefix, $config = FALSE, $object_name = NULL){
#TODO - original environment/config logic should be incorporated into require_file

		$path = $this->require_file('library', $class);
		if(empty($prefix) && substr($path, 0, strlen(BASEPATH)) == BASEPATH){
			$prefix = 'CI_'; //default to $CI prefix if the path is in the Codeigniter system files
		}

		$class_name = $prefix.$class;

		// Is the class name valid?
		if (!class_exists($class_name, FALSE)){
			log_message('error', 'Non-existent class: '.$class_name);
			show_error('Non-existent class: '.$class_name);
		}

		// Set the variable name we will assign the class to
		// Was a custom class name supplied? If so we'll use it
		if (empty($object_name)){
			$object_name = strtolower($class);
			if (isset($this->_ci_varmap[$object_name]))
			{
				$object_name = $this->_ci_varmap[$object_name];
			}
		}

		// Don't overwrite existing properties
		$CI =& get_instance();
		if (isset($CI->$object_name)){
			if ($CI->$object_name instanceof $class_name){
				log_message('debug', $class_name." has already been instantiated as '".$object_name."'. Second attempt aborted.");
				return;
			}

			show_error("Resource '".$object_name."' already exists and is not a ".$class_name." instance.");
		}

		// Save the class name and object name
		$this->_ci_classes[$object_name] = $class;

		// Instantiate the class
		$CI->$object_name = isset($config) ? new $class_name($config) : new $class_name();
	}
}