<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models
*/

/** */
require_model('direct_message');

/**
* System-Wide Messaes
*
* The VA has requested the ability for admins to be able to send a message to all Direct users.  This model represents the OUTGOING message that either will be or has been sent to all Direct users.
* System-wide messages will vary from normal Direct messages in that:
* - They will not be sent through the mail server, and will instead be inserted directly into the database.
* - To, CC, and BCC cannot be set.  The recipient will be hardcoded to the all users mock address.
* - The "From" value will reflect the notifications@direct address instead of the sender.  The original_sender field will record the actual sender.
* - The message cannot belong to a custom folder.
* - The message cannot be archived.
* - No patient data should be included in the message or its attachments, so disclosures will not be associated with this model.
*
* This model will only represent outgoing system-wide messages: incoming messages will be treated as standard Direct messages in a user's mailbox.
*
* @package direct-as-a-service
* @subpackage models
*/
class System_wide_message extends Direct_message{

	///////////////////////////
	// INSTANCE VARS
	///////////////////////////

	protected $_to = ALL_USERS_MOCK_ADDRESS;  //only valid value for to is the ALL_USERS_MOCK_ADDRESS

	protected $_readonly_fields = array(  'to', //to will be prepopulated with ALL_USERS_MOCK_ADDRESS
										  'cc', //no cc - only one recipient & that will be in 'to'
										  'bcc', //no bccs - only one recipient & that will be in 'to'
										  'archived', //system-wide messages cannot be archived
										  'protected_data', //system-wide messages will not have protected data
										  'folder_id', //we don't allow system-wide messages to be placed in custom folders
										);

	/**
	* Changes the message state to 'sent' and sends the message by inserting it into the database.
	*
	* Note that the raw_mime value is derived using the Mime_generator class, a descendent of the CI Email library, and will be configured in exactly the same way as the CI Email Library using
	* {@link _configure_mail_object}.
	*
	* @return boolean
	*/
	public function send(){
		if(!isset($this->id)) return $this->error->warning('Please save '.$this->describe().' before sending'); //saving will validate a number of values, so we need to do this first

		//make sure that the mailbox is active & we're allowed to send mail
		$mailbox = $this->mailbox;
		if(!$mailbox->is_active) return $this->error->warning('Cannot send '.$this->describe().'; '.$mailbox->describe().' is not active');

#TODO - Double check permissions?

		//make sure that the message values are valid for send
		if(!$this->draft) return $this->error->warning(ucfirst($this->describe()).' is not a draft and may not be sent');
		if($this->archived == MSG_ARCHIVED) return $this->error->warning(ucfirst($this->describe()).' has been archived and may not be sent until it has been restored.');
		if($this->property_is_empty('sender')) return $this->error->warning('Please specify a sender for '.$this->describe().' before sending');
		if(!$this->has_recipients()) return $this->error->warning('Please specify at least one recipient for '.$this->describe());

		$success = $this->db->query("INSERT INTO mail.dbo.messages (recipients, sender, [to], cc, bcc, attachments, subject, plain, html, [timestamp], folder_id, [size], flags, headers, raw_mime, seen, draft, sent, archived, message_id, mailbox_id, mailtype, priority, original_sender_id, protected_data)
											SELECT
												messages.recipients,
												'".EXTERNAL_EMAILS_FROM."' AS sender,
												messages.[to],
												NULL AS cc,
												NULL AS bcc,
												messages.attachments,
												messages.subject,
												messages.plain,
												messages.html,
												messages.timestamp,
												messages.folder_id,
												messages.size,
												messages.flags,
												messages.headers,
												messages.raw_mime,
												0 as seen,
												0 as draft,
												0 as [sent],
												0 as archived,
												NULL as message_id,
												mailboxes.id AS mailbox_id, 
												messages.mailtype, 
												messages.priority,
												messages.original_sender_id,
												messages.protected_data
											FROM mail.dbo.mailboxes 
											JOIN mail.dbo.messages ON messages.id=".$this->id."
											WHERE mailboxes.is_active=1 AND mailboxes.is_group=0");


		//first, make sure we can record the change for this in the database
		$this->_set_field_value('sender', EXTERNAL_EMAILS_FROM, 0, TRUE); //draft is usually readonly, so override the validation in this case only
		$this->_set_field_value('draft', false, 0, TRUE); //draft is usually readonly, so override the validation in this case only
		$this->_set_field_value('sent', true, 0, TRUE); //sent is usually readonly, so override the validation in this case only
		$this->_set_field_value('timestamp', now(), 0, TRUE); //timestamp is usually readonly, so override the validation in this case only
		$this->save();
		if(!$this->sent || $this->draft){
			return $this->error->warning('Unable to transition '.$this->describe().' from draft message to sent message.  The message has still been sent to the recipients.');
		}

		return $success;
	}

	protected function _values_for_save(){
		$values_for_save = parent::_values_for_save();
	        if(!isset($this->id)){
	        	$values_for_save['to'] = $this->to;
	        }
	        return $values_for_save;
	    }

	protected static function _set_conditions($id_or_conditions=array(), $offset=0){
		return (static::db()->where("([to]='".ALL_USERS_MOCK_ADDRESS."' AND (draft=1 OR sent=1))") && parent::_set_conditions($id_or_conditions, $offset));
	}

}
?>
