<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models-folder
* @filesource
*/

/** */
require_model('entity');

/**
* @package direct-as-a-service
* @subpackage models
*/
class Request extends DAAS_Entity {
	static $table = 'request';	
		
	//validation rules for settable properties on the method - these correspond to Validator methods		
	protected $_property_validation_rules = array('application_id' => 'nonzero_unsigned_integer',
												  'call' => 'nonempty_string',
												  'call_date' => 'unix_timestamp',
												  'response_code' => 'nonzero_unsigned_integer');
	

///////////////////
// INSTANCE
///////////////////	



	//if we're displaying $this->call to the user, we need to make sure it doesn't include any PII/PHI
	public function call_for_display(){
		$call = $this->call;
	
		//check for ssn in any query string and ensure it doesn't display SSN
		$parts = parse_url(strtok(strtolower($call),' '));
		if(!empty($parts['query'])) {
			parse_str($parts['query'], $query_str);
			if(!empty($query_str) && is_array($query_str)) {
				if(array_key_exists('ssn', $query_str)) {
					$sanitized_ssn = '';
					for($i = 0; $i < strlen($query_str['ssn']); $i++) {
						if($i < (strlen($query_str['ssn']) - 4)) { $sanitized_ssn .= '*'; }
						else { $sanitized_ssn .= $query_str['ssn'][$i]; }
					}
					$call = str_replace($query_str['ssn'], $sanitized_ssn, $call);
				}
			}
		}
		
		if(!empty($parts['path'])) {
			 //check if query string is being passed the CodeIgniter way and so the same thing if so
				$parts = explode('/', $parts['path']);
				array_change_key_case($parts, CASE_LOWER);
				$ssn_index = array_search('ssn', $parts);
				if($ssn_index) {
					$ssn = $parts[$ssn_index+1];
					if(is_numeric($ssn)) {
						$sanitized_ssn = '';
						for($i = 0; $i < strlen($ssn); $i++) {
							if($i < (strlen($ssn) - 4)) { $sanitized_ssn .= '*'; }
							else { $sanitized_ssn .= $ssn[$i]; }
						}
						$call = str_replace($ssn, $sanitized_ssn, $call);
					}
				}
		}
		
		return $call;	
	}

	public function human_readable_call_date(){
		if($this->property_is_empty('call_date')) return '';
		return date('m/d/Y h:i:s A', $this->call_date);
	}
	
	public function response_for_display(){	
		//hide PII in response if it's a Disclosure::audit_get() call
		if(string_begins_with('/disclosure', $this->call) && !string_begins_with('/disclosure/summary', $this->call) && $this->response_code == 200) { 
			$CI = get_instance();
			return $CI->json->encode(array('request_id'=>$CI->json->decode($this->response)->request_id));
		}
		return $this->response;
	}	

	function values_for_report(){
		$values = $this->values();
		$values['application_id'] = $values['application_name'];
		unset($values['application_name']);
		$values['call_date'] = $this->human_readable_call_date();
	
		foreach($values as $key => $value){
			$method_for_display = $key.'_for_display';
			if(method_exists($this, $key.'_for_display'))
				$values[$key] = $this->$method_for_display();
		}
	
		return $values;
	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
// GETTERS
// These are accessors for class variables prefaced by a _ and for database field values for this row 
// of the message table.  You can access these values by just treating them as normal variables, and
// {@link __get()} will obtain the value for you.  
///////////////////////////////////////////////////////////////////////////////////////////////////////////

	function application_id(){
		//autogenerate the application id _only_ if the request hasn't been saved to the database yet
		if(!isset($this->id) && !array_key_exists('application_id', $this->_values)){ 
			$headers = getallheaders();
			if(isset($headers['Authorization'])) { //if no authorization header, we don't have an option for this
				$auth_header = explode(' ',$headers['Authorization']);
				$key_hash = explode(':',$auth_header[1]);
				$public_key = $key_hash[0];
				$application_id = get_instance()->applicationmodel->get_application_id_from_public_key($public_key);
				if($this->is->nonzero_unsigned_integer($application_id)){
					$this->_values['application_id'] = $application_id;
				}				
			}			
		}
		return $this->_values['application_id'];
	}




////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SETTERS
// These methods will be called by __set() when the corresponding class variable or database field 
// value is set: e.g., calling $this->sender = 'gibbs_margaret@bah.com' calls $this->set_sender('gibbs_margaret@bah.com').
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	* Automatically removes data about messages from the data to be saved, and encodes data as JSON if needed
	* @param string|array|null
	* @return boolean
	*/
	public function set_response($value, $offset = 0){
		if(!is_string($value) && !is_array($value) && !is_null($value)) 
			return $this->error->property_value_should_be_a_string_or_an_array_or_null('request', get_class($this), $value, $offset+1);
		
		if(is_array($value)){
			if(array_key_exists('mail', $value)) unset($value['mail']); //don't include message data in our database for PII/PHI security
			$value = get_instance()->json->encode($value);
		}
		
		return $this->_set_field_value('response', $value, $offset+1);
	}


//////////////////////
// DATA MANAGEMENT
//////////////////////

	/**
	* Extends parent to populate the value for application_id and call_date on creation
	* @return boolean
	*/
	protected function _run_before_create(){ 
		if(!$this->property_is_empty('call')) $this->application_id; //make sure that application id gets populated if we have a value for call
		if(!parent::_run_before_create()) return false;
		if(!isset($this->call_date)) $this->_set_field_value('call_date', time());
		return true;
    }

//////////////////////
// STATIC
//////////////////////	
	
	public static function count_for_reports($id_or_conditions=array()){
		return static::count(static::fix_report_conditions($id_or_conditions));
	}
	
	public static function find_for_reports($id_or_conditions=array(), $key_by = null){				
		static::db()->select('request.*, application.name AS application_name');
		static::db()->join('application', 'request.application_id=application.id', 'left outer');
		return static::find(static::fix_report_conditions($id_or_conditions), $key_by);
	}
	
	//compensate for the truly terrible practices of the developers who created the adhoc reports and cursed us all by saving horrible SQL to the database to be reused forever	
	protected static function fix_report_conditions($id_or_conditions){
		if(!empty($id_or_conditions) && is_string($id_or_conditions) && !is_numeric($id_or_conditions)){			
			$id_or_conditions = trim(str_replace('0=0 AND', '', trim($id_or_conditions)));	
			$id_or_conditions = str_replace('[id]', '[request].[id]', $id_or_conditions);
			$original_id_or_conditions = $id_or_conditions;
			$id_or_conditions = preg_replace('/[^a-zA-Z0-9_\.]id[^a-zA-Z0-9_-]/', 'request.id', $id_or_conditions);
			if(string_begins_with('id', $id_or_conditions)) $id_or_conditions = 'request.'.$id_or_conditions;
		}
		return $id_or_conditions;		
	}	

	
}