<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
*//** */

require_once APPPATH.'controllers/services/mailbox_controller.php';

/**
* API methods to manage purpose of disclosure entries for a message.
*
* Inherits mailbox validation from the mailbox controller.  Mailbox name is a required field for all actions on this controller, with the exception of messages to send or save a messagage; these actions require the sender's email address to be provided.
*
* @author Elan Jaffee <jaffee_elan@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
*
* @todo There's no need for this to be in its own controller when there's only one method.  This would be fine on the message_controller.
*
* @package direct-as-a-service
* @subpackage controllers
*/
class Message_disclosure_controller extends Mailbox_controller{	

	//logs multiple disclosures for a single message
	//disclosures may include patient documents, attachments of any file type, or disclosures for data revealed in the message body
	//expected values: id, disclosures
	public function log_disclosures_for_message_post(){
				
		//set variables for each required field
		$required_fields = array('id', 'disclosures'); 
		foreach($required_fields as $field){
			$$field = element($field, $this->post(), null);
			if(is_null($$field)){
				$this->missing_required_fields[] = $field;
			}
		}
		
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('send');
		$this->respond_with_error_if_mailbox_is_inactive();
		
		
		//make sure that the values are valid
		if(!Message::formatted_like_an_id($id))
			$this->invalid_fields[] = 'id';
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox) || !$message->draft){
			$this->invalid_fields[] = 'id';
		}	
			
		if(!$this->is->array_of_arrays($disclosures) || empty($disclosures))
			$this->invalid_fields[] = 'disclosures';
		$this->respond_with_error_if_fields_are_invalid();	//don't try to process the disclosures further if we don't have avliad values
			
		//check to make sure that we don't have invalid duplicate data in the disclosures that have been passed to the API			
		$message_disclosures = array(); //disclosures that have no attachment - ssn must be unique for message
		$file_disclosures = array(); //disclosures that have an attachment or a file transfer - hash must be unique for message
		foreach($disclosures as $key => $disclosure_values){
			if(array_key_exists('hash', $disclosure_values)){			
				if(array_key_exists($disclosure_values['hash'], $file_disclosures))
					$this->response('Duplicate file hash in disclosures', 422);
				$file_disclosures[$disclosure_values['hash']] = $disclosure_values;
			}else{
				if(array_key_exists($disclosure_values['ssn'], $message_disclosures))
					$this->response('Duplicate SSN in message disclosures', 422);
				$message_disclosures[$disclosure_values['ssn']] = $disclosure_values;			
			}
		} 			
			
			
#TODO - more information for the end user about what's going wrong with disclosures?
		foreach($disclosures as $key => $disclosure_values){
			
			//purpose of disclosure is always required
			if(empty($disclosure_values['purpose'])){
				$this->invalid_fields[] = 'disclosures';
				break;			
			}
			
			//ssn is always required
			if(!array_key_exists('ssn', $disclosure_values) || !$this->is->nonzero_unsigned_integer($disclosure_values['ssn']) || strlen($disclosure_values['ssn']) != 9){
				$this->invalid_fields[] = 'disclosures';
				break;
			}
					
			if(array_key_exists('hash', $disclosure_values) && (empty($disclosure_values['hash']) || strlen($disclosure_values['hash']) != 40 || 
																(empty($message->attachments_matching_hash($disclosure_values['hash'])) ))){
				$this->invalid_fields[] = 'disclosures';
				break;
			}			
			
			$message_foreign_key = Disclosure::related_foreign_key('message');
			
			//before we go further, find/create new disclosure object (but don't save until we've validated all the disclosures)
			if(empty($disclosure_values['hash'])){
				$disclosure = first_element($message->message_disclosures(array('ssn' => $disclosure_values['ssn']))); //if we already have an entry for this ssn, reuse
				if(!Disclosure::is_an_entity($disclosure))
					$disclosure = new Disclosure(array($message_foreign_key => $message->id, 'ssn' => $disclosure_values['ssn']));
			}else{
				$disclosure = first_element($message->attachment_disclosures(array('hash' => $disclosure_values['hash']))); //if we already have an entry for this hash, reuse
				if(!Attachment_disclosure::is_an_entity($disclosure))
					$disclosure = new Attachment_disclosure(array($message_foreign_key => $message->id, 'hash' => $disclosure_values['hash']));			
			}

			$disclosure->purpose = $disclosure_values['purpose'];
			$disclosure->ssn = $disclosure_values['ssn'];
			
			//if this isn't a document that we can parse the given name/family name from, make sure that we have those values
			if(!$disclosure->property_exists('attachment') || !is_a($disclosure->attachment, 'Patient_document_attachment')){
				//given & family names are required when no document is available
				if(empty($disclosure_values['given_name']) || empty($disclosure_values['family_name'])){
#					if(is_on_local()) should_be('nonempty string', element('given_name', $disclosure_values));
#					if(is_on_local()) should_be('nonempty string', element('family_name', $disclosure_values));
					$this->invalid_fields[] = 'disclosures';
					break;	
				}
				
				$disclosure->first = $disclosure_values['given_name'];
				$disclosure->last = $disclosure_values['family_name'];	
			}
			
			$disclosures[$key] = $disclosure;
		}

			
		$this->respond_with_error_if_fields_are_invalid();	
			
		$success = true;
		foreach($disclosures as $disclosure){
			$success = $disclosure->save() && $success;
		}
	
#TODO - COULD BE MORE HELPFUL IN OUR MESSAGE - IDENTIFY WHICH ONES?  OR AT LEAST HOW MANY?		
		if(!$success) 
			return $this->response('Unable to save one or more disclosures.', 400);	

		$this->response('Successfully logged '.count($disclosures).' disclosures.', 200);
	}


	public function log_attachment_disclosure_post(){
		$required_fields = array('id','hash','purpose','ssn'); 
			
		//set variables for each required field
		foreach($required_fields as $field){
			$$field = element($field, $this->post(), null);
			if(is_null($$field)){
				$this->missing_required_fields[] = $field;
			}
		}
		
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('send');
		$this->respond_with_error_if_mailbox_is_inactive();
		
		//check to make sure that id is valid
		if(!Message::formatted_like_an_id($id))
			$this->invalid_fields[] = 'id';
		if(!is_numeric($ssn) || strlen($ssn) != 9){	//match ssn 9 numbers (preg match is more resource-intensive than necessary, just make sure it's a number of the right length)
			$this->invalid_fields[] = 'ssn';
		}
		
		if(empty($purpose)){
			$this->invalid_fields[] = 'purpose';
		}
				
		$this->respond_with_error_if_fields_are_invalid();	
		
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox) || !$message->draft){
			$this->response('Message not found.', 422);
		}
		
		if(!$message->has_attachments()){
			$this->response('Message does not have attachments that require disclosure', 422);
		}
		
		//make sure that the supplied hash matches an attachment on the message
		$attachments_that_match_hash = array();
		foreach($message->attachment_files as $name => $binary_string){
			if(sha1($binary_string) == $hash)
				$attachments_that_match_hash[$name] = $binary_string;
		}
		if(empty($attachments_that_match_hash))
			$this->response('Attachment not found', 422);
		
		//make sure that the attachment actually needs disclosure
		foreach($attachments_that_match_hash as $name => $binary_string){
			if(!Attachment_disclosure::content_requires_disclosure($name, $binary_string))
				$this->response('Attachment does not require disclosure', 422);
		}			
	
		$recipient = ''; //we'll set recipient to empty for drafts - disclosures will be set for all recipients when the message is sent
	
		//we'll switch the purpose of disclosures to being recipient-by-recipient when we send the message - it's too much work before the recipients are finalized
		$disclosure = Attachment_disclosure::update_or_create_for_message($message, compact('hash', 'purpose', 'recipient', 'ssn'));
		if(!Attachment_disclosure::is_an_entity($disclosure)) return $this->response('Unable to save log attachment disclosure', 400);						

		$this->response('Successfully logged disclosure', 200);
	}
}	