<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage libraries
*/

/**
* Extends the CI session library to allow for VLER customizations.
* Additionally, some methods have been extended/overridden to include updates from CI v2.2, and several changes in an attempt to fix the race condition 
* error described here: http://www.hiretheworld.com/blog/tech-blog/codeigniter-session-race-conditions.
* @package direct-project-innovation-initiative
* @subpackage libraries
**/
class DPII_Session extends VLER_Session {
	
	/**
	* Sets the mailbox within the session.
	* This should be used whenever the user is switching mailboxes.  
	* VLER method - does not override any methods in the parent class.
	* @param string - The name of the mailbox, e.g. tauser
	*/	
	function set_mailbox($mailbox){
		$CI = get_instance();	
		$user = User::find_from_session();
		
		if(!User::is_an_entity($user)){
			return $CI->error>warning("I can't set mailbox ".$CI->error->describe($mailbox).' when there is no logged-in user');
		}
		
		if(empty($user->mailbox_names())){
			return $CI->error->warning($user->describe().' does not currently have access to any mailboxes');
		}
		
		//make sure that this user has access to this mailbox
		if(!array_key_exists($mailbox, $user->mailbox_names())){
			$CI->error->warning($user->describe().' does not have access to mailbox '.$CI->error->describe($mailbox).', setting default mailbox '.$CI->error->describe($user->default_mailbox).' instead');
			$mailbox = $user->default_mailbox;
		}		
		
		//setting the mailbox group temporarily, but will use $_SESSION['mailbox'] in the near future
		$_SESSION['mailbox_group'] = $mailbox;
		$_SESSION['mailbox_group_cn'] = $mailbox;
		return $_SESSION['mailbox_group'];
	}	

	/**
	* Changes the system's idea of where we are in the inbox (drafts, sent messages, custom folders, etc.).
	* Should you need to accomplish this and reload the page, you can redirect to inbox/change_mailbox, but if 
	* this method will allow you to change the location in the session without doing a full redirect.
	* VLER method - does not override any methods in the parent class.
	* @param string|int Either the name of a location (inbox, draft, etc) or the numeric id of a custom folder
	*/
	function set_mailbox_location($location){
		//first - a couple actions we'll take care of no matter which location we're going to
		$_SESSION['page_start'] = 1; //bring user back to first page
		if(isset($_SESSION['filter_folder'])){unset($_SESSION['filter_folder']);}//remove archive filter
	

		//handle custom folders first so that we can default to inbox if we can't find the folder
		if(Folder::formatted_like_an_id($location)){
			$folder = Folder::find_one($location);
			
			if(Folder::is_an_entity($folder)){
				$_SESSION['folder'] = $location;
				$_SESSION['folder_name'] = $folder->name;
				return true; //we're done
			}
			
			//if we couldn't find the folder, trigger a warning and default to inbox
			get_instance()->error->warning(get_instance()->error->describe($location).' is not a known folder, defaulting to inbox');
			$location = 'inbox';
		}
				
		//location names mapped to the display name
		$known_locations = array('inbox' => 'Inbox',
								 'sent' => 'Sent',
								 'draft' => 'Drafts',
								 'archived' => 'Archive');
		
		if(!array_key_exists($location, $known_locations)){
			get_instance()->error->warning(get_instance()->error->describe($location).' is not a known folder, defaulting to inbox');
			$location = 'inbox';
		} 
		
		$_SESSION['folder'] = $location;
		$_SESSION['folder_name'] = $known_locations[$location];
		return true;
	}

	/**
	* Returns the current mailbox location, defaulting to inbox if not set.
	* @return string
	*/
	function mailbox_location(){
		if(empty($_SESSION['folder'])) $this->set_mailbox_location('inbox');
		return $_SESSION['folder'];
	}
	
	/**
	* Returns the display name of the current mailbox location, defaulting to Inbox if not set.
	* @return string
	*/
	function mailbox_location_display_name(){
		if(empty($_SESSION['folder_name'])) $this->set_mailbox_location($this->mailbox_location()); //will default to inbox if necessary, otherwise make sure display name for current folder is set
		return $_SESSION['folder_name'];
	}
	
	/**
	* Sets an error message in the flashdata, to be displayed the next time the page loads.
	* VLER method - does not override any methods in the parent class.
	* @param string
	*/	
	public function set_error_message($message){
		$this->set_flashdata('message',json_encode($message));
		$this->set_flashdata('message_class','error');
	}
	
	
	/**
	* Sets a service permission error in the flashdata, to be displayed the next time the page loads.
	* VLER method - does not override any methods in the parent class.
	* @param string 
	* @param string
	*/
	public function set_service_permission_error_message($service, $message){
		$this->session->set_flashdata('service_failure_due_to_permission',true);
		$this->session->set_flashdata('failed_service_name', json_encode($service));
		$this->session->set_flashdata('service_failure_due_to_permission_message',json_encode($message));       
	}
	
	/**
	* Sets a success feedback message in the flashdata, to be displayed the next time the page loads.
	* VLER method - does not override any methods in the parent class.
	* @param string
	*/
	public function set_success_message($message){
		$this->set_flashdata('message', json_encode($message));
		$this->set_flashdata('message_class','success');	
	}
	
	/**
	* Set the content for a banner that will be displayed on the site.
	* @param string An identifier for the banner
	*/
	public function display_banner($identifier, $content){
		if(!validates_as('nonempty_string', $identifier)) return should_be('nonempty string', $identifier);
		if(!validates_as('nonempty_string',$content)) return should_be('nonempty string', $content);
		
		if(!isset($_SESSION['banners'])) 
			$_SESSION['banners'] = array();
		$_SESSION['banners'][$identifier] = $content;
	}
	
	/**
	 * Turn off a banner so that it will no longer be displayed.
	 * @param string The identifier used to indicate this banner.
	 * @return boolean True if the variable existed and has been unset
	 */
	public function dismiss_banner($identifier) {
		if(!validates_as('nonempty_string', $identifier)) return should_be('nonempty string', $identifier);
		if(!array_key_exists('banners', $_SESSION) || !array_key_exists($identifier, $_SESSION['banners'])){
			//Uncomment this when debugging
			//get_instance()->error->notice('There is no banner set named '.get_instance()->error->describe($identifier).', you may want to check your spelling for typos');
			return true; //nothing to do, we're done.
		}
		
		unset($_SESSION['banners'][$identifier]);
		return true;
	}
	
	public function markup_for_banner($identifier, $attributes=array()){
		if(!validates_as('nonempty_string', $identifier)) return should_be('nonempty string', $identifier);
		if(!is_array($attributes)) return should_be('array', $attributes);
		
		if(empty($_SESSION['banners']) || empty($_SESSION['banners'][$identifier])) return; //not an error, but nothing for us to do
		
		$alert_types = array('announcement' => 'warning',
							 'automatic_replies' => 'danger');
							 
		$attributes = merge_attributes($attributes, array('class' => 'alert-banner', 'data-banner-name' => $identifier));
		
		return dismissable_alert($_SESSION['banners'][$identifier], element($identifier, $alert_types, 'info'), $attributes);
	}
		
} 