<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/ /** */

require_once APPPATH.'controllers/inbox.php';

/**
* @package direct-project-innovation-initiative
* @subpackage controllers	
*/
class Message_controller extends Inbox {

	/**
	* Autosaves drafts.
	*
	* Drafts are populated via POST in the compose action; files to attach will be in the attachment cache.
	* After saving the draft, this method will populate the draft id in the session so that the 
	* save/send actions know to edit/send the existing draft instead of creating a new one.
	*
	* If the message does not have any values yet, autosave should NOT save the empty draft - we don't
	* want to overwhelm the drafts folder with empty drafts.
	*/
    public function ajax_draft_save() {	
		$this->action_is_not_user_activity(); 
		
		//assuming the form submit went through the CSRF protection in order to get to this point,
		//give the new CSRF token as JSON output so the javascript can update it on the page
		$response = array('token'=>$this->security->get_csrf_hash(), 'total_size_of_attachments' => $this->user->attachment_cache_size_in_bytes());
		
		$message = $this->_message_from_post();
		$force_save = (element('force_save', $_POST) == 'true'); //by default, we won't save if nothing's been added to the message, but force save overrides this behavior
				
		//check to see if the message has been started yet - we don't want to autosave drafts that have no values
		$message_is_empty = (!isset($message->id) && empty($response['total_size_of_attachments']) && $message->importance == 'normal');
		foreach( array('subject', 'to', 'cc', 'bcc', 'body') as $field){
			if(!$message_is_empty) break; //something is populated, we can stop checking now
			$message_is_empty = ($message_is_empty && $message->property_is_empty($field));
		}
		
		if($message_is_empty && !$force_save){  //since message hasn't been populated yet, return true without any further action so the drafts folder isn't overwhelmed with empty drafts
			$response['success'] = true;
			echo $this->json->encode($response);
			return true;
		}		
		
		if($response['total_size_of_attachments'] > MAX_ATTACHMENTS_SIZE_IN_BYTES){
			$response['error'] = 'The attachments for this message exceed the current size limit of '.byte_format(DIRECT_ATTACHMENT_SIZE_LIMIT).'.  Please remove one or more attachments.';
			$response['success'] = false;
			echo $this->json->encode($response);
			return false;			
		}
		
		//make sure we have a message
		if(!Message::is_an_entity($message)){
			$response['error'] = 'Failed to find or create message object from POST';
			$response['success'] = false;
			echo $this->json->encode($response);
			return false;
		}
	
		//save the message
		if(!$message->save( $preserve_existing_attachments = FALSE )){
			$response['error'] = 'Failed to save '.$message->describe().' from POST';
			$response['success'] = false;
			echo $this->json->encode($response);
			return false;
		}
		
		//saving clears the cache, so make sure that the existing attachments are added to the cache
		$message->add_existing_attachments_to_cache();
		
		//if this is a reply/reply all/forward, set the flag on the parent message
		//we need to do this on autosave - otherwise, if a person exits out without saving and then re-opens the draft, we won't know what the parent id was and the parent message won't get flagged
		if(!empty($this->input->post('flag_value')) && Message::formatted_like_an_id($this->input->post('parent_message_id'))){
			$this->response_model->set_response_flag($message->id(),$this->input->post('parent_message_id',TRUE), $this->input->post('flag_value',TRUE));
		}
		
		$response['success'] = true;
		$response['id'] = $message->id;
		echo $this->json->encode($response);
	}
	
  /* This function prepares and loads the compose mail view. If given the optional inputs, it will
     * expect "reply" for replies and "forward" for forward, followed by the message id of the message to be
     * replied to or forwarded. It will use this to populate a quote of the old message, and in the case of
     * replies, populate the to field of the new lmessage from the from field of referenced message.
     */
/* Currently, this form submits to the inbox/form_check method, which calls on compose() again to validate the 
   form values and redisplay them if the message values don't validate.  However, since we're still using form_check,
   the URL if we're redisplaying the form with errors will be inbox/form_check.  Ultimately, it would be good to get this 
   form to submit to compose directly so that we don't have this problem.  */ 
/* Known issues as of 4-23-14:
	- Attachments are determined based on a user cache that is *not* specific to the message, so trying to have two compose windows
	  open at once for different messages will cause unexpected behavior for attachment saving. 
	- Attaching a file with the same name as an existing attachment will simply override the original file.  We need to be able to attach
	  multiple files with the same name, but note that existing attachment methods (view, download, etc.) use filename as the
	  unique identifier to get the attachment from the message.  It would probably be easiest to simply rename duplicate file names before
	  we add them in attachment_controller::attach(); e.g., "Thundervole The Mighty.jpg" becomes "Thundervole the Mighty (1).jpg", etc.
	- The javascript for adding/removing attachments should probably autosave the draft after adding/removing the attachments */
    public function compose($type = 'draft', $id = NULL) {
		if(!is_null($id) && !Message::formatted_like_an_id($id)) show_404();
		if(!in_array($type, array('draft', 'forward', 'reply', 'replyall'))) show_404();

		$this->session->set_mailbox_location('draft'); //set this so that we know which folder we're in if they use the search button

		$data = array(); //values that we're passing on to the view
		$parent_message = null;

		//do we have POST values?  if so, parse them & validate; we'll either save or re-display
		if(!empty($_POST) && (array_key_exists('save', $_POST) || array_key_exists('send', $_POST))){
			$message = $this->_message_from_post();
			$action = (array_key_exists('send', $_POST)) ? 'send' : 'save';
			
			//check to make sure we were able to find the message - if not, something has gone very wrong
			if(!Message::is_an_entity($message)){ 
				$this->session->set_error_message('An unexpected error occurred, and we were unable to '.$action.' your message.  Please try again, and contact an administrator if the problem persists.');
				return redirect('inbox/compose');
			}
			
			//validate that the message is saveable - otherwise, don't go to the overhead of accessing the API			
			$validation_errors = array();
			if($this->user->attachment_cache_size_in_bytes() >  MAX_ATTACHMENTS_SIZE_IN_BYTES){
				$validation_errors[] = 'The attachments for this message exceed the limit of '.byte_format(MAX_ATTACHMENTS_SIZE_IN_BYTES).'. Please remove one or more attachments.';
			}
			
			foreach($this->user->attachments_from_cache() as $filename => $attachment){
				if(!Message::attachment_has_valid_extension($filename))
					$validation_errors[] = $filename.' is not a valid file type.';
			}
			
			if(!empty($validation_errors)){ 	
				$data['feedback_message'] = implode_nonempty(' ', $validation_errors);	
				$data['feedback_class'] = 'error';	
			}else{		
				if(!$message->save( $preserve_existing_attachments = FALSE )){
					
					//message shouldn't say the message failed to save/send, because attachment errors still save message updates
					$data['feedback_message'] = 'An error was encountered. '.$this->api->message(); 
					$data['feedback_class'] = 'error';
				}else{
					//if this is a reply/reply all/forward, set the flag on the parent message
					if(!empty($this->input->post('flag_value',TRUE)) && Message::formatted_like_an_id($this->input->post('parent_message_id',TRUE))){
						$this->response_model->set_response_flag($message->id(),$this->input->post('parent_message_id',TRUE), $this->input->post('flag_value',TRUE));
					}

					//if we're not sending the draft, we're done and can redirect to a saved copy of the draft. 
#TODO - THIS WILL WIPE THE 7332 DATA SELECTION  FOR PATIENT DOCUMENTS.  NOT IDEAL.  					
					if(!array_key_exists('send', $_POST)){				
						$this->session->set_success_message('Message successfully saved.');
						return redirect('inbox/compose/draft/'.$message->id);	
					}
																			
					if(!array_key_exists('7332_data', $_POST)){
						$data['feedback_message'] = 'Please indicate whether or not this message contains 38 U.S. Code §7332 protected data.'; 
						$data['feedback_class'] = 'error';
					}elseif(!empty($this->input->post('7332_data')) && empty($this->input->post('7332_auth'))){
						$data['feedback_message'] = 'You do not have authorization to send this information.'; 
						$data['feedback_class'] = 'error';						
					}else{		
						$data['modal'] = $this->purpose_of_disclosure($message);
					}			
				}
			}
			
			//if we're redisplaying because of an error, set up the post data that's not on the message object			
			$data['flag_value'] = $this->input->post('flag_value', TRUE);
			$data['parent_message_id'] = $this->input->post('parent_message_id',TRUE);						
		}
		
		
		if(!isset($message)){ //if we're not displaying a form with errors, get the message data from parameters
			if(is_null($id)){ //if there's no id, then we're starting a new message from scratch
				$message = new Message(array('sender' => $this->mailbox->email_address));
				$to = $this->session->flashdata('mail'); //check to see if we had a direct mail link
				if(!empty($to) && $this->is->string_like_an_email_address($to))
					$message->to = $to;
			}elseif($type == 'draft'){ //edit an existing draft
				$message = Message::find_one(array('id' => $id, 'folder' => 'draft', 'mailbox' => $this->mailbox->name));
				if(!Message::is_an_entity($message)) show_404();  
				if(!$message->draft) redirect('inbox/viewmsg/'.$message->id);
			}else{ //create a new draft based on a parent (the message we're forwarding or sending to)
				$message = new Message(array('sender' => $this->mailbox->email_address));
				$parent_message = Message::find_one(array('id' => $id, 'mailbox' => $this->mailbox->name));
				if(!Message::is_an_entity($parent_message) || $parent_message->draft) redirect('inbox/compose');  //couldn't find the parent id, so we'll just redirect the user to a normal compose screen
				$data['parent_message_id'] = $parent_message->id;
			}	
		
			//set up the attachments - unless we're redisplaying from post, in which case the attachment cache is set up correctly and we shouldn't mess with it
			$this->clear_attachments(); //clear attachments when visiting compose for the first time in case any were not removed from previous messages
			if($message->has_attachments()) 
				$message->add_existing_attachments_to_cache();
			if($type == 'forward' && Message::is_an_entity($parent_message) && $parent_message->has_attachments()) 
				$parent_message->add_existing_attachments_to_cache(); //replies don't need to preserve attachments, but forwards do
		}
		
		//for purpose of disclosure form, we may sometimes have a parent id that hasn't been turned into a message yet (seems like we should be able to do this a little cleaner than this)
		if(!empty($_POST['parent_message_id']) && !Message::is_an_entity($parent_message)){
			$parent_message = Message::find_one(array('id' => $this->input->post('parent_message_id'), 'mailbox' => $this->mailbox->name));
		}		
			
		//set title and connection status
		$data['title'] = PORTAL_TITLE_PREFIX . 'Compose Mail';
		$mailbox_list_data = $this->mailformat->mailbox_list();
		$data['mailboxes'] = $mailbox_list_data['folder_list'];
		$data['message'] = $message;
		$data['type'] = $type;
		
		$data['subject'] = $message->subject;
		$data['to'] = $message->addresses_for_display_for_send('to');
		$data['cc'] =  $message->addresses_for_display_for_send('cc');
		
		//FIND THE ATTACHMENTS
		//We want to get this from the cache, not the message, since attachments may have changed since the last message save.
		//On the other hand, attachments from the message have more metadata (e.g. message_id), so if the same attachment is currently on the message, we'll make sure use the attachment
		//object created from the message.		
		$data['attachments'] = $this->user->attachments_from_cache();
		if(Message::formatted_like_an_id($message->id)){
			foreach($data['attachments'] as $name => $attachment){
				if($message->has_attachment($name, $attachment->binary_string)){
					$data['attachments'][$name] = $message->attachment($name); 
				}
			}
		}
		
				
		$data['quote_body'] = $message->body;		
		
		//for all forwards and replies, we'll want to use the same title as the original message and we'll need to set a flag on the original message after sending
		if(Message::is_an_entity($parent_message)){
			$data['flag_value'] = $type;
			$data['subject'] = $this->mailformat->msg_subject_sanitize($parent_message->subject, $type);
			$data['quote_body'] = "<br />------------------------------------------------------------------<br />\r\n";
			$data['quote_body'] .= "From: " . implode(", ", $parent_message->addresses_for_display('sender')) . "<br />\r\n";
			$data['quote_body'] .= "To: " . implode(", ", $parent_message->addresses_for_display('to')) . "<br />\r\n";
			if(!$parent_message->property_is_empty('cc')) {  $data['quote_body'] .= "CC: " . implode(", ", $parent_message->addresses_for_display('cc')) . "<br />\r\n"; }
			$data['quote_body'] .= "Date: " . date('n/j/Y g:i A',$parent_message->timestamp) . "<br />\r\n";
			$data['quote_body'] .= "Subject: " . $parent_message->subject_for_display . "<br /><br />\r\n\r\n";
			$data['quote_body'] .= $parent_message->body;			
		}
		
		//for replies, make sure that we're replying to the sender
		if(Message::is_an_entity($parent_message) && ($type === 'reply')) {
			$data['to'] = $parent_message->addresses_for_display_for_send('sender');
		}
		
		//for forwards make sure to include the forwarded message priority
		if($type === 'forward') {
			$importance = 'normal';
			if($parent_message->priority <= 2) { $importance = 'high'; }
			else if($parent_message->priority === 3) { $importance = 'normal'; }
			else if($parent_message->priority > 3) { $importance = 'low'; }
			else { $importance = 'normal'; }
			
			$data['message']->importance = $importance;
		}
		
		//for reply-alls, make sure everyone except the current mailbox is included on the cc
		if($type === 'replyall') {
			$to = array_unique(array_filter(explode(',', $parent_message->to),'strlen'));
			$to [] = $parent_message->sender;
			while(in_array($this->mailbox->email_address, $to)) unset($to[array_search($this->mailbox->email_address, $to)]);
			$data['to'] = implode(';', $to);
			$cc = array_unique(array_filter(explode(',', $parent_message->cc),'strlen'));
			if(in_array($parent_message->sender, $cc)) unset($cc[array_search($parent_message->sender, $cc)]);
			if(in_array($this->mailbox->email_address, $cc)) unset($cc[array_search($this->mailbox->email_address, $cc)]);
			$data['cc'] = implode(';', $cc);
			if(!empty($data['cc'])) $data['cc'] .= ';';
		}
	
		//pass data to the view and load it
		$this->template->set($data);
		$this->template->set('page_title', 'Compose Message');
		$this->template->set('menu_partial', 'inbox/compose/_menu');
		$this->template->load('inbox/template', 'inbox/compose', $data);		  

    }			
	
	
    /* This function loads a view to display a single message with the given id.
     */
    public function viewmsg($id) {
		$data['connected'] = TRUE;
		if(!Message::formatted_like_an_id($id)) { show_404(); }
		
		//get the message with the provided id
		$message = Message::find_one(array('folder' => $this->session->mailbox_location(), 'id' => $id, 'mark' => 'read'));	
		$this->show_error_if_not_a_message($message);
		if($message->draft) redirect('inbox/compose/draft/' . $message->id);
		
		$data['title'] =  PORTAL_TITLE_PREFIX . $message->subject_for_display;
		$data['message'] = $message;
		$data['mailbox'] = $this->mailbox;
		$data['user'] = $this->user;
		
		//refresh mailbox list
		$mailbox_list_data = $this->mailformat->mailbox_list();
		$this->template->set('mailboxes', $mailbox_list_data['folder_list']);

		//set message status
		$data['status_enabled'] = false;
		if($this->mailbox->is_group()) {
			$data['status_enabled'] = TRUE;
			
			$group_members = $this->get_group_members($this->mailbox->name);
			$data['group_members'] = array_combine(collect('id', $group_members), collect('name', $group_members));
			
			//get workflow data
			$this->load->model('workflow_model');
			if(isset($this->workflow_model)){
				$workflow_item = $this->workflow_model->find_for_message($message);
				if(!empty($workflow_item)){
					$data['workflow_item'] = $workflow_item;
				}
			}
						
		}
	
		$this->template->set($data);
		$this->template->set('page_title', 'Message');
		$this->template->set('form_destination', 'inbox/archive/'.$id);
		$this->template->set('menu_partial', 'inbox/_view_menu');
		$this->template->load('inbox/template', 'inbox/view', $data);
    }	
	
/////////////////////
// PROTECTED METHODS
/////////////////////	
	

	/**
	* Displays a form that prompts the user to provide information about why patient documents are attached to a message.
	* This message will be displayed before sending for all messages that have patient documents attached, though it may sometimes be pre-populated 
	* with data that was entered during the attachment process.  Basically, we want the user to have to review and sign off on the disclosures before actually sending.
	*
	* This method is not intended to be web-accessible and is currently called only after the message is saved in compose().  All compose data will be preserved as hidden fields
	* in the form so that we can call on send() once the disclosurse have been successfullys aved.
	*
	* @param Message $message
	*/
	protected function purpose_of_disclosure(Message $message){		
		if(!$message->draft) return $this->error->should_be_a_draft($message);
		if($message->has_attachments()) $message->add_existing_attachments_to_cache(); //rawrgh.  this is sometimes but not always necessary.  but it won't hurt.  -- MG 2014-10-09/
				
		require_library('disclosure_form_markup_generator');
		$markup_generator = new Disclosure_form_markup_generator($message); //setting the message will automatically set up the form fields
		$feedback_messages = array();
		
		
		//if we have purpose of disclosure data, validate the existing values
		if(array_key_exists('disclosures', $_POST)){
			$markup_generator->set_values_from_post(); //note that this will overwrite the values we previously set for the hidden fields	
						
			if(!$markup_generator->validates()){
				$feedback_messages['danger'] = '<strong>We need you to make a few changes to this form.</strong> '.ul_if_multiple($markup_generator->validation_messages);
			}else{
				
				$disclosures = $markup_generator->disclosures();		
				
				if(!empty($disclosures) && !$message->log_disclosures($disclosures)){
					$feedback_messages['danger'] = '<strong>We were unable to save your disclosures for this message.</strong> Please try again in a moment, and contact an administrator if the problem persists.';
				}else
					return $this->send($message); 
			}
		}else{
			
			//if the form hasn't been filled out yet, populate any existing values that were saved for the attached patient documents
			foreach($message->attached_patient_documents() as $filename => $attachment){
				if(!$attachment->property_is_empty('ssn'))
					$markup_generator->field('disclosures[patient_documents]['.$attachment->html_friendly_name.'][ssn]')->value = $attachment->ssn;
				if(!$attachment->property_is_empty('purpose'))
					$markup_generator->field('disclosures[patient_documents]['.$attachment->html_friendly_name.'][purpose_of_disclosure]')->value = $attachment->purpose;
			}
		}
			
		return $this->load->view('inbox/patient_data/_purpose_of_disclosure_form', compact('markup_generator', 'message', 'feedback_messages'), TRUE);	
	}

	
	protected function send(Message $message){
		if(!Message::is_an_entity($message) || !$message->draft) return $this->error->should_be_a_draft($message);
				
		//send the message!
		if(!$message->send()){
			//the API's message isn't so user-friendly, so use our own message
			$this->session->set_error_message('An error occurred and we were unable to send your message. It has been saved as a draft.'); 
			if($this->api->http_status == '403' && string_contains('application is not authorized', $this->message())){
				$this->session->set_service_permission_error_message('Send', 'Failed to send message because Send Direct Service is disabled.');
			}
			return redirect('inbox/compose/draft/'.$message->id); //since save was already successful, we can redirect to the saved draft
		}
		
		$this->response_model->check_response($message->id() , $this->mailbox->name);
		$this->session->set_success_message('Message successfully sent.');
		$this->session->set_mailbox_location('inbox'); //set the folder so that we redirect to the main inbox, not to the drafts page
		return redirect('inbox');	
	}		
	
}
/* End of file inbox.php */
/* Location: ./application/controllers/inbox.php */