<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/ /** */

require_once APPPATH.'controllers/restricted_controller.php';

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class Inbox extends Restricted_controller {
	var $mailbox; //the current mailbox - either the user's personal mailbox or a group mailbox
	
    function __construct() {
        parent::__construct();
		
		require_library('attachment');
		
		$this->load->model('response_model');
		
		//get group mailbox information
		$group_mailboxes = $this->session->userdata('group_mailboxes');
		
		//currently selected mailbox - personal or group
		$mailbox_group = $this->session->userdata('mailbox_group');
		
		//default to the user's personal mailbox if we don't have a mailbox selected or if no group mailboxes are available or if the mailbox group isn't valid
		if(empty($mailbox_group) || empty($group_mailboxes) || !array_key_exists($mailbox_group, $group_mailboxes) || !Mailbox::exists( array('user_name' => $mailbox_group) )){
			$mailbox_group = $this->session->userdata('username');
		}

		$this->mailbox = Mailbox::find_one( array('user_name' => $mailbox_group) );
		
#TODO - special case for if we still don't have a mailbox? or if mailbox isn't active?

		//set the session mailbox group in case we've changed it
		if($this->session->userdata('mailbox_group') != $mailbox_group)
			$this->session->set_userdata('mailbox_group', $mailbox_group);	

    }

    /* This function changes the current mailbox by setting the current mailbox in session
     * to the provided input. It is called from links, so it expects a url encoded base64 string,
     * which it will then decode. It redirects back to the index when done.
     */
    public function change_mailbox($folder_id) {
		$this->session->set_mailbox_location($folder_id); //set current mailbox
		redirect('inbox'); //redirect to index
    }
	
	/* This function allows the user to change their currently selected mailbox group */
	public function change_mailbox_group($group) {
		$group = rawurldecode($group);
		$this->session->set_mailbox($group);
		$this->session->set_mailbox_location('inbox');
		redirect('inbox');
	}	


	public function close_authorize() {
		$this->output->append_output('<script type="text/javascript">parent.$.fancybox.close();</script>');
	}

	
   /* This function checks to see which type of form was submitted so that forms with multiple buttons can be handled  
    * differently
    */
	//DEPRECATED - DO NOT ADD MORE THINGS TO THIS METHOD UNLESS ABSOLUTELY NECESSARY!  Forms should have their own destination, not just be added to this action
    public function form_check() {
        $post_array = $this->input->post(NULL,TRUE);

        if(!empty($post_array)) {
            $send = array_key_exists('send',$this->input->post(NULL,TRUE)); //detect send submission
            $save = array_key_exists('save',$this->input->post(NULL,NULL,TRUE)); //detect save submission
            $archive = array_key_exists('archive',$this->input->post(NULL,TRUE)) || array_key_exists('archive_hidden',$this->input->post(NULL,TRUE)) ; //detect archive submission
            $compose = array_key_exists('compose',$this->input->post(NULL,TRUE)); //detect compose submission
			$mark_as_read = array_key_exists('mark_as_read',$this->input->post(NULL,TRUE)) || array_key_exists('mark_as_read_hidden',$this->input->post(NULL,TRUE)) ; //detect mark as read submission
            $mark_as_unread = array_key_exists('mark_as_unread',$this->input->post(NULL,TRUE)) || array_key_exists('mark_as_unread_hidden',$this->input->post(NULL,TRUE)) ; ; //detect mark as unread submission

			//detect move submission (either from hidden move button or regular one)
            if(array_key_exists('move',$this->input->post(NULL,TRUE))) {
				$move = TRUE; $move_action = 'move';
			}
			else if(array_key_exists('move_hidden',$this->input->post(NULL,TRUE))) {
				$move = TRUE; $move_action = 'move_hidden'; $this->session->set_flashdata('last_button','move_hidden');
			} else { $move = FALSE; }
			
            //detect rename folder submission (submission only happens via button on hidden menu)
			foreach($post_array as $key => $input) {
				if(string_begins_with('rename_folder_', $key)) {
					$folder_id = $this->input->post('old_value_'.strip_from_beginning('rename_folder_', $key), TRUE);
				}
			}
			
			//based on detections, call appropriate functions
            if($send || $save){$this->compose();/*moving this functionality to the compose method so we can redisplay values from post if there are errors; ultimately, would like to bypass form_check and just use compose*/}
            else if($compose) { redirect('inbox/compose'); }
            else if($move) { $this->move($move_action); }
            else if($mark_as_read) { 
				if(array_key_exists('mark_as_read_hidden',$this->input->post(NULL,TRUE))) { $this->session->set_flashdata('last_button','mark_as_read_hidden'); }
				$this->mark_as_read(); 
			}
			else if($mark_as_unread) { 
				if(array_key_exists('mark_as_unread_hidden',$this->input->post(NULL,TRUE))) { $this->session->set_flashdata('last_button','mark_as_unread_hidden'); }
				$this->mark_as_unread(); 
			}
			else if(isset($folder_id)) { $this->rename_folder($folder_id); redirect('inbox'); }
			
			//for archive perform some additional actions
            else if($archive) {
				if(array_key_exists('archive_hidden',$this->input->post(NULL,TRUE)))
					 $this->session->set_flashdata('last_button','archive_hidden'); 
					 
				if(empty($_POST['selected_messages']) || !is_array($_POST['selected_messages']))
					show_404();
					
				return $this->archive(array_keys($_POST['selected_messages']));
			}
            else { show_404(); }
        }
        else { 
            //catch upload size > post_max_size error
            if(empty($_FILES) && empty($_POST) && isset($_SERVER['REQUEST_METHOD']) && strtolower($_SERVER['REQUEST_METHOD']) == 'post'){
				$this->session->set_error_message('Failed to send message, attachment size was greater than 10 MB. Failed to save draft since message size was greater than server request limit.');
                redirect('inbox'); 
            }
            else {
                $this->session->set_error_message('Form submission failed');
                redirect('inbox');
            }
        }
    }
	
	/** 
	* This searches the address book and contacts list and returns JSON encoded results for the auto-complete 
	* @todo DB queries belong in models, not controllers.
	*/
	public function get_contacts_search($input = null, $function = false, $contain_all = null){  
		$input = urldecode($input);
		if(is_null($input)) { $input = ""; }
		$result_arr = array();
		$added_addresses = array();
		//get addresses from global address book in LDAP
		if(is_null($contain_all)) { $contain_all = true; }
		$entries = ($this->ldap->search($input,NULL,NULL,NULL,NULL,$contain_all));
		foreach($entries as $key => $val) {
			if($this->valid_formatted_address($val['mail'])) {
				$contact_arr = array('name'=>(isset($val['displayname'])?$val['displayname']:"Undefined"),'id'=>$val['mail']);
				array_push($result_arr,$contact_arr);
				array_push($added_addresses,$contact_arr['id']);
			}
		}
		if(!$function){
			//get global distribution lists
			$list_result = $this->public_distribution_list_model->find(array('cn'=>$input.'*'));
			
			foreach($list_result as $index => $entry) {
				$address_tokens = array();
				$list_result[$index]['id'] = $this->public_distribution_list_model->alias($entry['id']);
				$list_result[$index]['description'] = '(Global) '.$list_result[$index]['description'];
				
				if(!empty($list_result[$index]['addresses'])){
					//get display names and addresses for distribution list expansion
					$address_array = preg_split("/;/", $list_result[$index]['addresses'], null, PREG_SPLIT_NO_EMPTY);
					$display_names_array = $this->distribution_list_model->display_names_for_direct_addresses($address_array);
					foreach($address_array as $address) {
						if(!isset($display_names_array[$address])) {
							$display_names_array[$address] = trim($address);
						}
					}
					foreach($display_names_array as $address => $display_name) {
						array_push($address_tokens,$display_name . ' (' . trim($address) . ')');
					}
					
					$display_names = implode('; ', $display_names_array);
					$addresses_in_display_order = implode('; ', array_keys($display_names_array));
					$list_result[$index]['display_names'] = $display_names;
					$list_result[$index]['addresses_in_display_order'] = $addresses_in_display_order;
					$list_result[$index]['address_tokens'] = json_encode($address_tokens);
					array_push($result_arr,$list_result[$index]);
				}
			}
		}
		
		//get contacts from contact list in database
        $contacts = $this->db->query('SELECT * FROM contacts WHERE (last_name LIKE (' . $this->db->escape($input) . "+ '%') OR first_name LIKE (" . $this->db->escape($input) . "+ '%') OR middle_name LIKE (" . $this->db->escape($input) . "+ '%') OR mail LIKE (" . $this->db->escape($input) . "+ '%')) AND user_id = (SELECT user_id FROM users WHERE  user_deleted_flag=0 AND user_name = " . $this->db->escape($this->session->userdata("username")) . ")"); 
		for($i = 0; $i < $contacts->num_rows(); $i++) {
			$row = $contacts->row_array($i);
			$displayname = $row['last_name'] . ', ' . $row['first_name'];
			if(isset($row['middle_name']) && (strlen($row['middle_name']) > 0)) { $displayname .= ' ' . $row['middle_name']; }
			
			if($this->valid_formatted_address($row['mail'])){
				$contact_arr = array('name' => $displayname, 'id' => $row['mail']);
				//don't return blank/null attributes
				foreach($contact_arr as $attr => $val) {
					if(!isset($val) || strlen($val) <= 0) {
						unset($contact_arr[$attr]);
					}
				}
				if($function){
					$wasadded = array_search($contact_arr['id'],$added_addresses);
					if (!($wasadded === 0 || $wasadded > 0 )){
						array_push($result_arr,$contact_arr);
						array_push($added_addresses,$contact_arr['id']);
					}
				}
				else{
					array_push($result_arr,$contact_arr);
				}
			}
		}
		//get any contacts from the admin panel contact list that are shared with this user
		if ($this->user->is_admin) {
			$admin_contacts = $this->db->query("SELECT * FROM admin_contact_list WHERE (last_name LIKE (" . $this->db->escape($input) . "+ '%') OR middle_name LIKE (" . $this->db->escape($input) . "+ '%') OR first_name LIKE (" . $this->db->escape($input) . "+ '%') OR direct_address LIKE (" . $this->db->escape($input) . "+ '%'))");
		}
		else{
			$admin_contacts = $this->db->query("SELECT * FROM admin_contact_list WHERE (last_name LIKE (" . $this->db->escape($input) . "+ '%') OR middle_name LIKE (" . $this->db->escape($input) . "+ '%') OR first_name LIKE (" . $this->db->escape($input) . "+ '%') OR direct_address LIKE (" . $this->db->escape($input) . "+ '%')) AND sharing LIKE ('%" .$this->session->userdata("username"). "%')");
		}
		for($i = 0; $i < $admin_contacts->num_rows(); $i++) {
			$row = $admin_contacts->row_array($i);
			$displayname = $row['last_name'] . ', ' . $row['first_name'];
			if(isset($row['middle_name']) && (strlen($row['middle_name']) > 0)) { $displayname .= ' ' . $row['middle_name']; }
			$admin_contact_arr = array('name' => $displayname, 'id' => $row['direct_address']);
			//don't return blank/null attributes
			foreach($admin_contact_arr as $attr => $val) {
				if(!isset($val) || strlen($val) <= 0) {
					unset($admin_contact_arr[$attr]);
				}
			}
			array_push($result_arr,$admin_contact_arr);
		}
		
		if(!$function){
			//get private distribution lists and load into results array
			$this->db->order_by('name');
			$this->db->like('name',$input,'after');
			$dist_list_result = $this->private_distribution_list_model->find(array());
			for($i = 0; $i < count($dist_list_result); $i++) {
				$address_tokens = array();
				$row = $dist_list_result[$i];
				$list_arr = array('name'=>$row['name'],'id'=>$this->private_distribution_list_model->alias($row['id']),'description'=>'(Private) '.$row['description']);
				//don't return blank/null attributes
				foreach($list_arr as $attr => $val) {
					if(!isset($val) || strlen($val) <= 0) {
						unset($list_arr[$attr]);
					}
				}
				if(!empty($row['addresses'])){
					//get display names and addresses for distribution list expansion
					$address_array = preg_split("/;/", $row['addresses'], null, PREG_SPLIT_NO_EMPTY);
					$display_names_array = $this->distribution_list_model->display_names_for_direct_addresses($address_array);
					foreach($address_array as $address) {
						if(!isset($display_names_array[$address])) {
							$display_names_array[$address] = trim($address);
						}
					}
					foreach($display_names_array as $address => $display_name) {
						array_push($address_tokens,$display_name . ' (' . trim($address) . ')');
					}
					$display_names = implode('; ', $display_names_array);
					$addresses_in_display_order = implode('; ', array_keys($display_names_array));
					$list_arr['display_names'] = $display_names;
					$list_arr['addresses_in_display_order'] = $addresses_in_display_order;
					$list_arr['address_tokens'] = json_encode($address_tokens);
					array_push($result_arr,$list_arr);
				}
			}
		}
		//sort combined result array (natural order, case insensitive)
		usort($result_arr, function( $el1, $el2) { return strnatcasecmp( $el1['name'], $el2['name']); });
		
		//add whatever the user is currently typing so that it is allowed as well
		if($this->valid_formatted_address($input)) {
			$contact_arr = array('name' => $input, 'id' => $input);
			array_push($result_arr,$contact_arr);
		}
		array_walk_recursive($result_arr, function (&$value) {
			$value = htmlentities($value);
		});
		//only echo if the request comes from AJAX
		if($contain_all){
			$result_arr = array_slice($result_arr, 0,10);
		}

		if(IS_AJAX && !$function) {
			echo json_encode($result_arr);
		}
		return $result_arr;
	}	

    /* This is the main function for the inbox. It will get the header data for all messages in the current mailbox
     * which can then be displayed in a table in the inbox/index view.
     */
    public function index() {    		
		$this->load->model('flag_model');
		if($this->mailbox->is_group) { $this->load->model('workflow_model'); }
	
		$current_user = $this->user->id;
		$display_name = $this->user->cn;
		$folder = strtolower(element('folder', $_SESSION, 'inbox'));
		$folder_name = strip_from_beginning(CUSTOM_MAILBOX_PREFIX, element('folder_name', $_SESSION, 'Inbox'));
		$show_workflow = ($this->mailbox->is_group && isset($this->workflow_model) && !in_array($folder, array('archived', 'draft', 'sent')));
		$title = PORTAL_TITLE_PREFIX.$folder_name;

		//SET UP TEMPLATE VARIABLES	
		$mailbox_list_data = $this->mailformat->mailbox_list();
		$this->template->set('mailboxes', $mailbox_list_data['folder_list']);
		$this->template->set('page_title', 'Message List');
		$this->template->set('form_destination', 'inbox/form_check');
		$this->template->set('menu_partial', 'inbox/_inbox_menu');
		
		if($this->mailbox->is_group){
			$this->template->set('other_members', $this->get_group_members($this->mailbox->name));
		}
		
		//check to see what message we should start with
		$page_start = element('page_start', $_SESSION, 1);
		if(!$this->is->nonzero_unsigned_integer($page_start)){
			$page_start = 1;
			$_SESSION['page_start'] = $page_start;
		}
							
		//FIND THE MESSAGES	
		$limit = $display_per_page = INBOX_DISPLAY_PER_PAGE; //configure how many emails to display per page 
		$start = $page_start - 1;
		$order_by = $sort_column = element('sort_column', $_SESSION);
		$order = $sort_direction = element('sort_direction', $_SESSION);
		
		$criteria = array_filter(compact('folder', 'limit', 'order', 'order_by', 'start'), 'strlen');
		$criteria['part'] = 'headers'; 
		
		if($order_by == 'flag' || $order_by == 'workflow'){
			unset($criteria['order_by'], $criteria['order']);  //the API can't sort these - we'll sort them lwhen we get the messages back
		}
		$filter_folder = element('filter_folder', $_SESSION);
		if($folder === 'archived' && in_array($filter_folder, array('inbox','sent','draft'))){
			$filter = array('original_folder' => $filter_folder);
			$criteria['filter'] = rawurlencode(base64_encode(json_encode($filter)));
		}		
		$messages = $this->mailbox->messages( $criteria  );

		//if we have any problems retrieving the messages, show the error right away before calculating all the other things		
		if($this->api->http_status == 403 && $this->api->message() == 'Access denied. The application is not authorized for this action.'){
			return $this->template->load('inbox/template', 'inbox/_service_permission_error');
		}elseif($this->api->http_status != 200 || !empty($this->api->format_errors)){
			return $this->template->load_string('inbox/template', 'Unable to retrieve messages at this time. If the problem persists, please contact a system administrator.');
		}
		
		//if there are no messages in this location, tell us that.
		if(empty($messages)){
			if($filter_folder === 'draft') { $filter_folder = 'drafts'; } //pluralize drafts
			$data = compact('current_user', 'display_name', 'folder_name', 'show_workflow', 'title', 'filter_folder');
			$this->template->set($data);
			return $this->template->load_string('inbox/template', 'There are no messages to display.');
		}
		
		$total = element('total', $this->api->output_as_array());
		$total_unseen = $_SESSION['unseen'] = element('total_unseen', $this->api->output_as_array());
		$message_count = count($messages);
		if(!empty($_SESSION['unseen'])) {
			$title .= ' ('.$_SESSION['unseen'].')';
		}
		if($filter_folder === 'draft') { $filter_folder = 'drafts'; } //pluralize drafts
		$data = compact('current_user', 'display_name', 'folder_name', 'show_workflow', 'title', 'filter_folder');
		$this->template->set($data);
		//if we have an invalid page_start value, set it to something sensible if po ssible and start this script over again
		if($this->is->nonzero_unsigned_integer($total) && $page_start > $total){
			if($total > INBOX_DISPLAY_PER_PAGE)
				$_SESSION['page_start'] =  $total - INBOX_DISPLAY_PER_PAGE;
			else
				unset($_SESSION['page_start']);			
			redirect(current_url());
		}
		$this->template->set(compact('page_start', 'message_count', 'total'));
		$custom_flags = $this->flag_model->find_for_messages(array_keys($messages));
		foreach($custom_flags as $custom_flag){
			$messages[$custom_flag['message_id']]->flag = $custom_flag;
		}
		
		if($show_workflow){
			$workflow_items = $this->workflow_model->find_for_messages(array_keys($messages));
			foreach($workflow_items as $workflow_item){
				$messages[$workflow_item['message_id']]->workflow_item = $workflow_item;
			}
		}
		
		//SET UP THE SORT ICONS
		$sort_icon = '';
		if($sort_direction == 'asc') {  
			$sort_icon = ' <span class="sort_icon">&#x25b2;</span>'; 
		}elseif($sort_direction == 'desc') { 
			$sort_icon = ' <span class="sort_icon">&#x25bc;</span>'; 
		}
		$data = $data + compact('sort_icon', 'sort_column', 'sort_direction');	
				

		//handle special sort situations
		if($sort_column == 'flag') {
			usort($messages,'cmp_flag');
		}
		elseif($sort_column == 'workflow') {
			usort($messages,'cmp_workflow');
		}
		
		
		//set headers within data to pass to view
		$data['headers'] = $data['messages'] = $messages;
		$data['show_workflow'] = $show_workflow;
		
		//get last login data
		$get_last_logon = $this->db->query('SELECT TOP(2) * FROM logins WHERE username=' . $this->db->escape($this->session->userdata('username')) . ' ORDER BY login_time DESC');
		$row = $get_last_logon->row_array(1);
		if($row['success'] == TRUE) { $success = 'successful'; } else { $success = "<span style=\"font-weight: bold;\">unsuccessful</span>"; }
		$data['first_login_message'] = '<div id="logon_notice" class="modal_msg">';
		$data['first_login_message'] .= 'Last login attempt was ' . $success . ' from IP address: ' . $row['ip_address'] . ' at ' . date('m/d/y h:i:s A',$row['login_time']);
		$data['first_login_message'] .= '<a onclick="$(\'.modal_msg\').delay(1000).fadeOut(1000,function() { $(\'.modal_msg\').remove(); });"><img src="/jscripts/fancybox/fancy_close.png" alt="Close Login Message" /></a>';
		$data['first_login_message'] .= '</div>';
		if($get_last_logon->num_rows() === 1){ //first time logging in
			$data['first_login_message'] .='<script>$().ready(function(){getWebservicePermission(\"by_mailbox_group\");});</script>';
		}
		
		if($folder == 'sent' || $folder == 'draft'){			
			return $this->template->load('inbox/template', 'inbox/_messages_sent_draft', $data);
		}
		
		$this->template->load('inbox/template', 'inbox/_messages', $data);		
    }
	
     /* Provides number of messages in inbox since the index was last accessed. For use with ajax request to see
      * if new messages are waiting.
      */
    public function message_waiting_count() {
		$this->action_is_not_user_activity(); 
		
		if(!(isset($_SESSION['filter_folder']) && $this->session->mailbox_location() === 'archived')){ // being filtered in archive	
			$count = 0;
			$folder = element($this->session->mailbox_location(), $this->mailbox->folders);
			if(Folder::is_an_entity($folder)){
				$count = $folder->new;
			}
			
			if(isset($_SESSION['unseen'])) {
				echo $count - $_SESSION['unseen']; //don't show count with the unseen messages already displayed
				$_SESSION['unseen'] = $count;
			}
			else { 
				echo $count; 
			}
		}
		else{
			echo 0;
		}
    }

    /* This function changes the page start session value. The page start session value determines
     * the message id that the index function will start the current page display at.
     * TO DO: Add input validation. Low priority since invalid call will simply not return messages.
     */
    public function page($start) {
        $_SESSION['page_start'] = $start;
        redirect('inbox');
    }
	 /*This functions changes the the filter for the archive
	 */
	public function filter_archive($folder){
		if(in_array($folder, array('inbox','sent','draft'))){ $_SESSION['filter_folder'] = $folder; }
		else { unset($_SESSION['filter_folder']); }
		redirect('inbox');
	}
	
	public function sort($item) {
		//set sort column (if valid)
		$valid_sort_columns = array('workflow', 'flag', 'priority', 'timestamp', 'id', 'sender', 'to', 'cc', 'bcc', 'attachments', 'size', 'subject', 'plain');
		$same_column = FALSE;
		if(in_array($item, $valid_sort_columns)) {
			if(isset($_SESSION['sort_column']) && $_SESSION['sort_column'] === $item)
				$same_column = TRUE;
			$_SESSION['sort_column'] = $item;
		}
		//set sort direction
		if(!isset($_SESSION['sort_direction'])) { 
			$_SESSION['sort_direction'] = 'asc'; 
		}else {
			if($_SESSION['sort_direction'] === 'asc' && $same_column) { 
				$_SESSION['sort_direction'] = 'desc'; 
			}else if($_SESSION['sort_direction'] === 'desc' && $same_column) { 
				unset($_SESSION['sort_direction']); 
				unset($_SESSION['sort_column']);
			}
			else { 
				$_SESSION['sort_direction'] = 'asc'; 
			}
		}
		redirect('inbox');
	}
		
    /* This uses imap_search() to search the imap server for messages relating to the search term.
     * This function is currently implemented to check the subject, text, to and from fields only.
     * imap_search() is limited to IMAP2 search parameters (meaning no OR statement), so we have to perfrom
     * a seperate search for each criteria and then combine results.
     */
#KNOWN ISSUE - Sorting messages clears the search
    function search($advanced_search = FALSE) {
        $this->form_validation->set_rules('search_input','Search Input','xss_clean');
        if($this->form_validation->run() === TRUE) {
			$current_user = $this->user->id;
			$display_name = $this->user->cn;
			$title = 'Searching '.$this->mailbox->current_folder->name_for_display();
			$search_input = $this->input->post('search_input',TRUE);
			//advanced search filters
			$from = $this->input->post('from',TRUE);
			$recipient = $this->input->post('recipient',TRUE);
			$subject_title = $this->input->post('subject_title',TRUE);
			$plain_message = $this->input->post('plain_message',TRUE);
			$priority = $this->input->post('priority',TRUE);
			$original_folder = $this->input->post('original_folder',TRUE);
			$time_frame = $this->input->post('time_frame',TRUE);
			$folders_to_search = $this->input->post('folder_search_select',TRUE);
			if(empty($folders_to_search)) $folders_to_search = array();
			$folder_ids_to_search = array_map( function($value){ return first_element(explode('_', $value)); }, $folders_to_search);
			$is_multiple_folder_search = (count($folder_ids_to_search) > 1);

			$mailbox_list = $this->mailformat->mailbox_list($is_multiple_folder_search);
			$this->template->set('title', PORTAL_TITLE_PREFIX.$title);
			$this->template->set('mailboxes', $mailbox_list['folder_list']);
			$this->template->set('page_title', 'Message List');
			$this->template->set('form_destination', 'inbox/form_check');
			$this->template->set('menu_partial', 'inbox/_inbox_menu');

			//check to see what message we should start with
			$page_start = element('page_start', $_SESSION, 1);
			if(!$this->is->nonzero_unsigned_integer($page_start)){
				$page_start = 1;
				$_SESSION['page_start'] = $page_start;
			}
			
			//FIND THE MESSAGES	
#DON'T APPLY PAGINATION FOR NOW - SWITCHING PAGES CLEARS THE SEARCH
/*			$limit = $display_per_page = INBOX_DISPLAY_PER_PAGE; //configure how many emails to display per page 
			$start = $page_start - 1; */
			$order_by = $sort_column = element('sort_column', $_SESSION); //don't allow sorts for now - sorting gets us out of the search
			$order = $sort_direction = element('sort_direction', $_SESSION);

			if($advanced_search) {
				//set up original folder filter (in Archive mailbox only)
				$original_folder_str = '';
				if(!is_null($original_folder) && $original_folder !== false){
					foreach($original_folder as $value) {
						if(strpos($value,'all') === false) {
							$original_folder_str .= $value . ',';
						}
					}
				}
				else {
					$original_folder_str = '';
				}
				$original_folder_str = replace_last_with(',', '', $original_folder_str);
				$search_words = array(
						'contains_sender' => $from,
						'contains_recipients' => $recipient,
						'original_folder' => $original_folder_str,
						'contains_subject' => $subject_title,
						'contains_plain' => $plain_message
				);
				$filters = array();
				foreach ($search_words as $key=>$value){
					if($value && !empty($value)){
						$filters[$key] = $value;
					}	
				}
				if(!empty($subject_title)){
					$filters['contains_subject'] = $subject_title;
				}
				if(!empty($plain_message)){
					$filters['contains_plain'] = $plain_message;
				}
				if(is_numeric($priority)){
					$filters['priority'] = $priority;
				}
				if(is_numeric($time_frame)){  //check is a number.  Can be zero for today
						$filters['first_date'] = strtotime('-'.$time_frame.' days midnight');
				}
				$filter = rawurlencode(base64_encode(json_encode($filters)));					
			}
			else {
				if(empty($search_input))
					$search_input = '*';
				$filter = rawurlencode(base64_encode(json_encode(array( 'fuzzy' => $search_input))));
			}
			
			$folder = element('folder', $_SESSION, 'inbox');
			if(!empty($folder_ids_to_search)) {
				$folder = implode(',', $folder_ids_to_search);
			}
			
			$part = 'headers';
			$headers = $this->mailbox->messages( array_filter(compact('filter', 'folder', 'limit', 'order', 'order_by', 'part', 'start'), 'strlen') );
			$this->template->set('folder_name', element('folder_name', $_SESSION, 'Inbox'));
		
			//if we have any problems retrieving the messages, show the error right away before calculating all the other things		
			if($this->api->http_status == 403 && $this->api->message() == 'Access denied. The application is not authorized for this action.'){
				return $this->template->load('inbox/template', 'inbox/_service_permission_error');
			}elseif($this->api->http_status != 200 || !empty($this->api->format_errors)){
				return $this->template->load_string('inbox/template', 'Unable to retrieve search results. Please contact an administrator if the problem persists.');
			}
			
			if(empty($headers)){
				return $this->template->load_string('inbox/template', 'The search did not yield any results.');
			}
			
			$message_count = count($headers);
			$total = element('total', $this->api->output_as_array(), $message_count);
			
			$sort_icon = '';
			if($sort_direction == 'asc') {  
				$sort_icon = ' <span class="sort_icon">&#x25b2;</span>'; 
			}elseif($sort_direction == 'desc') { 
				$sort_icon = ' <span class="sort_icon">&#x25bc;</span>'; 
			} 
				

			$data = compact('current_user', 'display_name', 'headers', 'message_count', 'page_start', 'sort_column', 'sort_direction', 'sort_icon', 'total','original_folder');
			$data['is_multiple_folder_search'] = $is_multiple_folder_search;
			if(!empty($folder_ids_to_search)) {
				$data['folders'] = $mailbox_list['folders'];
			}
			
			//create a human-readable description of the search
			$search_message = 'Found '.$total.' '.pluralize_if_necessary('message', $total);					
			if(!$advanced_search || empty($folders_to_search)) {
				$search_message .= ' in '.$this->mailbox->current_folder->name_for_display();
				if(array_key_exists('search_input', $_POST))
					$search_message .= ' for "'.htmlentities($this->input->post('search_input',TRUE)).'"';
			}elseif(array_keys($this->mailbox->folders_in_hierarchical_order) == $folder_ids_to_search){
				$search_message .= ' in All Folders';		
			}elseif(count($folder_ids_to_search) == 1){
				$search_message .= ' in '.$this->mailbox->folders[first_element($folder_ids_to_search)]->name_for_display(); 
			}else{
				$search_message .= ' in '.count($folder_ids_to_search).' folders ('.array_to_human_readable_list( collect('name_for_display', array_intersect_key($this->mailbox->folders_in_hierarchical_order, array_flip($folder_ids_to_search)))).')';	
			}
			$this->template->set('search_message', $search_message);
			
			//load correct template for the folder we are in
			if($folder == 'sent' || $folder == 'draft'){
				return $this->template->load('inbox/template', 'inbox/_messages_sent_draft', $data); 
			}
			$this->template->load('inbox/template', 'inbox/_messages', $data);
		}
		else{
			redirect('/inbox');
		}
    }

	public function valid_formatted_address($str) {
		$valid_format = (!preg_match("/^([a-z0-9\+_\-]+)(\.[a-z0-9\+_\-]+)*@([a-z0-9\-]+\.)+[a-z]{2,6}$/ix", $str)) ? FALSE : TRUE;
		return $valid_format;
	}

	//$str should be a semi-colon separated list of addresses
	public function valid_trusted_address($address) {
		$address_array = explode(';',base64_decode(rawurldecode($address)));
		$address_array = $this->check_dist_list_before_sending($address_array); //check for distribution lists
		$address = rawurlencode(base64_encode(implode(';',$address_array)));

		$this->api->resource = 'direct/validate';
		$this->api->data = compact('address'); 
		$this->api->call();
		echo $this->api->raw_output; //echo so that js can use this
		return (bool)element('valid', $this->api->output_as_array());
	}
	
	/*Displays the feedback form view, meant for use in a modal window or pop-up */
	public function assign_search_menu() {
		$this->load->view('inbox/assign_message');
	}
	
	/* This function grabs the email address from the Get request and passes to the compose function on the flash data,
	 * so, that we do not have to display the link on the url
     */
	public function redirectCompose(){
		$mail = $_GET['mail'];
		$this->session->set_flashdata('mail', $mail);
		//Redirect to Compose
		redirect('/inbox/compose/');
	}
		
	public function search_group_members($search_term = NULL) {
		$group_name = $this->session->userdata('mailbox_group');
		$members = $this->get_group_members($group_name);
		$search_term = strtolower(rawurldecode($search_term));
		if(!is_null($search_term)) { 
			foreach($members as $index => $member) {
				if(!string_begins_with($search_term,strtolower($member['name'])) && !string_begins_with($search_term,strtolower($member['cn']))) {
					unset($members[$index]);
				}
			}
			$members = array_values($members);
		}
		//only echo if the request comes from AJAX
		if(IS_AJAX) {
			echo json_encode($members);
		}
	}     
	
	/**
	 * Dismiss a banner that is currently displayed on the site.
	 * @param string The string used to identify this banner in the session
	 */
	public function ajax_dismiss_banner($identifier){
		if(!IS_AJAX) show_404();
		$this->session->dismiss_banner($identifier);
	} 
    
//////////////////////////////////////////////////////////
// THINGS I WOULD LIKE TO MOVE TO THE MESSAGE CONTROLLER 
/////////////////////////////////////////////////////////		

    /**
	* Archives one or more messages.
	*
	* @todo Ideally, this should be on the message controller, but we can't move it there yet because it's still called directly from form_check
	* @param int|array $ids Either a single message id or an array of message ids
    */
    function archive($ids){
		if(Message::formatted_like_an_id($ids)) $ids = array($ids); //if we access via URL, just one message id will be passed - if it's from the form_check, it will be an array
		
		//validate that we have reasonable data being submitted
		if(empty($ids) || !$this->is->array_of_nonzero_unsigned_integers($ids)){
			$this->error->should_be_an_array_of_message_ids($ids);
			return show_404();
		}

		$errors = array();
		foreach($ids as $id){
			$message = Message::find_one($id);
			if(!Message::is_an_entity($message)){
				$this->error->should_be_a_message_id($id);
				return show_404();
			}
			
			if($message->archived) continue;
			if(!$message->archive() || !$message->archived){
				$this->error->warning('Could not archive '.$message->describe());
				$errors[] = $message->id;
			}
		}
		
		if(!empty($errors)){
			$feedback = 'An error occurred and ';
			
			if(count($ids) == 1)
				$feedback .= 'this message'; //we were only trying to archive one message
			else
				$feedback .= number_as_text(count($errors)).' '.pluralize_if_necessary('message', count($errors));
				
			$feedback .= ' could not be archived.  Please try again in a moment, and contact the site administrator if the problem persists.';

			$this->session->set_error_message($feedback);		
			redirect('inbox');
		}
		
		if(count($ids) == 1)
			$feedback = 'Success!  This messsage has been archived.';
		else
			$feedback = 'Success! '.ucfirst(number_as_text(count($ids))).' '.pluralize_if_necessary('message', count($ids)).' have been archived.';
		
		$this->session->set_success_message($feedback);
		redirect('inbox');
    }

	/**
	* Moves a message to a new location within this mailbox.
	* This may either be a system folder (Inbox, Drafts, Sent) or a custom folder.  Note that there is a separate method to {@link archive} messages.
	*
	* @todo Ideally, this should be on the message controller, but we can't move it there yet because it's still called directly from form_check
	*/ 
	protected function move() {
		$destination_id = $this->input->post('move',TRUE);
		if(empty($destination_id)){
			$destination_id = $this->input->post('move_select',TRUE); 
		}
		
		if(!array_key_exists($destination_id, $this->mailbox->folders)){
			$this->error->should_be_a_folder_for_this_mailbox($destination_id);
			show_404();
		}
		
		if(empty($_POST['selected_messages'])){
			$this->error->warning('No messages were selected to be moved to '.$destination_id.' for '.$this->error->describe($this->mailbox));
			show_404();
		}
		
		$errors = array();
		$message_ids = array_keys($_POST['selected_messages']);
		
		foreach($message_ids as $message_id){
			//locate and verify that we have a valid messsage id
			$message = Message::find_one($message_id);
			if(!Message::is_an_entity($message)){
				$this->error->should_be_a_valid_message_id($message_id);
				show_404();
			}
			
			$reason = ''; //this will remain blank if the move is a success
			if($destination_id == 'draft' && !$message->draft)
				$reason = 'only archived drafts may be moved to the Draft folder.';
			elseif($destination_id != 'draft' && $message->draft)
				$reason = 'archived drafts may only be moved to the Draft folder.';
			elseif($destination_id == 'sent' && !$message->sent)
				$reason = 'only sent messages may be moved to the Sent folder.';
			elseif($destination_id != 'sent' && $message->sent)
				$reason = 'archived sent messages may only be moved to the Sent folder.';
			elseif(!$message->move_to($destination_id))
				$reason = 'an unknown error occured.  Please try again in a moment, and contact the site administrator if the problem persists.';
	
			if(!empty($reason)){
				if(!isset($errors[$reason])) $errors[$reason] = array();
				$errors[$reason][] = $message->id;
			}
		}
		
		if(!empty($errors)){
			$feedback = '';
			if(count($message_ids) == 1){ //we were only attempting to move one message
				$feedback = 'This message could not be moved because '.array_first_key($errors); 
			}else{
				if(count($errors) > 1)
					$feedback = 'Some messages could not be moved. ';
				
				foreach($errors as $reason => $message_ids_with_errors){
					if(count($message_ids) == count($message_ids_with_errors))
						$feedback .= 'These '; //all messages failed to be moved, and they all had the same error
					else
						$feedback = ucfirst(number_as_text(count($message_ids_with_errors))).' '; //only some of the messages are being grouped under this error
					
					$feedback .= pluralize_if_necessary('message', count($message_ids_with_errors)).' could not be moved because '.$reason.' ';
				}
			}
			$this->session->set_error_message($feedback);
			redirect('inbox');
		}

		$this->session->set_success_message('Successfully moved '.number_as_text(count($message_ids)).' '.pluralize_if_necessary('message', count($message_ids)).'.');
		redirect('inbox');
    }
		
////////////////////////////////////
// PROTECTED HELPER METHODS
////////////////////////////////////

	protected function check_dist_list_before_sending($addresses) {		
		$addresses = $this->public_distribution_list_model->substitute_addresses_for_aliases($addresses);
		return $this->private_distribution_list_model->substitute_addresses_for_aliases($addresses);
	}
	
	//should be on attachment controller, but called by an inbox method as well as being a public helper method
    protected function clear_attachments() {
        return $this->user->clear_attachment_cache();
    }	
	
	
    #TODO - We should be getting this via the API in future 
    protected function get_group_members($group_name){
    	$dn = LDAP_GROUPS_DN;
    	$members  = array();
    	//get group information from ldap
    	if($this->ldap->connected()) {
    		$ldap_result = $this->ldap->search(NULL,NULL,array('cn','ou','description','member'),'(&(ObjectClass=groupofNames)(ou=' . $group_name . '))',$dn);
    		if(count($ldap_result) > 0) {
    			$data['cn'] = $ldap_result[0]['cn'];
    			$data['description'] = isset($ldap_result[0]['description']) ? $ldap_result[0]['description'] : '' ;
    			//get group member info from ldap
    			if(is_array($ldap_result[0]['member'])) {
    				foreach($ldap_result[0]['member'] as $member) {
    					$username = str_replace('uid=','',$member);
    					$username = str_replace(','.LDAP_ACCOUNTS_DN,'',$username);
    					//if($this->user_model->logged_in_user('user_name') !== $username){//remove own username
    					//check that user value corresponds to valid user
    					$get_user = $this->db->query("SELECT user_id FROM users WHERE user_deleted_flag=0 AND user_name=" . $this->db->escape($username));
    					if($get_user) {
    						if($get_user->num_rows() == 1) {
    							$user_result = $this->ldap->search(null,1,array('cn','displayname'),'(uid='.$username.')');
    							$get_user = $get_user->result();
    							if(count($user_result) > 0) {
    								$member = array('cn' => $user_result[0]['cn'], 'name' => $user_result[0]['displayname'], 'username' => $username, 'id'=>$get_user[0]->user_id);
    							}
    							else { $member = array('name' => $username, 'username' => $username,'id'=>$get_user[0]->user_id); }
    							array_push($members ,$member);
    						}
    					}
    					//}
    				}
    			}
    		}
    	}
    	return $members;
    }

    /* This function marks mail as read */
    protected function mark_as_read() {
        foreach($this->input->post('selected_messages', TRUE) as $key => $value) { 
            if($value == 'on') {
				Message::find( array('id' => $key, 'part' => 'flags', 'mark' => 'read', 'limit' => 1) );		
            }
        }
        redirect('inbox');
    }
	
    /* This function marks mail as unread */
    protected function mark_as_unread() {
        foreach($this->input->post('selected_messages', TRUE) as $key => $value) { 
            if($value == 'on') { 
                Message::find( array('id' => $key, 'part' => 'flags', 'mark' => 'unread', 'limit' => 1) );	
            }
        }
        redirect('inbox');
    }	
	
	protected function _message_from_post(){		
		//FIND OR CREATE THE MESSAGE
		$id = $this->input->post('msg_id',TRUE);
		if(Message::formatted_like_an_id($id)){
			$message = Message::find_one(array('id' => $id, 'sender' => $this->mailbox->email_address()));
			if(!Message::is_an_entity($message)) return $this->error->should_be_a_message_id($id);
		}else{
			$message = new Message( array('sender' => $this->mailbox->email_address()) );
			if(!Message::is_an_entity($message)) return $this->error->warning("I couldn't create a new message for ".$this->mailbox->describe());
		}
		
		//FIND THE VALUES FROM THE FORM
		$values = array( 'original_sender' => $this->user->username,
						 'importance' => $this->input->post('priority', TRUE),
						 'to' => $this->input->post('message_to', TRUE),
						 'cc' => $this->input->post('message_cc', TRUE),
						 'subject' => $this->input->post('message_subject',TRUE),
						 'body' => $this->input->post('message_body', FALSE),
						 'protected_data' =>  $this->input->post('7332_data', FALSE));
			 
		//SET THE FORM VALUES *IFF* THEY'RE DIFFERENT FROM WHAT WE CURRENTLY HAVE (we don't want to make an API call unless we need to)				 				 
		foreach($values as $field => $value){
			if($message->$field != $value)
				$message->$field = $value; 
		}
		
		return $message;
	}	
	
	/**
	* Standard error responses if we're not able to look up a message by id
	* @param Message The result of a Message::find_one() lookup.
	*/
	protected function show_error_if_not_a_message($message){
		if(!Message::is_an_entity($message)){
			if(Message::api()->http_status == 200)
				show_404(); //api call worked, but the message doesn't exist
			elseif(Message::api()->http_status == 403)
				show_error('Access denied. The application is not authorized for this action.');
			else
				show_error('Unable to retrieve message. Please contact an administrator if the problem persists.');
		}
	}
	

}	
/* End of file inbox.php */
/* Location: ./application/controllers/inbox.php */