<?PHP  if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package vler
* @subpackage libraries
*/ /** */

//make sure that the class knows not to try to make conversions available if this application isn't configured for it.
if(!defined('USE_DOC_TO_PDF_CONVERSION_PROCESS')) define('USE_DOC_TO_PDF_CONVERSION_PROCESS', FALSE);

/**
* @todo replace Message:: calls with something that checks for the model message class name first.
* @package vler
* @subpackage libraries
*/
class Word_attachment extends Attachment{
	protected $_use_lightbox_preview = false;	
	protected $_pdf_attachment_class;

	/**
	* Creates a PDF version of this Word attachment and returns it as an attachment object.
	* @return PDF_attachment
	*/
	public function to_pdf(){
		if(!USE_DOC_TO_PDF_CONVERSION_PROCESS) return false;
		if($this->pdf_exists_in_cache()) return $this->pdf_from_cache();

		//add Word version of this attachment to cache.
		if(!$this->add_file_to_pdf_cache($this->name, $this->binary_string)){
			$this->error->warning('Unable to convert '.$this->describe().' to PDF: could not write '.$this->name.' to system cache');
		}

		//Try calling LibreOffice from the command line directly if we want to do this as per constants
# Code review note - can we do the COM check first and try using the command line if the COM library isn't available/fails?  -- MG 2015-01-08
		if(USE_COMMAND_LINE_DOC_TO_PDF) {
			if(!$this->convert_to_pdf_using_command_line()) return false;
		}else{
			if(!$this->convert_to_pdf_using_com()) return false;
		}

		if(!$this->pdf_exists_in_cache())
			return $this->error->warning('Unable to convert '.$this->describe().' to PDF: '.$this->pdf_cache().$this->name_for_pdf().' was not created successfully.');

		return $this->pdf_from_cache();
	}

	public function url_for_preview(){
		if(!$this->use_pdf_preview) return $this->url_for_download();
		
		$pdf_attachment_class = $this->pdf_attachment_class;
		return site_url($pdf_attachment_class::pdf_viewer_path())."?file=".rawurlencode($this->url_for_view_inline());
	}
	
	public function url_for_view_inline(){
		return $this->url_for_view();
	}

	/**
	* True if the word-to-pdf conversion tool is available and working.
	*
	* Note that the application may be configured not to allow documents to be converted to PDF via {@link USE_DOC_TO_PDF_CONVERSION_PROCESS}.
	*
	* @see USE_DOC_TO_PDF_CONVERSION_PROCESS
	* @return boolean
	*/
	public function use_pdf_preview(){
		if(!USE_DOC_TO_PDF_CONVERSION_PROCESS) return false;
		$pdf_attachment_class = $this->pdf_attachment_class;
		if(!$pdf_attachment_class::pdf_viewer_is_available()) return false;
		if($this->bytes > DOC_TO_PDF_MAX_SIZE) return false;
		return is_a($this->to_pdf(), 'PDF_attachment');
	}

	public function view(){
		if(!$this->use_pdf_preview()) return $this->download();
		
		//convert this Word document to PDF
		$pdf = $this->to_pdf();
		
		//if the conversion worked, display the PDF
		if(is_a($pdf, 'PDF_attachment'))
			return $pdf->view();
		
		//default to downloading the document if we ran into problems.
		return $this->download();
	}

	public function view_inline(){
		if(!$this->use_pdf_preview()) return $this->download();

		//convert this Word document to PDF
		$pdf = $this->to_pdf();

		//if the conversion worked, display the PDF
		if(is_a($pdf, 'PDF_attachment'))
			return $pdf->view_inline();

		//default to downloading the document if we ran into problems.
		return $this->download();
	}

/////////////////////////
// CACHE METHODS
/////////////////////////	

	/**
	* Directory where converted pdfs and their Word source files are stored.
	* Includes a trailing slash
	* @return string 
	*/
	public function pdf_cache(){
		$user = User::find_from_session();
		if(method_exists($user, 'system_cache'))
			return $user->system_cache().'word_to_pdf_conversions/';
		return $this->error->warning('There is no pdf cache available for '.$this->describe().' because the User model does not have a system_cache() method');
	}	

	/**
	* @return string
	*/
	public function name_for_pdf(){
		return replace_last_with($this->extension, 'pdf', $this->name);
	}	


	function add_file_to_pdf_cache($name, $binary_string){			
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!is_string($binary_string)) return $this->error->should_be_a_binary_string($binary_string);
		if(empty($this->pdf_cache())) return $this->error->warning('I cannot add '.$this->describe().' to the pdf cache because the path for the PDF cache has not been specified');

		$user = User::find_from_session();
		if(!User::is_an_entity($user)) return $this->error->warning("Could not determine logged-in user");

		//make sure that we can create a pdf cache
		if(!directory_exists($this->pdf_cache())) mkdir($this->pdf_cache(),0777,true); //attempt to create the cache if needed
		if(!directory_exists($this->pdf_cache())) return $this->error->warning('Could not create an attachment cache for '.$this->describe().' at '.$this->attachment_cache());

		if(file_exists($this->pdf_cache().$name) && file_get_contents($this->pdf_cache().$name) == $binary_string) return true; //exact file is already in the cache; we're done

		$path = $this->pdf_cache().$name;		
		$success = file_put_contents($path, $binary_string);	
		if($success === FALSE)
			return $this->error->warning('Unable to write '.$path.' to '.$this->pdf_cache());
		return true;
	}

	/**
	* Removes the PDF cache and all of its files.
	* @return boolean True on success
	*/
	public function clear_pdf_cache(){
		if(empty($this->pdf_cache())) return $this->error->warning('I cannot add clear the pdf cache for '.$this->describe().' the path for the PDF cache has not been specified');
		if(!directory_exists($this->pdf_cache())) return true; //we're done!
		delete_files($this->pdf_cache());
		return rmdir($this->pdf_cache());
	}		

	/**
	* Checks to see if a pdf with the right name exists in the cache and checks its Word source file to make sure that the content matches.
	* @return boolean
	*/
	public function pdf_exists_in_cache(){
		if(empty($this->pdf_cache())) return $this->error->warning('I cannot check to see if the PDF for '.$this->describe().' is in the PDF cache because the path for the PDF cache has not been specified');
		if(!file_exists($this->pdf_cache().$this->name_for_pdf)) return false;
		if(!file_exists($this->pdf_cache().$this->name)) return false; //if the word file isn't in the cache, we don't have anything to compare to, so assume false
		$word_binary_string = file_get_contents($this->pdf_cache().$this->name);
		if($word_binary_string === FALSE){
			return $this->error->warning('Could not read '.$this->pdf_cache().$this->name);
		}

		return ($word_binary_string == $this->binary_string);	
	}

	/**
	* @return PDF_attachment|null
	*/
	protected function pdf_from_cache(){
		if(empty($this->pdf_cache())) return $this->error->warning('I cannot retrieve the pdf for '.$this->describe().' from the cache because the path for the PDF cache has not been specified');
		if(!$this->pdf_exists_in_cache()) return null;

		$values = array('name' => $this->name_for_pdf, 'directory' => $this->pdf_cache());
		if(Message::formatted_like_an_id($this->message_id))
			$values['message_id'] = $this->message_id; //make sure message id gets preserved so that URLs are correct

		return new_instance_of('library', 'attachments/pdf_attachment', compact('values'));	
	}

	/**
	* Removes both the pdf and the word file from the pdf cache.
	* @return boolean True on success
	*/
	public function remove_pdf_from_cache(){
		if(empty($this->pdf_cache())) return $this->error->warning('I cannot remove the pdf for '.$this->describe().' from the cache because the path for the PDF cache has not been specified');
		if(!directory_exists($this->pdf_cache())) return true; //we're done!		

		$success = true;
		foreach(array($this->name, $this->name_for_pdf) as $name){
			$path = $this->pdf_cache().$name;
			if(!file_exists($path)) return true; //we're done!
			if(!unlink($path)) $success && $this->error->warning('Failed to remove '.$path);
		}
		return $success;		
	}


////////////////////////
// PROTECTED FUNCTIONS
///////////////////////

	//use the PHP COM library to connect to a C# library that converts the doc using the LibreOffice SDK
	//returns boolean true on success
	protected function convert_to_pdf_using_com(){
		if(empty($this->pdf_cache())) return $this->error->warning('I cannot convert '.$this->describe().' to PDF because the path for the PDF cache has not been specified');
		
		//fail gracefully if COM is not available on this server
		if(!class_exists('COM')) return $this->error->warning('Unable to convert '.$this->describe().' to PDF: COM library is not available'); 

		try{
			$helper = new COM("newPDFConversion.Class1");
			$helper->convert_to_pdf(escapeshellarg($this->pdf_cache().$this->name),$this->pdf_cache());
		}
		catch(Exception $e) {
			return $this->error->warning('Unable to convert with COM Object '.$this->describe().' to PDF: '.$e->getMessage());
		}

		return true;
	}

	//use the LibreOffice command line tool to convert this word attachment to a PDF
	//@see LibreOffice link
	protected function convert_to_pdf_using_command_line(){
		if(empty($this->pdf_cache())) return $this->error->warning('I cannot convert '.$this->describe().' to PDF because the path for the PDF cache has not been specified');
		
		if(!USE_COMMAND_LINE_DOC_TO_PDF) return false;

		$conversion_tool_directory = "C:\Program Files (x86)\LibreOffice 4\program";
		if(!directory_exists($conversion_tool_directory)) 
			return $this->error->warning("I couldn't find the word-to-pdf conversion tool on this server: ".$conversion_tool_directory);

#TODO - I'd rather not have us attempt to switch directories if we can avoid it - can't we just run the command line tool by specifying the full path for it? -- MG 2015-02-04
		if(!chdir($conversion_tool_directory)) 
			return $this->error->warning("I wasn't able to navigate to the word-to-pdf conversion tool directory: ".$conversion_tool_directory);	

		$CI = get_instance();						
		$source_path = $this->pdf_cache().$this->name;
		$target_directory = $this->pdf_cache();

		$run_exe   = 'soffice';
		$env_param   = ' --headless --nofirststartwizard -convert-to pdf -outdir '. escapeshellarg($target_directory).' '. escapeshellarg($source_path);

		$run_path = $run_exe .' '. $env_param;

		$descriptorspec = array (
				0 => array("pipe", "r"),
				1 => array("pipe", "w"),
		);

		#TODO - Consider figuring out a way to pass the PHP "max_execution_time" config varaible as the timeout constraint instead of a static value. - BH 2015-02-02
		#TODO - You can do this using ini_get(), but keep in mind that there are server-level constraints that might reduce max execution time - MG 2015-02-04
		$timeout = 150; //timeout in seconds
		$start_time = time();

		if(!is_resource( $prog = proc_open("start /wait " . $run_path, $descriptorspec, $pipes, $conversion_tool_directory, NULL) ) )
			return $this->error->warning('Command-line conversion of Word Document to PDF failed.'); 

#TODO - BH, are you sure you don't want to use log_message instead of a warning message?  Warnings output HTML messages that can interfere with rendering a pdf file. -- MG 2015-02-04
		while(time() < $start_time + $timeout){
			//$ppid = proc_get_status($prog)['pid'];
			$ppid = proc_get_status($prog);
			if ($ppid['running']){
				#if(is_on_local()) $this->error->warning('Command-line conversion is still running for file '. $source_path );
				sleep(1);
			}
			else{
				#TODO - BH, do you mean to have an error message here?  Error messages should only be triggered when something goes wrong.
				#if(is_on_local()) $this->error->warning('Command-line conversion is NOT running');
				return true;
			}
		}

		proc_terminiate($prog);
		return $this->error->warning('Conversion of '.$this->path.' to PDF timed out');
	}

////////////////////
// GETTERS
////////////////////
	
	public function pdf_attachment_class(){
		if(!isset($this->_pdf_attachment_class)){
			//since we're allowing the application to extend attachment classes, we can't assume that it will be the actual PDF_attachment class
			return get_instance()->load->class_name_for('library', 'attachments/pdf_attachment');
		}
	}


/////////////////////
// STATIC FUNCTIONS
/////////////////////

	public static function create($name, $binary_string, $more_values = array()){
		if(empty($binary_string)) return get_instance()->error->should_be_an_x('name of a '.humanize(get_called_class()), $name);
		if(!static::matches_file($name, $binary_string)) return get_instance()->error->should_be_an_x('name of a '.humanize(get_called_class()), $name);
		return new Word_attachment(array_merge(compact('name', 'binary_string'), $more_values));
	}


	/**
	* True if the given file can be cast as an instance of this class.
	* @param string Name of the file
	* @param string Content of the file
	* @return boolean
	*/
	public static function matches_file($name, $binary_string){
		if(!parent::matches_file($name, $binary_string)) return false;
		return in_array(strtolower(pathinfo($name, PATHINFO_EXTENSION)), array('doc', 'docx'));
	}

}
?>