<?php
/**
* @package vler
* @subpackage libraries
*/ /** */

load_helpers('date', 'environment', 'inflector', 'string');

/**
* This is a functional accessor for {@link Validator}, to be used when there is not a validator object in scope.  Using the validator object directly 
* is the preferred best practice.  This is the equivalent of $validator->$rule().  See the documentation for {@link Validator} for more information.
* @param string $rule The equivalent of any of the validator methods with the "is_a" removed.
* @param mixed The value to be validated
* @return boolean
*/
if(!function_exists('validates_as')){
	function validates_as($rule, $value){
		$parameters = func_get_args();
		$validator = new Validator();
		if(count($parameters) == 2)
			return $validator->$rule($value);
		else{
			$rule = $parameters[0];
			unset($parameters[0]);
			return call_user_func_array(array($validator, $rule), $parameters);
		}
	}
}

if(!function_exists('get_validator')){
	$VALIDATOR;
	function get_validator(){
		global $VALIDATOR;
		if(!is_a($VALIDATOR, 'Validator'))
			$VALIDATOR = new Validator();
		return $VALIDATOR;
	}
}

/**
* Methods to type-check and validate data, as well as some magical methods to faciliate checking arrays, lists, and other groupings of objects
*
* All of the methods prefaced with "is_a" or "is_an" are validation rules.  Validation rules always take an unknown value as a parameter and return
* true if the value matches the validation rule.  You can access them in each of the following ways:
* <code>
*	$validator->is_an_associative_array($value);
*	//works, but the method above is more efficient & the best practice
*	$validator->associative_array($value);
*	//best practice when a Validator object is not in scope (e.g., functions in a helper library) 
*	validates_as('associative_array', $value); 
* </code>
*
* Additionally, validation rules that end with "_x" and are listed in {@link custom_prefixes} are magical methods intended to be paired with other validation rules.
* <code>
*	//standard way of calling 'is_an_array_of_x' paired with 'is_a_nonzero_unsigned_integer'
*	$validator->is_an_array_of_nonzero_unsigned_integers($value);
*	//equivalent when a Validator object is not in scope
*	validates_as('array_of_nonzero_unsigned_integers');
*	//what __call is calling when you use either of the methods above (more efficient, but less readable)
*	$validator->is_an_array_of_x('nonzero_unsigned_integer', $value); 
* </code>
* 
* By convention, validation rule names distinguish between rules that check to make sure that a value *could* be something and rules to check that it actually is.
* For example:
*	<code>
*	//checks to make sure that the string is formatted in a way that it *could* be a file path
*	$validator->is_a_string_like_a_file_path($value); 
*	//checks to make sure that there's actually a file with this path
*	$validator->is_a_file_path($value);
*
* //checks to make sure it's a six-digit numeric
*	$validator->is_a_string_like_an_emplid($value);
* //checks the database to make sure it's an existing emplid.
*	$validator->is_an_emplid($value); 
* </code>
*
* @package vler
* @subpackage libraries
* @author M. Gibbs <gibbs_margaret@bah.com>
*/
class Validator{
	protected $custom_prefixes = array( 'is_an_array_of_arrays_where_key_value_is_an_x' => array('is_an_array_of_arrays_where_key_value_', 'array_of_arrays_where_key_value_'),
										'is_an_array_of_x' => array('is_an_array_of_', 'array_of_'),
										'is_an_associative_array_of_x' => array('is_an_associative_array_of_', 'associative_array_of_'),
										'is_a_comma_separated_list_of_x' => array('is_a_comma_separated_list_of_', 'comma_separated_list_of_'),
										'is_a_comma_and_whitespace_separated_list_of_x' => array( 'is_a_comma_and_whitespace_separated_list_of_', 
										'comma_and_whitespace_separated_list_of_'));
	

	/**
	* True if this library is being run from a CodeIgniter application.
	* Checks to make sure that we have access to the main CI controller object.
	* @return boolean
	*/
	function is_in_a_codeigniter_context(){
		if(!defined('CI_VERSION')) return false;
		if(!function_exists('get_instance')) return false;
		return is_a(get_instance(), 'CI_Controller');
	}

//begin the phpdocumentor template for validation rules -- all of these will take an unknown type as a param and return a boolean
/**#@+
* @param mixed
* @return boolean
*/
	function scalar($value){
		return is_scalar($value);
	}
	
	function boolean($value){
		return is_bool($value);
	}

	function null($value){
		return is_null($value);
	}
	
	function object($value){
		return is_object($value);
	}
	
	function object_with_a_property_named($value, $property){
		if(!$this->is_an_object($value)) return false;
		if(!$this->is_a_nonempty_string($property)) return false;
		return property_exists($value, $property);
	}
	
	function entity($value){
		return is_a($value, 'Entity');
	}


//////////////////
// ARRAYS
//////////////////	
	function is_an_array($value){
		return is_array($value);
	}
	
	function nonempty_array($value){
		return !empty($value) && $this->is_an_array($value);
	}

	/** Checks to make sure that at least some of the keys are non-numeric strings.
	* Be warned that this method is not smart enough to distinguish between an array with no keys specified and an array that has
	* non-arbitrary numeric keys (for example, ids corresponding to database rows. */
	function associative_array($value){
		if(!$this->is_an_array($value)) return false;
		if(empty($value)) return true;  //it counts if it's empty, so long as it's an array
		return !$this->is_an_array_of_integers(array_keys($value));
	}

  
	/**
	* True if every entry in $array matches rule $x.
	* Note that you can access this method by replacing x in the method name with the validation rule, e.g.:
	*
	* <code>
	* $this->validator->is_an_array_of_strings(array('a','b','c')); //returns true
	* $this->validator->is_an_array_of_numerics(array('a', 1, 2)); //returns false
	* </code>
	*
	* @param string The validation rule that each entry in the array should match. This should correspond to anothe rmethod in the {@link Validator} class.
	* @param mixed The value that is being checked
	*/
	function array_of_x($x, $array){				
		if(!$this->is_an_array($array)) return false;
		if(empty($array)) return true;	//OK for it to have nothing at all in it
				
		if($this->is_a_validation_rule($x))
			$rule = $x;
		elseif($this->is_a_validation_rule(singular($x)))
			$rule = singular($x);
			
		if(!isset($rule)){
			trigger_error('"'.$x.'" is not a known validation rule');
			return false;
		}
		
		$still_valid = true;
		foreach($array as $value){
			$still_valid = $this->$rule($value);
			if(!$still_valid) return false;
		}
		
		return $still_valid;
	}

	/**
	* True if $array is an associative array and every entry in $array matches rule $x.
	* See {@link is_an_associative_array} for more information on what Validator considers to be an associative array.
	* 
	* Note that you can access this method by replacing x in the method name with the validation rule, e.g.:
	* <code>
	* $this->validator->is_an_associative_array_of_strings(array('one' => 'a', 'two' => 'b', 'three' => 'c')); //returns true
	* $this->validator->is_an_associative_array_of_numerics(array('one' => 'a', 'two' => 1, 'three' => 2)); //returns false
	* </code>
	*
	* @param string The validation rule that each entry in the array should match. This should correspond to anothe rmethod in the {@link Validator} class.
	* @param mixed The value that is being checked
	*/
	function associative_array_of_x($x, $array){		
		return ($this->is_an_associative_array($array) && $this->is_an_array_of_x($x, $array));
	}
	
	function array_of_arrays_with_key_called($array, $key){
		if(!$this->is_an_array($array)) return false;  //we could check is_an_array_of_arrays, but don't want to iterate the array twice in case it's huge.
		foreach($array as $child_array){
			if(!$this->is_an_array($child_array)) return false;
			if(!array_key_exists($key, $child_array)) return false;
		}
		return true;
	}
	
	//most awkward name EVER -- if you can come up with a better name, create a wrapper
	function array_of_arrays_where_key_value_is_an_x($x, $array, $key){
		if(!$this->is_an_array($array)) return false;  //we could check is_an_array_of_arrays_with_key_called, but don't want to iterate the array twice in case it's huge.
		foreach($array as $child_array){
			if(!$this->is_an_array($child_array)) return false;
			if(!array_key_exists($key, $child_array)) return false;
			if(!$this->$x($child_array[$key])) return false;
		}
		return true;
	}
	
	function array_of_objects_with_a_property_named($array, $property){
		if(!$this->is_an_array($array)) return false;
		foreach($array as $object){
			if(!$this->is_an_object_with_a_property_named($object, $property)) return false;
		}
		return true;
	}
	
//////////////
// DATES
//////////////	
	
	function mysql_date($value){
		if(!$this->is_a_nonempty_string($value)) return false;
		if(mb_strlen($value) != 10) return false;
		$pattern = '/^([0-9]{4})-([0-1][0-9])-([0-3][0-9])$/';
		return (boolean)preg_match($pattern, $value);
	}
	
	function mysql_datetime($value){
		if(!$this->is_a_nonempty_string($value)) return false;
		if(mb_strlen($value) != 20) return false;
		$pattern = '/^([0-9]{2,4})-([0-1][0-9])-([0-3][0-9]) ([0-2][0-9]):([0-5][0-9]):([0-5][0-9])$/';
		return (boolean)preg_match($pattern, $value);
	}
	
	function mysql_time($value){
		if(!$this->is_a_nonempty_string($value)) return false;
		if(mb_strlen($value) != 20) return false;
		$pattern = '/^([0-2][0-9]):([0-5][0-9]):([0-5][0-9])$/';
		return (boolean)preg_match($pattern, $value);
	}
	
	function nonempty_mysql_date($value){
		if(!$this->is_a_mysql_date($value)) return false;
		return !mysql_date_is_empty($value);
	}	
	
	function nonempty_mysql_datetime($value){
		if(!$this->is_a_mysql_datetime($value)) return false;
		return !mysql_date_is_empty($value);
	}
	
	function nonempty_mysql_time($value){
		if(!$this->is_a_mysql_time($value)) return false;
		return !mysql_time_is_empty($value);
	}
	
	///OK, seems like we should be more sophisticated than this check, but at the same time, quick research is not being clear on whether we're
	//required to pad out to a certain number of digits or if it's just convention.  For simplicity of the moment, keep things as simple as possible.
	//also, 0 is allowed because we might actually reference December 31, 1969; negative numbers are ok because some systems allow negatives for pre-1969.
	function unix_timestamp($value){
		if(!$this->is_a_numeric($value)) return false;
		return $value == floor($value); //this should return true for whole numbers
#		return $this->is_an_integer($value); //can't just do integer check -- timestamps are sometimes too big to be integers
	}	

///////////////////////
// NUMBERS
////////////////////////
	
	function is_numeric($value){
		return is_numeric($value);
	}
	
	//wrapper for consistency
	function numeric($value){
		return $this->is_numeric($value);
	}
		
	/** 
	* Checks to see if the value matches PHP's definition of an integer.
	* Note that this rule doesn't care if the value is currently cast as an integer, but whether or not it can
	* be cast as an integer without rounding or truncating values. 
	*/
	function integer($value){		
		//make sure that this looks like a number
		if(!is_numeric($value))
			return false;
		
		//check to make sure it doesn't have a decimal component
		$before_casting = $value;		
		return $before_casting == (int)$value;
	}
		
	function unsigned_integer($value){
		return $this->is_an_integer($value) && $value > -1;
	}
	
	function nonzero_unsigned_integer($value){
		return $this->is_an_integer($value) && $value > 0;
	}
	
	function float($value){
		return is_float($value);
	}
	
	function double($value){
		return $this->is_a_float($value); //doubles and floats are the same thing in PHP
	}
	
	/** 
	* Checks to see if the value matches the mathematical definition of an integer.
	* To check if the value matches PHP's definition of an integer, see {@link is_an_integer}.	
	*/
	function whole_number($value){
		if(!$this->is_numeric($value)) return false;
		return $value == floor($value); 
	}
		
		
///////////////////
// STRINGS
//////////////////	

	function string($value){
		return is_string($value);
	}
			
	function nonempty_string($value){
		return !empty($value) && is_string($value);
	}
	
	function string_with_no_whitespace($string){
		if(!$this->is_a_string($string)) return false;
		if($string == '') return true;
		return !preg_match('/\s/', $string);
	}
	
	function nonempty_string_with_no_whitespace($string){
		return ($this->is_a_nonempty_string($string) && $this->is_a_string_with_no_whitespace($string));
	}
	
	//empty strings are ok
	//note - this currently doesn't allow anything but ascii as the domain name.  this is ok for now, 
	//but in the long run, we should allow other letters: http://www.domainnameshop.com/faq.cgi?id=8&session=106ee5e67d523298
	function string_like_a_url($string){
		if(!$this->is_a_string_with_no_whitespace($string)) return false;
		if(empty($string)) return true;

		//pattern & comments from http://geeksandgod.com/forums/web-development/php-/-mysql/validate-url-regex
		$pattern = "/^"; 									 # Start at the beginning of the text
		$pattern .= "(?:ftp|https?):\/\/"; # Look for ftp, http, or https
		$pattern .= "(?:(?:[\w\.\-\+%!$&'\(\)*\+,;=]+:)*[\w\.\-\+%!$&'\(\)*\+,;=]+@)?"; # Userinfo (optional)
		$pattern .= "(?:[a-z0-9\-\.%]+)";  # The domain
		$pattern .= "(?::[0-9]+)?";        # Server port number (optional)
		$pattern .= "(?:[\/|\?][\w#!:\.\?\+=&%@!$'~*,;\/\(\)\[\]\-]*)?"; # The path (optional)
		$pattern .= "$/xi"; #end

		return (boolean)preg_match($pattern, $string);
	}
	
	function string_like_an_email_address($string){
		if(!$this->is_a_string_with_no_whitespace($string)) return false;
		if(empty($string)) return true;
		
		return ($string == filter_var($string, FILTER_VALIDATE_EMAIL));
	}
	
	function string_like_a_direct_address($string){
		return ($this->is_a_string_like_a_patient_address($string) || $this->is_a_string_like_a_clinical_address($string));
	}
	
	function string_like_a_patient_address($string){
		if(!defined('PATIENT_DOMAIN') || !$this->is_a_direct_domain(PATIENT_DOMAIN)) return false;
		if(!$this->is_a_string_like_an_email_address($string)) return false;
		return string_ends_with('@'.PATIENT_DOMAIN, $string);
	}
	
	function string_like_a_clinical_address($string){
		if(!defined('CLINICAL_DOMAIN') || !$this->is_a_direct_domain(CLINICAL_DOMAIN)) return false;
		if(!$this->is_a_string_like_an_email_address($string)) return false;
		return string_ends_with('@'.CLINICAL_DOMAIN, $string);
	}
	
	function direct_domain($domain){
		return in_array($domain, valid_domains());
	}
	
	function string_like_an_ip_address($value){
		return (filter_var($value, FILTER_VALIDATE_IP) !== false);
	}
	
	function string_like_a_social_security_number($value){
		if(!$this->is_a_nonempty_string($value)) return false;
		
		$pattern = '/^[0-9]{3}-[0-9]{2}-[0-9]{4}$/';
		if(!(boolean)preg_match($pattern, $value)) return false; //if it doesn't match the right dash pattern, return false
		
		//check for special cases that don't work -- see http://en.wikipedia.org/wiki/Social_security_number#Structure
		/*if((boolean)preg_match('/^(000|666)-[0-9]{2}-[0-9]{4}$/', $value)) return false; //can't have 000 or 666 as the area number
		if((boolean)preg_match('/^[0-8][0-9]{2}-00-[0-9]{4}$/', $value)) return false; //group number can't be 00
		if((boolean)preg_match('/^[0-8][0-9]{2}-[0-9]{2}-0000$/', $value)) return false; //serial number can't be 0000*/
		
		return true;
	}		
	
	function comma_separated_list_of_x($x, $string, $trim_whitespace=false){		
		if(!$this->is_a_string($string)) return false;
		if(empty($string)) return true;	//OK for it to have nothing at all in it
				
		if($this->is_a_validation_rule($x))
			$rule = $x;
		elseif($this->is_a_validation_rule(singular($x)))
			$rule = singular($x);
			
		if(!isset($rule)){
			trigger_error('"'.$x.'" is not a known validation rule');
			return false;
		}
		
		$values = explode(',', $string);
		if($trim_whitespace) $values = array_map('trim', $values);
		return $this->is_an_array_of_x($x, $values);
	}
	
	function comma_and_whitespace_separated_list_of_x($x, $string){
		return $this->is_a_comma_separated_list_of_x($x, $string, true);
	}
		
	
///////////////
// FILES
///////////////	
	
	/**
	* True if this is a string that *could* be the name of a file path.
	* This method just checks to make sure that the value is a string that wouldn't be completely and obviously invalid as the name of a file.
	* If you want to check to see if a string is the name of a file that actually exists, use {@link is_a_file_path}.
	* @see is_a_file_path
	* @todo add more stringent tests; currently just checks to make sure it's not empty and there are no whitespaces
	*/
	function string_like_a_file_path($value){
		#return ($this->is_a_nonempty_string($value) && $this->is_a_string_with_no_whitespace($value));
		//previous code was for a Unix-based system, spaces are OK in Windows - would be nice to do better enforcement here, though.
		//Microsoft has a page specifying their requirements: http://msdn.microsoft.com/en-us/library/aa365247%28v=vs.85%29.aspx
		return ($this->is_a_nonempty_string($value));   
	}
	
	/**
	* True if this is a path to an existing file. 
	* Note that the value must be a file, not a directory.  If it's not required that this be the name of a file that actually exists, use 
	*{@link is_a_string_like_a_file_path}.
	* @see is_a_string_like_a_directory_path
	*/
	function file_path($value){
		if(!$this->is_a_string_like_a_file_path($value)) return false;
		return (file_exists($value) && !is_dir($value));
	}
	
	/**
	* True if this is a string that *could* be the name of a directory path.
	* This method just checks to make sure that the value is a string that wouldn't be completely and obviously invalid as the name of a directory.
	* If you want to check to see if a string is the name of a directory that actually exists, use {@link is_a_directory_path}.
	* @see is_a_directory_path
	* @todo add more stringent tests; currently just checks to make sure it's not empty and there are no whitespaces
	*/
	function string_like_a_directory_path($value){
		return ($this->is_a_nonempty_string($value) && $this->is_a_string_with_no_whitespace($value));
	}
	
	/**
	* True if this is a path to an existing directory.
	* If it's not required that this be the name of a directory that actually exists, use {@link is_a_string_like_a_directory_path}
	* @see is_a_string_like_a_directory_path
	*/
	function directory_path($value){
		if(!$this->is_a_string_like_a_directory_path($value)) return false;
		return file_exists($value) && is_dir($value);
	}

//////////////////////////////
// XML
//////////////////////////////	
	
	/** True if the value is a valid XML string */
	function xml_string($value){
		if(!$this->is_a_nonempty_string($value)) return false;
			
		$original_use_errors_value = libxml_use_internal_errors(true);
		$is_valid = is_a(simplexml_load_string($value, 'SimpleXMLElement', LIBXML_NOCDATA), 'SimpleXMLElement');
		libxml_use_internal_errors($original_use_errors_value);
		return $is_valid;		
	}
	
		/** True if the value is the XML content of a c32 document */
	function c32_xml_string($value){
		if(!$this->is_an_xml_string($value)) return false;
	
		$xml = new DOMDocument; 
		$xml->loadXml($value);
		
		$template_ids = $xml->getElementsByTagName('templateId');
		//loop through template ids and find which oids are present
		foreach($template_ids as $id) {
			switch($id->getAttribute('root')) {
				//C32 template
				case '2.16.840.1.113883.3.88.11.32.1':
					return TRUE;
					break;
				default:
					break;
			}
		}
		return FALSE;
	}

	/** True if the value is the XML content of a ccda document */
	function ccda_xml_string($value){
		if(!$this->is_an_xml_string($value)) return false;
	
		$xml = new DOMDocument; 
		$xml->loadXml($value);
		
		$template_ids = $xml->getElementsByTagName('templateId');
		//loop through template ids and find which oids are present
		foreach($template_ids as $id) {
			switch($id->getAttribute('root')) {
				//CCDA document template
				case '2.16.840.1.113883.10.20.22.1.2':
					return TRUE;
					break;
				//another CCDA document template
				case '2.16.840.1.113883.10.20.22.1.2.2':
					return TRUE;
					break;
				//US Realm Code template, all C-CDAs must at LEAST have this per MU2
				case '2.16.840.1.113883.10.20.22.1.1':
					return TRUE;
					break;
				default:
					break;
			}
		}
		return FALSE;
	}
	
	/** True if the value is the XML content of a CCR document */
	function ccr_xml_string($value){
		if(!$this->is_an_xml_string($value)) return false;
	
		$xml = new DOMDocument; 
		$xml->loadXml($value);

		return (isset($xml->documentElement->namespaceURI) && $xml->documentElement->namespaceURI === "urn:astm-org:CCR");
	}

	/** True if the value is the XML content of a C62 document */
	function c62_xml_string($value){
		if(!$this->is_an_xml_string($value)) return false;
	
		$xml = new DOMDocument; 
		$xml->loadXml($value);
		
		$template_ids = $xml->getElementsByTagName('templateId');
		//loop through template ids and find which oids are present
		foreach($template_ids as $id) {
			switch($id->getAttribute('root')) {
				case '2.16.840.1.113883.3.88.11.62.1':
					return TRUE;
					break;
				default:
					break;
			}
		}
		return FALSE;
	}
	
	
	
///////////////////////////////
// CODEIGNITER-RELATED METHODS
///////////////////////////////
	
	function model_class($model_class){
		if(!$this->is_a_nonempty_string($model_class) || !$this->is_a_string_with_no_whitespace($model_class)) return false;
		
		if(!$this->is_in_a_codeigniter_context()) return false;
		$CI = &get_instance();
		return $CI->load->model_exists($model_class);
	}
	
	function model($value){
		if(!$this->is_an_object($value)) return false;
		return $this->is_a_model_class(get_class($value));
	}
	
	//could (and should) make this more stringent, but this gives us a start
	function string_like_a_view($value){
		return $this->is_a_string_with_no_whitespace($value);
	}
	
	function view($value){
		if(!$this->is_a_string_like_a_view($value)) return false;
		
		if(!$this->is_in_a_codeigniter_context()) return false;
		$CI = &get_instance();
		return $CI->load->view_exists($value);	
	}
	
	function validation_rule($rule){
		if(!$this->is_a_nonempty_string($rule)) return false;	

		$possible_methods = array($rule, 'is_a_'.$rule, 'is_an_'.$rule);
		foreach($possible_methods as $method){
			if(method_exists($this, $method))
				return true;
		}
		
		foreach($this->custom_prefixes as $possible_prefixes){
			foreach($this->custom_prefixes as $possible_prefixes){	
				foreach($possible_prefixes as $prefix){
					if(string_begins_with($prefix, $rule)){ 
						if($this->is_a_validation_rule(strip_from_beginning($prefix, singular($rule)))) return true;
						if(singular($rule) !== $rule){	
							if($this->is_a_validation_rule(strip_from_beginning($prefix, $rule)))	return true;
						}
					}
				}
			}
		}
	
		return false;
	}
	
	/**#@-*/ //end phpdoc template for validation rules
	
	/**
	* Checks to see if a call to a non-existing method was a valid way of calling a validation rule, and if so, calls the correct method. 
	* Triggers a warning if there is no matching method or validation rule.
	* If you're unfamiliar with the use of {@link __call}, see {@link http://php.net/manual/en/language.oop5.overloading.php}.
	*/
	function __call($rule, $arguments){

		//if the method name was a reserved word in php, we named it the older "is_a_" format
		if($rule == 'array')
			if(method_exists($this, 'is_an_array')) return call_user_func_array(array($this, 'is_an_array'), $arguments);
		
		
		//if the method name has a '|', assume we've been passed more than one rule	
		if(mb_strpos($rule, '|') !== false){
			$validation_methods = explode('|', $rule);
			foreach($validation_methods as $method){
				$is_valid = call_user_func_array(array($this, $method), $arguments);
				if($is_valid) return true;  //once one of the rules validates, we know that the expression is valid
			}
			return false;
		}
		
		if(mb_strpos($rule, '&') !== false){
			$validation_methods = explode('&', $rule);
			$is_valid = true;
			foreach($validation_methods as $method)
				$is_valid = ($is_valid && call_user_func_array(array($this, $method), $arguments));
			return $is_valid;  //both rules must validate for the expression to be valid
		}
				
		$possible_methods = array(strip_from_beginning('is_a_', $rule), strip_from_beginning('is_an_', $rule));
		foreach($possible_methods as $method){
			if(method_exists($this, $method)) return call_user_func_array(array($this, $method), $arguments);
		}
		
		foreach($this->custom_prefixes as $intended_method => $possible_prefixes){
			foreach($possible_prefixes as $prefix){
				if(string_begins_with($prefix, $rule)){
					$arguments = array_merge(array(strip_from_beginning($prefix, $rule)), $arguments);
				 	return call_user_func_array(array($this, $intended_method), $arguments);
				}
			}
		}
				
		trigger_error('There is no method or validation rule named '.$rule, E_USER_WARNING);
		return false;
	}	
}