<?php
defined('BASEPATH') OR exit('No direct script access allowed');
if(!defined('SESSION_CACHE_ROOT')) define('SESSION_CACHE_ROOT', str_replace('\\', '/', ini_get('session.save_path').'/'));

/**
* Extension to the CI session that ensures that allows us to cache files on a per-session basis.
*
* Cached files will be written to the same location as the session, and destroyed whenever a session is destroy or cleaned up
* via garbage collection.  Note that this extension relies on a session files driver extension as well in order to ensure 
* that the cached files are actually destroyed/cleaned up.
*
* Keep in mind that codeigniter sessions lock until session_write_close() is called, and therefore sessions may need to be closed
* while dealing with large files or other scripts that take a long time to run.
*
* @see http://www.codeigniter.com/user_guide/libraries/sessions.html#a-note-about-concurrency
*
* @author M. Gibbs <gibbs_margaret@bah.com>
* @package vler
* @subpackage libraries
*/ 

/**
* @package vler
* @subpackage libraries
*/ 
class VLER_Session extends CI_Session {

	public function __construct(array $params = array()){
		parent::__construct($params);
		
		//make sure that we have the validator and the error libraries available
		$CI = get_instance();
		$CI->load->library('error');
		$CI->load->library('validator', array(), 'is');
	}

	/** 
	* Destroys the session AND removes the session cookie and unsets $_SESSION data.
	* The native session_destroy() does not remove the session data during this page load.  This is not usually what we want.
	* We don't want to actually change the behavior of the session driver, since it's implementing PHP's expected outcome, 
	* but we do want the ability to destroy things a little more thoroughly.
	*/
	function destroy_entirely(){
		$this->sess_destroy();
		
		//start code snippet from the PHP manual: http://php.net/manual/en/function.session-destroy.php
		
		// Unset all of the session variables.
		$_SESSION = array();
		
		// If it's desired to kill the session, also delete the session cookie.
		// Note: This will destroy the session, and not just the session data!
		if (ini_get("session.use_cookies")) {
			$params = session_get_cookie_params();
			setcookie(session_name(), '', time() - 42000,
				$params["path"], $params["domain"],
				$params["secure"], $params["httponly"]
			);
		}
		
		//end code snippet from PHP manual
	}

//////////////////////////////////////////
// CACHE METHODS
//////////////////////////////////////////

//the idea of the cache is to temporarily store files for the current session which will *not* be read into the $_SESSION array
//files added to this cache will be deleted when the session is destroyed or when garbage collection deletes the cache data
//they can also be manually cleared

	/**
	* The path for the cache for this session
	* @return string
	*/
	function cache_root($subpath = null){
		if(!empty($subpath) && !get_instance()->is->string_like_a_file_path($subpath)) 
			return $this->error->should_be_a_file_path($subpath);
						
		return SESSION_CACHE_ROOT.'cache_'.$this->_config['cookie_name'].session_id().'/'.$subpath;
	}
		
	/**
	* Verifies that a file is not only present, but matches the given file content.
	*
	* If you want to verify that a file exists without checking its content, use {@link path_exists_in_cache()}.
	*
	* @param string The path of the file, relative to the root cache
	* @param string The binary contents of the file
	* @return boolean
	*/
	function file_exists_in_cache($path, $binary_string){
		if(!$this->path_exists_in_cache($path)) return false;
		
		$current_binary_string = file_get_contents($this->cache_root($path));
		if($current_binary_string === FALSE){
			get_instance()->error->warning('Could not read '.$this->cache_root($path));
			return true; //assume that they're the same, since we can't determine otherwise
		}
	
		return ($current_binary_string == $binary_string);
	}

	/**
	* @param string The path of the file, relative to the root cache
	* @return boolean
	*/	
	function file_from_cache($path){
		if(!$this->path_exists_in_cache($path)) return false;
		
		$binary_string = file_get_contents($this->cache_root($path));
		if($binary_string === FALSE) get_instance()->error->warning('Could not read '.$path);
		
		return $binary_string;			
	}
	
		
	function files_from_cache($path=null){		
		if(!directory_exists($this->cache_root($path))) return array();
		$filenames = get_filenames($this->cache_root($path));
		if(!is_array($filenames)) return get_instance()->error->warning('Could not read '.$this->cache_root($path));
		
		return $filenames;
	}
	
	/**
	* @param string The path relative to the root cache
	* @return boolean
	*/
	function path_exists_in_cache($path){
		if(empty($path) || !get_instance()->is->string_like_a_file_path($path)) return get_instance()->error->should_be_a_file_path($path);
		if(!directory_exists($this->cache_root())) return false;
		return file_exists($this->cache_root($path)); //note that file_exists works for directories as well as files, because php is weird
	}	

	/**
	* Adds a file to the cache, creating any subdirectories needed along the way.
	*
	* Note that this will overwrite any file that may already exist with the same name, so you may want to make use of {@link file_exists_in_cache} before
	* using this method.
	*
	* @param string The path of the file, relative to the root cache
	* @param string The binary contents of the file
	* @return boolean
	*/
	function add_file_to_cache($path, $binary_string){
		if(empty($path) || !get_instance()->is->string_like_a_file_path($path)) return get_instance()->error->should_be_a_file_path($path);
		if(!is_string($binary_string)) return get_instance()->error->should_be_a_binary_string($binary_string);

		//if this file is already in the cache, we're done				
		if($this->file_exists_in_cache($path, $binary_string)) return true; //attachment is already in cache; this is the EXACT attachment, with the same binary string & everything
		
		$this->add_subdirectories_to_cache($path);
		
		if(!file_put_contents($this->cache_root($path), $binary_string)){
			$warning = 'Unable to write '.$path.' to the cache for session '.session_id();
			return get_instance()->error->warning($warning);
		}
			
		return $this->cache_root($path);
	}
	
	function move_uploaded_file_to_cache($name, $path=null){
		if(!is_uploaded_file($_FILES[$name]['tmp_name'])) return get_instance()->error->should_be_an_uploaded_file($name);
		if(!is_null($path)) $this->add_subdirectories_to_cache($path);

		$success = move_uploaded_file($_FILES[$name]['tmp_name'], $this->cache_root($path.basename($_FILES[$name]['name'])));
		if(!$success) return false; //move_uploaded_file would give an error if something went wrong
		
		//$success = $this->add_file_to_cache($path.basename($_FILES[$name]['name']), file_get_contents($_FILES[$name]['tmp_name']));
		if(file_exists($_FILES[$name]['tmp_name']) && !unlink($_FILES[$name]['tmp_name'])) get_instance()->error->notice('Unable to delete tmp file '.$_FILES[$name]['tmp_name']);
		return $success;
	}
	
	function remove_path_from_cache($path){
		if(!directory_exists($this->cache_root())) return true; //we're done!
		
		$path = $this->cache_root($path);
		if(!file_exists($path)) return true; //we're done!
		if(!unlink($path)) return get_instance()->error->warning('Failed to remove '.$path);
		return true;
	}
	
	
	function clear_cache($path=null){
		if(!directory_exists($this->cache_root($path))) return true; //we're done!
		delete_files($this->cache_root($path), TRUE);
		return rmdir($this->cache_root($path));
	}	

////////////////////////////////////////
//
////////////////////////////////////////


	protected function add_subdirectories_to_cache($path){
		
		//first, make sure we have a cache		
		if(!directory_exists($this->cache_root())) mkdir($this->cache_root(),0777,true); //attempt to create the cache if needed
		if(!directory_exists($this->cache_root())) return get_instance()->error->warning('Could not create the session cache root at '.$this->cache_root());
		
		//make sure that we have any subdirectories needed
		if(string_contains('/', $path)){
			$existing_path = $this->cache_root();
			$subdirectories = explode('/', substr($path, 0, strrpos($path, '/')));
			foreach($subdirectories as $subdirectory){
				$existing_path .= $subdirectory.'/';				
				
				if(directory_exists($existing_path)) continue;
				mkdir($existing_path, 0777, true); //attempt to make the subdirectory
				if(!directory_exists($existing_path)){
					$warning = 'Could not create directory '.$existing_path.' in order to add '.$path.' to the cache for session '.session_id();
					session_start();
					return get_instance()->error->warning($warning);
				}
			}
		}
		
		return true;
	}

}