<?PHP  if ( ! defined('BASEPATH')) exit('No direct script access allowed'); 
/**
* Any changes to this library need to be made in both Webmail and the API.
* @package vler
* @subpackage libraries
*/ /** */

require_library('object');
require_model('message');

/**
* @todo replace Message:: calls with something that checks for the model message class name first.
* @package vler
* @subpackage libraries
*/
class Attachment extends Object{		
	protected $name; //the actual name + extension: e.g. 'myfile.txt', without the full path
	protected $_directory; //the full path EXCLUDING the filename: i.e. /www/htdocs/, not /www/htdocs/myfile.txt.  Will always have a trailing slash.
	protected $_binary_string;
	protected $_message_id;
	
	protected $_filetype;
	protected $_extension;
	protected $_cid;
	protected $_url;
	protected $_html_friendly_name; //a version of the file name that will be appropriate for form names and css classes

	protected $_use_lightbox_preview = false;
	protected $_use_pdf_preview = false;
	
	//files inside of a zip/xdm/archive file will reference that zip/xdm/archive file as their parent
	protected $_parent; //reference to the actual zip/xdm/archive Attachment object
	protected $_path_in_parent; //if this file is within a subdirectory, this is the relative path
	protected $_index_in_parent; //the order in which this appears in the parent, so that we can reference it easily

	protected $_property_validation_rules= array( 'binary_string' => 'string',
												  'extension' => 'string_with_no_whitespace',
												  'filetype' => 'string_with_no_whitespace',
												  'name' => 'string_like_a_file_path',
												  'cid' => 'nonempty_string',
												  'url' => 'string_like_a_url',
												  'path_in_parent' => 'string_like_a_file_path',
												  'index_in_parent' => 'unsigned_integer'
												 );
												 
	function classes(){
		$classes = array_merge( array(get_class($this) => get_class($this)), class_parents($this));
		unset($classes['Object']);
		return array_map('make_string_css_friendly', $classes);
	}
	
	//describe this attachment for error messages - analogous to Entity::describe()
	function describe(){
		if(Message::formatted_like_an_id($this->message_id))
			return strtolower(humanize(get_class($this))).' for message#'.$this->message_id.' '.$this->error->describe($this->name);
		if(!$this->property_is_empty('directory'))
			return $this->error->describe($this->path());
		return strtolower(humanize(get_class($this))).' '.$this->error->describe($this->name);
	}
	
	function download(){		
		if($this->property_is_empty('name')) return $this->error->warning("Can't set up download without a name for this file.");
		header("Content-Type: application/force-download");
		header("Content-Disposition: attachment; filename=\"" . $this->name . "\"");
		header("Content-Transfer-Encoding: binary");
		echo $this->binary_string; //note - in some places in the code, we ran base-64 on this in the past.
	}

	function url_for_download(){
		//hokay.  in an ideal world, we would extend Attachment differently in DPII and DAAS, but that would interfere with all of our child attachment classes (how would they know who to inherit from?)
		//so, since URLs are the only difference in the parent class, we'll cheat and break OOP principles until we figure out some method of dynamic inheritance in class definitions.
		if(config_item('subclass_prefix') == 'DPII_'){
			$name = rawurlencode($this->name);
			if(isset($this->index_in_parent))
				$name = rawurlencode($this->parent->name).'/'.$this->index_in_parent;
			if(Message::formatted_like_an_id($this->message_id))
				return site_url('inbox/message/'.$this->message_id.'/attachments/download/'.$name);
			return site_url('inbox/compose/draft/attachments/download/'.$name);
		}else{
			$this->error->notice('Attachment::url_for_download() cannot be used until it is defined by an application extension');
		}
	}
	
	//by default, this is the same as the view url, but may be different for specific attachment types
	function url_for_preview(){
		return $this->url_for_view();
	}
	
	function url_for_view(){
		//hokay.  in an ideal world, we would extend Attachment differently in DPII and DAAS, but that would interfere with all of our child attachment classes (how would they know who to inherit from?)
		//so, since URLs are the only difference in the parent class, we'll cheat and break OOP principles until we figure out some method of dynamic inheritance in class definitions.
		if(config_item('subclass_prefix') == 'DPII_'){
			$name = rawurlencode($this->name);
			if(isset($this->index_in_parent))
				$name = rawurlencode($this->parent->name).'/'.$this->index_in_parent;			
			if(Message::formatted_like_an_id($this->message_id))
				return site_url('inbox/message/'.$this->message_id.'/attachments/'.$name);
			return site_url('inbox/compose/draft/attachments/'.$name);
		}else{
			$this->error->notice('Attachment::url_for_view() cannot be used until it is defined by an application extension');
		}
	}
		

	
	/**
	* @param string
	* @param array
	* @return string
	*/
	function link($text=null, $attributes = array()){
		if(is_null($text)) $text = $this->name;
		return link_to_url($this->_url_for_link(), $text, $this->_attributes_for_link($attributes));		
	}
		

												 
												 
////////////////////
// GETTERS
////////////////////
		
#TODO -> do we need to update this if they supply content?		
	function binary_string(){
		if(isset($this->_binary_string)) return $this->_binary_string;
		if(!empty($this->path())){
			if(!file_exists($this->path())) $this->error->warning('Unable to find file: '.$this->path());
			$this->_binary_string = file_get_contents($this->path());
			return $this->_binary_string;
		}
		return $this->error->warning('No binary string has been set');
	}
	
	function bytes(){
		if(!empty($this->path()) || isset($this->binary_string)) return string_length_in_bytes($this->binary_string);
		return 0; //file doesn't have a path or a binary string, so its current size is 0
	}
	
	function extension(){
		if(!isset($this->_extension)) $this->_extension = strtolower(pathinfo($this->name, PATHINFO_EXTENSION));
		return $this->_extension; //we may have grabbed an extension from a mime for an attached file, so we may have better info than just the filename
	}
	
	function hash(){
		if(!$this->property_is_empty('binary_string'));
		return sha1($this->binary_string);
	}
	
	function html_friendly_name(){
		return make_string_css_friendly($this->name);
	}
	
	function filetype(){
		if(!isset($this->filetype)) return $this->extension;
		$this->_filetype;
	}
	
	function filename(){
		return $this->name;
	}
	
	function metadata(){
		$properties = array_keys(get_object_vars($this));
		foreach($properties as &$property)
			$property = strip_from_beginning('_', $property);
				
		$properties_to_exclude = array('is', 'validator', 'error', 'property_validation_rules', 'binary_string', 'parent', 'parser', 'use_lightbox_preview', 'use_pdf_preview');
	
		$properties = array_merge(array_diff($properties, $properties_to_exclude), array('bytes', 'path', 'url', 'classes'));
		$metadata = array();
		$only_if_not_empty = array('message_id', 'cid', 'das_save_date', 'schema', 'description', 'url');
		foreach($properties as $property){
			$property = strip_from_beginning('_', $property);
			if(!in_array($property, $only_if_not_empty) || !$this->property_is_empty($property)){
				$metadata[$property] = $this->$property;
			}
		}
		
		if(is_a($this->parent, 'Attachment'))
			$metadata['parent'] = $this->parent->describe();
		
		return $metadata;
	}
	
	//the full path of the directory + filename; will only be supplied if both directory path and name has been populated
	function path(){
		if(isset($this->directory) && isset($this->name)){
			return $this->directory.$this->name;
		}
	}
	
	//deprecated alias for binary_string
	function string(){
		return $this->binary_string;	
	}
	
	function url(){
		return $this->url_for_view();
	}


//////////////////////
// SETTERS
//////////////////////

	//ensure that the path value always has a trailing slash && that the directory exists
	function set_directory($value){
		if(!$this->is->directory_path($value)) return $this->error->property_value_should_be_an_existing_directory_path('directory', $this, $value);
		$this->_directory = strip_from_end('/', $value).'/'; //make sure we always have a trailing slash, but only one
	}
	
	
	function set_message_id($value){
		if(!Message::formatted_like_an_id($value)) return $this->error->property_value_should_be_a_message_id('message_id', $this, $value);
		$this->_message_id = $value; //we could check to see if the message really exists, but that would be expensive (API call) - we'll trust the developers in this case
	}
	
	
	function set_parent($value){
		if(!is_a($value, 'Attachment')) return $this->error->property_value_should_be_a_message_id('parent', $this, $value);
		$this->_parent = $value;
	}
	
	
//////////////////////
// PROTECTED METHODS
//////////////////////
	/**
	* @param array
	* @return array
	*/
	protected function _attributes_for_link($attributes = array()){		
		if(!empty($this->_source_class)) { 
			$classes = array_merge($this->classes, $this->_source_class->classes);
		} 
		else { $classes = $this->classes; }
		$default_attributes = array('class' => implode(' ', $classes), 'title' => $this->name);
		
		if($this->use_lightbox_preview)
			$default_attributes = merge_attributes($default_attributes, array('class' => 'lightbox'));
		elseif($this->use_pdf_preview)
			$default_attributes = merge_attributes($default_attributes, array('class' => 'lightbox iframe'));
		return merge_attributes($default_attributes, $attributes);
	}
	
	/**
	* Returns the URL to be used by {@link}.
	*
	* By default, this is {@link url_for_preview} as long as {@link _use_lightbox_preview} or {@link _use_pdf_preview} is enabled.  Will revert 
	* to {@link url_for_download} if neither form of preview is available.  Child classes may override or extend this behavior.
	*
	* @return string
	*/
	protected function _url_for_link(){
		if($this->use_lightbox_preview || $this->use_pdf_preview)
			return $this->url_for_preview().'?'.get_instance()->security->get_csrf_token_name().'='.get_instance()->security->get_csrf_hash();
		elseif(method_exists($this, 'view'))
			return $this->url_for_view();			
		else
			return $this->url_for_download();
	}
/////////////////////
// STATIC FUNCTIONS
/////////////////////

	/**
	* Checks the content of a file and casts it as the appropriate child class.
	*
	* For example, if a C32 file is passed to this method, {@link matches_file} will be used to determine that the content of the file
	* matches a child class and the file will be cast as a {@link C32_attachment}. Note that it would be more efficient to just create 
	* a new C32_attachment object if you were sure that a file was a C32; this method should be used when you don't know how a file should be cast.
	*
	* @param string Name of the file
	* @param string Content of the file
	* @param array Any additional metadata, formatted array('message_id' => $message_id, 'directory' => $directory)
	* @return Attachment
	*/
	public static function create($name, $binary_string, $more_values = array()){
		
		foreach(array('CSV_attachment', 'Image_attachment', 'PDF_attachment', 'Word_attachment', 'XML_attachment', 'Zip_attachment', 'Text_attachment') as $child_class){
			$child_class = class_name_for_library('attachments/'.strtolower($child_class));
			if($child_class::matches_file($name, $binary_string)){
				return $child_class::create($name, $binary_string, $more_values);
			}
		}
		
		
		return new_instance_of('library', 'attachment', array(array_merge(compact('name', 'binary_string'), $more_values)));
	}

	public static function from_session_cache($path){
		$CI = get_instance();
		if(!$CI->session->path_exists_in_cache($path)) return null;
				
		$filename = $path;
		$directory = null;
			
		if(string_contains('/', $path)){		
			$filename = substr($path, strrpos($path, '/') + 1);
			$directory = strip_from_end($filename, $path);
		}

		return static::create($filename, $CI->session->file_from_cache($path), array('directory' => $CI->session->cache_root($directory)));
	}


	/**
	* True if the given file can be cast as this class.
	*
	* For example, if this is an extension of the default Attachment class for XML attachments, this method
	* will check to make sure that the content of the file is valid XML.  
	*
	* @param string Name of the file (not including path)
	* @param string Contents of the file
	* @return boolean
	*/
	public static function matches_file($name, $binary_string){
		return true; //maybe make sure that the name is a valid file name and the binary string isn't empty
	}
	
}
?>