<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package vler
* @subpackage core
/** */
require_once APPPATH.'helpers/environment_helper.php'; //this class is loaded before the loader class, can't rely on loading dependencies normally

/**
* DPII CodeIgniter Security Class Extension
*
* @package vler
* @subpackage core
* @author Adam Bell
*/
class VLER_Security extends CI_Security {
	
	public function __construct()
	{
		parent::__construct();
		//check for mcrypt, necessary to encrypt the CSRF cookie contents
		$mcrypt_exists = ( ! function_exists('mcrypt_encrypt')) ? FALSE : TRUE;
		if(!$mcrypt_exists) {
			show_error('Mcrypt extension must be loaded in order to use the extended Security class.');
		}
		//check for encryption key to perform cookie encryption
		if(config_item('encryption_key') == '') {
			show_error('In order to use the extended Security class, it is required that you set an encryption key in your config file.');
		}
	}
	
	/**
	* Overrides parent to enforce that CSRF tokens must be included in AJAX GET calls.
	* Additionally, expects the CSRF cookie values to have been encoded using {@link encode} and provides better debugging messages via customized {@link csrf_show_error}.
	* @return VLER_Security|boolean
	*/
	public function csrf_verify()
	{
		//VLER addition - require GET calls via AJAX to contain the CSRF token in their URLS
		if(IS_AJAX && mb_strtoupper($_SERVER['REQUEST_METHOD']) === 'GET' && $this->is_ajax_accessible_public_page()){
			
			//make sure we have a token in GET
			if(empty($_GET[$this->_csrf_token_name])){
				return $this->csrf_show_error('the CSRF token name ('.$this->_csrf_token_name.') is not in $_GET ');
			}
			
			//make sure we have a token in COOKIE
			if(empty($_COOKIE[$this->_csrf_cookie_name])){
				return $this->csrf_show_error('the CSRF cookie name ('.$this->_csrf_cookie_name.') is not in $_COOKIE ');
			}
		
			//make sure the COOKIE and GET tokens match
			$cookie_token = $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name]));
			if($_GET[$this->_csrf_token_name] != $cookie_token){
				return $this->csrf_show_error('the GET token and the COOKIE token do not match. ('.$_GET[$this->_csrf_token_name].' != '.$cookie_token.')');
			}
			
			return $this; //don't reset the token after verifying - async calls will probably be followed with more calls relying on the same token
		}
		
		// If it's not a POST request we will set the CSRF cookie
		if (strtoupper($_SERVER['REQUEST_METHOD']) !== 'POST'){	
			return $this->csrf_set_cookie();
		}		
		
		// Check if URI has been whitelisted from CSRF checks
		if ($exclude_uris = config_item('csrf_exclude_uris'))
		{
			$uri = load_class('URI', 'core');
			foreach ($exclude_uris as $excluded)
			{
				if (preg_match('#^'.$excluded.'$#i'.(UTF8_ENABLED ? 'u' : ''), $uri->uri_string()))
				{
					return $this;
				}
			}
		}

		// Do the tokens exist in both the _POST and _COOKIE arrays?
		if(!isset($_POST[$this->_csrf_token_name])){
			$this->csrf_show_error('the CSRF token is not in $_POST');
		}
		
		if (!isset($_COOKIE[$this->_csrf_cookie_name])){
			$this->csrf_show_error('the CSRF token is not in $_COOKIE');
		}

		// Do the tokens match?
		if ($_POST[$this->_csrf_token_name] != $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name]))){
			$this->csrf_show_error('the CSRF token in $_POST does not match the value in $_COOKIE');
		}


		// We kill this since we're done and we don't want to pollute the _POST array
		unset($_POST[$this->_csrf_token_name]);

		// Regenerate on every submission?
		if (config_item('csrf_regenerate')){
			// Nothing should last forever
			unset($_COOKIE[$this->_csrf_cookie_name]);
			$this->_csrf_hash = NULL;
		}

		$this->_csrf_set_hash();
		$this->csrf_set_cookie();

		log_message('info', 'CSRF token verified');
		return $this;
	}
	
	
	/** 
	* Overrides parent to adjust for proxy headers from nginx.
	* Also uses {@link encode} to encode CSRF cookie values
	* @return VLER_Security|boolean
	*/
	public function csrf_set_cookie(){
		$expire = time() + $this->_csrf_expire;
		$secure_cookie = (bool) config_item('cookie_secure');
		
		//we are behind nginx, so check the proxy_set_header value of HTTP_HTTPS to determine if we are using HTTPS
		if ($secure_cookie && (empty($_SERVER['HTTP_HTTPS']) OR mb_strtolower($_SERVER['HTTP_HTTPS']) === 'off'))
		{
			return FALSE;
		}

		setcookie(
			$this->_csrf_cookie_name, 
			base64_encode($this->encode($this->_csrf_hash)), 
			$expire,
			config_item('cookie_path'),
			config_item('cookie_domain'),
			$secure_cookie,
			config_item('cookie_httponly')
		);
		log_message('info', 'CRSF cookie sent');

		return $this;
	}
	
	
	/** 
	* Overrides parent to encode cookie values and to satisfy Fortify secure randomness requirements.
	* @return string
	*/
	protected function _csrf_set_hash(){
		if(empty($this->_csrf_hash)){ //CI2 checked for '', CI3 checks for NULL, I don't think there's a time when we'd have a valid hash that's empty, so ...
			// If the cookie exists we will use its value.
			// We don't necessarily want to regenerate it with each page load since a page could contain embedded sub-pages causing this feature to fail
			if (isset($_COOKIE[$this->_csrf_cookie_name]) &&
				preg_match('#^[0-9a-f]{64}$#iS', $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name]))) === 1)
			{
				return $this->_csrf_hash =  $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name]));
			}
			return $this->_csrf_hash = hash('sha256',openssl_random_pseudo_bytes(32));
		}
		return $this->_csrf_hash;
	}
	
	public function csrf_show_error($reason_why_action_is_not_allowed = ''){
		global $RTR; //this method gets run early enough that we don't have access to the controller or the standard router - grab the early global version of the router class instead
		$controller = $RTR->class;
		$action = $RTR->method;
				
		//log the message instead of displaying an error so we won't interfere w/redirect		
		$message = 'The action you have requested ('.$controller.'/'.$action.') is not allowed';
		if(!empty($reason_why_action_is_not_allowed))
			$message .= ' because '.$reason_why_action_is_not_allowed;
		log_message('error', $message); 
		
		$this->log_out();
				
		show_error('The action you have requested is not allowed', 403); //worst case scenario, our redirect didn't work - give the standard message to the user saying action is not allowed.
	}		
	
	/**
	 * Custom Mcrypt AES decryption for CSRF cookie decryption
	 *
	 * @param	string
	 * @return	string
	 */
	protected function decode($data) {
		$key = hash('sha256',('encryption_key'),true);
		$init_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_CBC);

		//note - security gets loaded so early that we can't load the string helper, but our method of measuring $data should match whatever is being used by string_length_in_bytes() -- MG
		if ($init_size > mb_strlen($data, '8bit')){ 
			return FALSE;
		}
		
		$init_vect = mb_substr($data, 0, $init_size, '8bit');
		$data = mb_substr($data, $init_size, null, '8bit');
		return mcrypt_decrypt(MCRYPT_RIJNDAEL_256, $key, $data, MCRYPT_MODE_CBC, $init_vect);
	}
	
	/**
	 * Custom Mcrypt AES encryption for CSRF cookie encryption
	 *
	 * @param	string
	 * @return	string
	 */
	protected function encode($data) {
		$key = hash('sha256',('encryption_key'),true);
		$init_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_CBC);
		$init_vect = mcrypt_create_iv($init_size, MCRYPT_RAND);
		return $init_vect.mcrypt_encrypt(MCRYPT_RIJNDAEL_256, $key, $data, MCRYPT_MODE_CBC, $init_vect);
	}
	
	protected function is_ajax_accessible_public_page(){
		trigger_error("This method should be overridden in child applications to include the log out page and any other place that AJAX calls may be redirected to if CSRF validation fails.", E_USER_NOTICE);
		return false;
	}
	
	protected function log_out(){
		session_destroy();
		trigger_error("This method should be overridden in child applications to ensure that users are logged out to an appropriate location after being logged out.", E_USER_NOTICE);
	}
}