<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models-user
*//** */

require_model('authentication_log_entry');

/**
* @package direct-as-a-service
* @subpackage models
*/
class User extends DAAS_Entity {
	static $table = 'users';
	static $primary_key = 'user_id';	
	
	protected $_admin_ldap_connection;
	protected $_ldap_connection;
	protected $_ldap_info;
	protected $_navigation_tabs;
	protected $_permissions;
	protected $_applications;
	protected $_domain;
	
	protected static $_relationships = array( 'application_request' => array('type' => 'has_many', 'foreign_key' => 'requestor', 'key_for_relationship' => 'user_id'),
											  'mailbox' => array('type' => 'belongs_to', 'related_foreign_key' => 'username', 'key_for_relationship' => 'name' ),
											  'ticket' => array('type' => 'has_many', 'key_for_relationship' => 'user_id'),
											);
	
	protected $_property_validation_rules = array( 'user_org_id' => 'whole_number',
												   'user_ext_mail' => 'string_like_an_email_address',
												   'user_ep' => 'nonempty_string' );		
	
	protected $_readonly_fields = array( 'user_created_time' ); 
	
	

////////////////////////////////
// INSTANCE METHODS
////////////////////////////////
	
	public function activate(){
		if($this->is_active) return true; //if this ist active, we're already done		
		
		$mailbox = $this->mailbox;
		$mailbox_activated = false;
		if(Mailbox::is_an_entity($mailbox) && !$mailbox->is_active){
			$mailbox->is_active = true;
			$success = $mailbox_activated = $mailbox->save();
			if(!$success) return $this->error->warning('Cannot activate '.$this->describe().'; '.$mailbox->describe.' could not be activated.');
		}
		
		$this->_ldap_info = null; //force a reload of our ldap information when we're done
		
		if(!is_resource($this->admin_ldap_connection))return $this->error->warning('Cannot activate '.$this->describe().' without an LDAP connection');			
		$success = ldap_rename($this->admin_ldap_connection, $this->disabled_dn,'uid=' . $this->username, LDAP_ACCOUNT_GROUP, TRUE);
		if(!$success){
			$this->error->warning('Unable to activate '.$this->describe().': LDAP says '.$this->error->describe(ldap_error($this->admin_ldap_connection)));
			if($mailbox_activated){
				$mailbox->is_active = false;
				if(!$mailbox->save()) $this->error->warning('Unable to deactivate '.$mailbox->describe().'; please manually update the database.');
			}
		}
		return $success;
	}	
	
	public function deactivate(){
		if(!$this->is_active) return true; //if this isn't active, we're already done		
		
		$mailbox_deactivated = false;
		$mailbox = $this->mailbox;
		if(Mailbox::is_an_entity($mailbox) && $mailbox->is_active){
			$mailbox->is_active = false;
			$success = $mailbox_deactivated = $mailbox->save();
			if(!$success) return $this->error->warning('Cannot deactivate '.$this->describe().'; '.$mailbox->describe.' could not be deactivated.');
		}
		
		$this->_ldap_info = null; //force a reload of our ldap information when we're done
		
		if(!is_resource($this->ldap_connection))return $this->error->warning('Cannot deactivate '.$this->describe().' without an LDAP connection');			
		$success = ldap_rename($this->ldap_connection, $this->dn(),'uid=' . $this->username,LDAP_DISABLED_ACCOUNT_GROUP, TRUE);
		if(!$success){
			$this->error->warning('Unable to deactivate '.$this->describe().': LDAP says '.$this->error->describe(ldap_error($this->ldap_connection)));
			if($mailbox_deactivated){
				$mailbox->is_active = true;
				if(!$mailbox->save()) $this->error->warning('Unable to reactivate '.$mailbox->describe().'; please manually update the database.');
			}
		}
		return $success;
	}	
	
	function has_access_to_mailbox($mailbox){
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->should_be_a_mailbox($mailbox);
		if(!$this->is_active || !$mailbox->is_active) return false;
		if($this->username == $mailbox->name) return true;
		
		$groups_by_name = $this->groups(array('name' => $mailbox->name), 'name');
		return array_key_exists($mailbox->name, $groups_by_name);
	}
	
	/**
	* @param array
	* @return boolean
	*/
	function has_submitted_ticket_today($conditions = array()){
		$conditions['open_date >'] = strtotime("-1 day");
		return (bool)static::count_related('tickets', $conditions);
	}
	
	function is_active(){
		return $this->active(); //currently debating the best wording, so allow both for now
	}	
	
	function is_admin(){
		if($this->property_is_empty('permissions') || !is_array($this->permissions)) return false;
		if(!array_key_exists('API', $this->permissions)) return false;
		return !empty($this->permissions['API']['admins']);
	}
	
	public function is_logged_in(){
		if(empty($_SESSION) || empty($_SESSION['user_id'])) return false;
		if($this->property_is_empty('user_org_id')) return false;

		return ($this->user_org_id == get_instance()->encrypt->decode($_SESSION['user_id']));	
	}
	
	
////////////////////////////////
// GETTERS
//////////////////////////////

	function active(){
		if($this->property_is_empty('ldap_info') || !is_object($this->ldap_info)) return false; //if we can't access LDAP, default user to inactive
		return (isset($this->ldap_info->disabled) && !$this->ldap_info->disabled); //yargh, this is why using generic objects as containers is difficult - no way to just test for empty, can't use property_exists().
	}	

	public function dn(){
		return 'uid='.$this->username.','.LDAP_ACCOUNT_GROUP;
	}
	
	public function disabled_dn(){
		return 'uid='.$this->username.','.LDAP_DISABLED_ACCOUNT_GROUP;
	}
	
	public function display_name(){
		return implode_nonempty(' ', array($this->ldap_info('givenname'), $this->ldap_info('sn')));
	}
	
	public function domain(){
		if(!isset($this->_domain)){
			$this->_domain = (array_key_exists('domain', $this->_values)) ? $this->_values['domain'] : $this->mailbox->domain;
		}
		return $this->_domain;
	}
	
	public function email_address(){		
		if(empty($this->domain())) return '';
		
		return $this->username.'@'.$this->domain();
	}

	public function groups($id_or_conditions = array(), $key_by = null){
		$group_names = array();
		foreach(array(str_replace('dc=api,', '', LDAP_GROUPS_GROUP), LDAP_DISABLED_GROUPS_GROUP) as $group_dn){			
			$ldap_entries = ldap_get_entries($this->ldap_connection(), 
											 ldap_search($this->ldap_connection(), 
														 $group_dn, 
														 '(&(member='.str_replace('dc=api,', '', $this->dn()).'))',
														 array('ou')));
														 
			foreach($ldap_entries as $entry){
				if(is_array($entry) && array_key_exists('ou', $entry)){
					$group_names[] = $entry['ou'][0];
				}
			}														
		}
		
		if(empty($group_names)) return array();
		
		Group::db()->where_in('name', $group_names);
		return Group::find($id_or_conditions, $key_by);
	}

	public function admin_ldap_connection() {
		if(!isset($this->_admin_ldap_connection)){
			$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
			if($ldap_conn === FALSE) return $this->error->warning('Could not connect to LDAP host: '.$this->error->describe(LDAP_HOSTNAME));
			
			if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
			if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
			
			//get the currently logged user's password from the database to bind to LDAP (NOT the user for this object);
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$this->_admin_ldap_connection = $ldap_conn;
		}
		return $this->_admin_ldap_connection;
	}	
	
	public function applications(){
		if(!isset($this->_applications)){
			$entries = ldap_get_entries($this->ldap_connection(), ldap_search($this->ldap_connection(), LDAP_BASE_RDN, '(&(member='.$this->dn.'))', array('uid', 'cn')));
			$applications = array();
			
			for($i = 0; $i < $entries['count']; $i++) { 
				$applications[] = $entries[$i]['cn'][0]; 
			}
			$this->_applications = $applications;
		}

		return $this->_applications;	
	}
	
	public function ldap_connection() {
		if(!isset($this->_ldap_connection)){
			$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
			if($ldap_conn === FALSE) return $this->error->warning('Could not connect to LDAP host: '.$this->error->describe(LDAP_HOSTNAME));
			
			if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
			if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
			
			//get the currently logged user's password from the database to bind to LDAP (NOT the user for this object);
			$logged_in_user = User::find_from_session();	
			if(!User::is_an_entity($logged_in_user) && in_api_context()){
				$dn = LDAP_ANON_ADMIN_USERNAME;
				$pwd = LDAP_ANON_ADMIN_PASSWORD;
			}else{
				$dn = $logged_in_user->dn;
				$pwd = get_instance()->encrypt->decode($logged_in_user->user_ep);
			}	
					
			$ldap_bind = ldap_bind($ldap_conn, $dn, $pwd);				
#			if(is_on_local()) $ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$this->_ldap_connection = $ldap_conn;
		}
		return $this->_ldap_connection;
	}	
	
	function ldap_info($property = null){
		if(!isset($this->_ldap_info) && !$this->property_is_empty('username')){
		
			$fields = array('uid','cn','givenname','initials','sn','title','departmentnumber','o','physicaldeliveryofficename','telephonenumber','mobile');
				
			foreach(array(LDAP_ACCOUNT_GROUP, LDAP_DISABLED_ACCOUNT_GROUP) as $dn){
				$entries = ldap_get_entries($this->ldap_connection, ldap_search($this->ldap_connection, $dn, '(&(uid='.$this->username.'))', $fields));
		
				$ldap_info = array();
				for($i = 0; $i < $entries['count']; $i++) {
					foreach($fields as $field){
						if(!empty($entries[$i][$field]))
							$ldap_info[$field] = $entries[$i][$field][0];
					}
				}

				
				if(empty($ldap_info)) continue;
				$ldap_info['disabled'] = ($dn == LDAP_DISABLED_ACCOUNT_GROUP) ? true : false;
				break;
			}
	
			$this->_ldap_info = (Object)$ldap_info;
		}
		
		if(!is_null($property) && is_object($this->_ldap_info))
			return $this->_ldap_info->$property;
		
		return $this->_ldap_info;
	}
	
	public function navigation_tabs(){
		if(!isset($this->_navigation_tabs)){
			$this->_navigation_tabs = get_instance()->permissions->set_tab_access_from_permissions($this->permissions);
		}
		return $this->_navigation_tabs;
	}

	public function permissions() {
		if(!isset($this->_permissions)){

			get_instance()->load->model('rolesmodel');

			if(!$this->ldap_connection) return false;
	
			//search for groups that this user is in (added !(mail=*) to filter out mail groups)		
			$search = ldap_search($this->ldap_connection, LDAP_BASE_RDN, '(&(member=' . $this->dn . ')(!(mail=*)))', array('dn'));
			$entries = ldap_get_entries($this->ldap_connection, $search);
			//set default permissions to lowest possible
			$permissions = array('API'=>array('admins'=>FALSE),'Application'=>NULL,'Registered'=>TRUE,'Role'=>array());


			foreach($entries as $entry) {
				//if they are in the API admin group, set API admin permissions
				if($entry['dn'] === LDAP_API_ADMIN_GROUP) { 
					$permissions['API']['admins'] = TRUE;
				}
				//if the value is a role
				elseif(!empty($entry['dn']) && string_contains(LDAP_ROLES_GROUP, $entry['dn'])){
					$exploded_dn_values = ldap_explode_dn($entry['dn'],1);
					$permissions['Role'][]=$exploded_dn_values[0];
				}
				//everything else will be an application group, so we will have to check the group's parent to see what app it is
				elseif(isset($entry['dn'])) {
					$exploded_dn = ldap_explode_dn($entry['dn'],0); //explode dn so we can find parent dn
					unset($exploded_dn['count'], $exploded_dn[0]); //remove values unncessary for parent dn
					$parent_dn = implode(',',$exploded_dn); //impode array to get parent dn
					
					//get parent entry and grab the uid (which corresponds to the application's unique id in the database)
					$parent_search = ldap_read($this->ldap_connection, $parent_dn, '(objectclass=*)', array('uid'));
					$parent_entry = ldap_get_entries($this->ldap_connection, $parent_search);
					
					//if the parent entry is an application, it should have a uid attribute, ignore anything else
					if(isset($parent_entry[0]['uid'])) {
						$uid = $parent_entry[0]['uid'][0];
						
						//get raw values of dn parts so we can set group name  in permissions
						$exploded_dn_values = ldap_explode_dn($entry['dn'],1); 
						$permissions['Application'][$uid][$exploded_dn_values[0]] = TRUE;
					}
				}
			}

			$permissions['Permission'] = get_instance()->rolesmodel->get_permissions_from_roles($permissions['Role']);
			$this->_permissions = $permissions;
		}
		return $this->_permissions;
	}
	
	//attempts to give the user's first name (or if added in future, nickname)
	//will revert first to cn if first name is unavailable, and username as a last resort
	public function preferred_name(){
		if(is_object($this->ldap_info)){
			if(!empty($this->ldap_info->givenname)) return $this->ldap_info->givenname;
			if(!empty($this->ldap_info->cn)) return $this->ldap_info->cn;
		}
		return $this->username;	
	}

	
	public function readonly_fields(){
		$readonly_fields = $this->_merge_with_parent_array('readonly_fields');
		if(isset($this->id)){
			$readonly_fields[] = 'username';
			$readonly_fields[] = 'user_org_id';
		}
		
		return $readonly_fields;
    }
	
	public function values_for_api(){
		$values = array('id' => $this->id,
						'username' => $this->username,
						'mailbox' => $this->username);
		
		$fields_to_ldap_attributes = array(  'first_name' => 'givenname',
											  'middle_name' => 'initials',
											  'last_name' => 'sn',
											  'job_title' => 'title', 
											  'department' => 'departmentnumber',
											  'organization' => 'o',
											  'telephone' => 'telephonenumber',
											  'mobile' => 'mobile',
											  'location' => 'physicaldeliveryofficename' );
		
		foreach($fields_to_ldap_attributes  as $field_name => $ldap_attribute){
			$values[$field_name] = (isset($this->ldap_info->$ldap_attribute)) ? $this->ldap_info->$ldap_attribute : '';
		}
		
		$values['external_email'] = $this->user_ext_mail;
		$values['organization_id'] = $this->user_org_id;
		$values['is_active'] = $this->is_active();
		$values['facility_id'] = get_instance()->usersettingsmodel->get_facility_id_in_mailbox($this->username, false);
		return $values;
	}
	
//////////////////////////////////////
// SETTERS
//////////////////////////////////////	

	public function set_username($value, $offset = 0){
		if(isset($this->id)) return $this->error->warning('Cannot change the username for '.$this->describe().', which has already been saved to the database.', $offset+1);
		if(!$this->is->nonempty_string($value)) return $this->error->property_value_should_be_a_nonempty_string('username', get_class($this), $value, $offset+1);
		if(!$this->is->string_like_a_direct_address($value.'@'.CLINICAL_DOMAIN)) return $this->error->property_value_should_be_a_valid_first_half_of_an_email_address('username', get_class($this), $value, $offset+1);
		#if(User::exists(array('username' => $value ))) return $this->error->warning('Cannot set '.get_class($this).'::$username to '.$this->error->describe($value).'; this username is already in use', $offset+1);
		//check against all mailboxes, including groups
		if(Mailbox::exists(array('name' => $value ))) return $this->error->warning('Cannot set '.get_class($this).'::$username to '.$this->error->describe($value).'; this username is already in use', $offset+1);
		
		return $this->_set_field_value('username', $value, $offset+1);
	}

	
//////////////////////////////////////
// DATA MANAGEMENT
//////////////////////////////////////

	protected function _run_before_create(){
		$success = $this->_set_field_value('user_created_time', time(), $error_offset = 0, $override_validation = TRUE);
		return $success;
	}
	
	protected function _values_are_valid_for_create(){
		if($this->property_is_empty('username')) return $this->error->warning(get_class($this).'::$username is a required field, and must be set before saving');
		if(!Mailbox::name_is_available($this->username)) return $this->error->warning('The mailbox name '.$this->error->describe($this->username).' is not available');
		return true;
	}
	
	protected function _values_are_valid_for_create_and_update(){
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		$required_fields = array( 'username', 'user_org_id', 'user_created_time', 'user_ep' );
		foreach($required_fields as $required_field){
			if($this->property_is_empty($required_field)) 
				return $this->error->warning(get_class($this).'::$'.$required_field.' is a required field, and must be set before saving'); 
		}
		return true;
	}	

#TODO - ADD LDAP CODE HERE INSTEAD OF NEEDING TO RUN USERSMODEL::CREATE_USER
	protected function _run_after_create(){
		$mailbox = Mailbox::create( array( 'name' => $this->username, 'is_active' => true ) );
		if(!Mailbox::is_an_entity($mailbox)){
			if(User::delete($this->id))
				return $this->error->warning('Unable to create a mailbox for '.$this->describe().'; removing user entry from the database.');
			return $this->error->warning('Unable to create a mailbox for '.$this->describe().', but the user remains in the database.  Please manually remove '.$this->describe().' from the database');
		}
		
		return true;
	} 
	
	protected static function _run_before_delete(&$user){ 
		$mailbox = $user->mailbox;
		if(Mailbox::is_an_entity($mailbox)){
			if(!Mailbox::delete($mailbox->id)) return $this->error->warning('Could not delete '.$mailbox->describe());
		}
		
		return true; 
	}
	

/////////////////////////////////
// STATIC METHODS
/////////////////////////////////

	public static function find_from_session(){
		$CI = get_instance();
		
		if(in_api_context() && isset($CI->mailbox)){
			return $CI->mailbox->user;
		}
		
		$user_org_id = static::organization_id_from_session();
		if(empty($user_org_id)) return false; //it's OK for the user to not be in the session - don't want to trigger errors if we just need to have the user log in 
		if(!$CI->is->nonzero_unsigned_integer($user_org_id)) return $CI->error->should_be_a_user_organization_id($user_org_id); //we do actually want to know if there's some weird invalid ID in the system
		
		if(isset($CI->user) && User::is_an_entity($CI->user) && $CI->user->user_org_id == $user_org_id){
			return $CI->user;
		}
		
		$user = User::find_one( compact('user_org_id') );
# don't give an error - unregistered users will be in the session but not yet in the system.  ultimately, it would be nice to have a different way of doing this, but this will do for now.
#		if(!User::is_an_entity($user)) return $CI->error->warning('No user with an organization ID of '.$CI->error->describe($user_org_id).' was found in the system.'); 
		if(!isset($CI->user)) $CI->user = $user; //make sure we're not overwriting something else in one of these controllers
		return $user;
	}
	
	//note that this does *not* require that the user actually have an active account in order to return true,
	//just that we were able to identify the user's org id and set up their informatio in the session.
	public static function log_in(){
		$CI = get_instance();
		
		//if needed, load the authentication library
		if(!$CI->load->is_loaded('authentication')){
			if (USE_PIV_AUTH === TRUE)
				$CI->load->library('authentication/PIV_authentication', array(), 'authentication');
			elseif(USE_CAC_AUTH === TRUE)
				$CI->load->library('authentication/CAC_authentication', array(), 'authentication');
			else
				return $CI->error->warning('No form of authentication is currently configured.');		
		}
		
		//make sure the session still reflects the logged in user
		$organization_id = User::organization_id_from_session();
		if(!empty($organization_id) && $organization_id != $CI->authentication->organization_id){
			User::log_out(); //log out, but continue on and allow the new person to log in if possible
		}
		
		//make sure that we have an organization id before proceeding
		$organization_id = $CI->authentication->organization_id;
		if(empty($organization_id)){
			Authentication_log_entry::login_failure('Error: Organization ID is not numeric');
			User::log_out();
			return false;
		}
			
		//find the user, if they already exist in our database
		$user = User::find_from_session();
		if(!User::is_an_entity($user))
			$user = $CI->authentication->user;
			
		//if the user is already logged in, we're done
		if(User::is_an_entity($user) && $user->is_logged_in()) 
			return true;
		if(array_key_exists('user_id', $_SESSION) && $_SESSION['user_id'] == $CI->encrypt->encode($organization_id))
			return true;
			
		//todo - add middle name, email, uid to user info that we're collecting
		
		//set up the user's information in the session, from the best information that we have for them
		$message_for_log = 'Success';
		if(User::is_an_entity($user) && $user->is_active){
			$CI->user = $user;
			$user_info = array('first_name' => $user->ldap_info->givenname, 'last_name' => $user->ldap_info->sn);	
		}else{
			$account_request = Account_request::find_from_session();
			$message_for_log = 'Unregistered';
			if(Account_request::is_an_entity($account_request)){
				$user_info = array('first_name' => $account_request->first_name, 'last_name' => $account_request->last_name);
			}else{
				$user_info = array('first_name' => ucfirst(mb_strtolower($CI->authentication->given_name)), 'last_name' => ucfirst(mb_strtolower($CI->authentication->family_name)));
			}
		}
			
		$_SESSION['user_info'] = $user_info;
		$_SESSION['user_id'] = $CI->encrypt->encode($organization_id);
		Authentication_log_entry::login_success($message_for_log);
		
		return true;
	}		
	
	public static function log_out(){
		$CI = get_instance();
		
		//destroy saved session data, including saved information about the user
		$CI->session->destroy_entirely(); 
		
		//unset the saved user variable from the controller
		if(isset($CI->user)) unset($CI->user);
		
		//restart the session in case we're doing something besides redirecting after logging out
		session_start(); //todo - should this be loading the session library again? the loader's going to assume it's already loaded and fail to re-load it
	}
	
	public static function new_username($first, $last) {
		//sanitize username input to remove any non-alphanumeric characters
		$first = preg_replace("/[^\p{L}\p{M}\p{N}]/u", '', $first);
		$last = preg_replace("/[^\p{L}\p{M}\p{N}]/u", '', $last);
		
		//theoretically we could hit max username lengths for last names, so don't allow more than 100 characters total
		$last = mb_substr($last,0,99);
		
		if(empty($first) && empty($last)) return FALSE; //we can't do anything without a first and last name
			
		$username = mb_strtolower($first . '.' . $last);
		if(!User::exists(compact('username'))) return $username;
	
		for($i=2; $i <= 999; $i++){
			if(!User::exists(array('username' => $username.$i))) return  $username.$i;
		}
		
		return FALSE; //returns FALSE if more than 999 users have the same combo of first initial / last name
	}	
	
	public static function organization_id_from_session(){
		$CI = get_instance();
		if(array_key_exists('user_id', $_SESSION))
			return $CI->encrypt->decode($_SESSION['user_id']);
		if(isset($CI->authentication))
			return $CI->authentication->organization_id;
	}
	
	public static function where_user_is_active(){
		$user = User::find_from_session();
		if(!User::is_an_entity($user))
			$user = User::find_one(); //hack to get the admin ldap connection - we should have some kind of ldap library that can supply the connection
			
		$entries = ldap_get_entries($user->admin_ldap_connection, ldap_search($user->admin_ldap_connection, LDAP_ACCOUNT_GROUP, '(&(uid=*))', array('uid')));

		$usernames = array();
		for($i = 0; $i < $entries['count']; $i++) {
			$usernames[] = $entries[$i]['uid'][0];
		}
		
		if(empty($usernames))
			User::db()->where('0=1', NULL, FALSE); //TODO: refactor this in a better way and not with this manual query.  query is written purposely to always have no results 
		else
			User::db()->where_in('username', $usernames);
		
		return User::db();	
	}
	
	public static function where_user_is_disabled(){
		$user = User::find_from_session();
		if(!User::is_an_entity($user))
			$user = User::find_one(); //hack to get the admin ldap connection - we should have some kind of ldap library that can supply the connection
			
		$entries = ldap_get_entries($user->admin_ldap_connection, ldap_search($user->admin_ldap_connection, LDAP_DISABLED_ACCOUNT_GROUP, '(&(uid=*))', array('uid')));
		
		$usernames = array();
		for($i = 0; $i < $entries['count']; $i++) {
			$usernames[] = $entries[$i]['uid'][0];
		}	
		
		if(empty($usernames))
			User::db()->where('0=1', NULL, FALSE); //TODO: refactor this in a better way and not with this manual query.  query is written purposely to always have no results 
		else
			User::db()->where_in('username', $usernames);
		
		return User::db();	
	}
	
}
