<?PHP
if (! defined ( 'BASEPATH' )) exit ( 'No direct script access allowed' );

/**
* @package direct-as-a-service
* @subpackage models
*/

/** */
require_model('entity');

define('FILE_TRANSFER_UPLOAD_CHUNK_SIZE', bytes_from_shorthand('32M')); //we save large file transfers to the db in pieces to improve performance - how big should they be?

/**
 * @package direct-as-a-service
 * @subpackage models
 */
class File_transfer extends DAAS_Entity {
	static $database_group = 'mail_db';
	static $table = 'file_transfers';
	
	protected $_content_filepath;

	protected $original_save_queries_value; //we're going to turn off save queries during insert statements so they don't get cached - note that this means they won't show up on the profiler!  
	
	protected $notification_message_id; //note - this is ONLY used during create, don't try to grab this info at other points!  (there's got to be a more graceful way to do this)

	protected $_property_validation_rules = array('created_by' => 'nonzero_unsigned_integer',
												  'downloaded_by' => 'nonzero_unsigned_integer',
												  'downloaded_by_ip' => 'string_like_an_ip_address',
												  'modified_by' => 'nonzero_unsigned_integer',
												  'name' => 'nonempty_string',
												  'recipient' => 'null|nonempty_string|string_like_an_email_address');

	protected static $_relationships = array('downloader' => array('type' => 'belongs_to', 'model' => 'user', 'key_for_relationship' => 'user_id', 'related_foreign_key' => 'downloaded_by'),
											 'message' => array('type' => 'belongs_to'),
											 'sender' => array('type' => 'belongs_to', 'model' => 'user', 'key_for_relationship' => 'user_id', 'related_foreign_key' => 'created_by'),
											);

	protected $_readonly_fields = array('size', 'created_at', 'modified_by', 'modified_at');

	//very large file transfers cause memory allocation errors if we try to download them by loading their string into memory first
	//instead, we'll pull them in pieces out of the db - defaulting to 64MB, could experiment with size to improve performance
	//might want to consider using the actual mime type in future
	public function download(){
		header("Content-Type: application/force-download");
		header("Content-Disposition: attachment; filename=\"" . $this->name . "\"");
		header("Content-Transfer-Encoding: binary"); 
		
		$piece_size = bytes_from_shorthand('64M'); //echo out in 64MB chunks, since the system gets cranky about echo
		
		if($this->size < $piece_size)
			$piece_size = $this->size;
		
		if(!$this->property_is_empty('content')){
			for($i = 0; $i < ceil($this->size/$piece_size) + 1; $i++){
				echo substr($this->content, $i*$piece_size, $piece_size);
			}
		}else{
			for($i = 0; $i < ceil($this->size/$piece_size) + 1; $i++){
				$result = static::db()->query('SELECT SUBSTRING(content, '.$i * $piece_size.', '.$piece_size.') AS piece FROM '.static::$table.' WHERE '.static::$primary_key.'='.$this->id)->unbuffered_row();
				echo $result->piece;
			}
		}
	}


	public function has_expired(){
		return ($this->expires_at < now() || $this->downloaded_at > 1);
	}

	public function mark_as_downloaded(){
		$this->downloaded_at = now();
		$this->downloaded_by_ip = get_instance()->input->ip_address();
		$recipient_name = $this->recipient;
		
		//ultimately, this should be some sort of no-replies address, but we currently don't have a trusted one
		//in the meantime, default to the sender's mailbox in case our recipient is not a VA user
		$mailbox_for_notification = $this->sender->mailbox;

		//if we have a logged in user, record that they were the one who downloaded this file
		$user = User::find_from_session();
		if(User::is_an_entity($user)){
			$this->downloaded_by = $user->id;
			$recipient_name = $user->display_name();
			$mailbox_for_notification = $user->mailbox;
		}

		if($this->notify_on_download){
			$message_values = array('to' => $this->sender->email_address(),
									'subject' => $recipient_name.' has downloaded the file you sent',
									'body' => get_instance()->load->view('api/file_transfer/emails/download_notification', array('file_transfer' => $this, 'recipient_name' => $recipient_name), TRUE),
									);

			$message = $mailbox_for_notification->create_message($message_values);
			$message->send();
		}

		if($this->save()){
			static::clear_expired_content();
			return true;
		}
		return false;
	}

	//file transfers uploaded on api admin are simple - they only have one recipient
	//when we create file transfers in webmail, things are more complicated.  In that case, we need to create as many records as there are recipients.
	//to keep life simple, we don't do that until the message is sent (and the recipient list is therefore finalized
	//note that this method is really only intended to be called from within Message->send()
	public function populate_recipients($recipients){
		if(!$this->is->array_of_nonempty_string($recipients)) return $this->error->warning->should_be_an_array_of_email_addresses($recipients);
		if(empty($recipients)) return $this->error->warning->should_be_a_nonempty_array_of_email_addresses($recipients);
		
		//the first recipient can just be saved to this file transfer
		$this->recipient = first_element($recipients);
		$this->save();
		array_shift($recipients);
		
		//any remaining recipients will need to have a clone of this file transfer
		foreach($recipients as $recipient){
			$file_transfer = new File_transfer($this->writeable_values());
			$file_transfer->recipient = $recipient;
			$file_transfer->save();
		}
	}	
	
	public function url_for_view($force_file_transfer_domain=false){
		$path = 'message/'.$this->message_id.'/file_transfer/'.$this->hash;
		if($force_file_transfer_domain || in_api_context())
			return file_transfer_url($path);
		return site_url($path);
	}

///////////////////////////////////////
// GETTERS
///////////////////////////////////////

	public function disclosures(){
		return $this->message->attachment_disclosures(array('hash' => $this->hash));
	}
	
	public function expires_at(){		
		if(!$this->property_is_empty('created_at'))
			return $this->created_at + (FILE_TRANSFER_EXPIRATION_IN_DAYS * 60 * 60 * 24);
	}
	
	public function mailbox(){
		if(isset($this->message_id)){
			Mailbox::db()->select('mailboxes.*');
			Mailbox::db()->join('messages', 'mailboxes.id = messages.mailbox_id');
			return Mailbox::find_one(array('messages.id' => $this->message_id));
		}
		return $this->sender->mailbox;
	}
	
	public function recipient_display_name(){
		if(!$this->property_is_empty('recipient')){
			if($this->is->string_like_a_direct_address($this->recipient)){
				$mailbox = Mailbox::find_by_email_address($this->recipient);
				return $mailbox->display_name;
			}
			return $this->recipient;
		}
	}

///////////////////////////////////////
// SETTERS
///////////////////////////////////////

	protected function set_message_id($value, $offset=0){
		if(!Message::formatted_like_an_id($value))
			return $this->error->property_value_should_be_a_message_id('message_id', get_class($this), $value, $offset+1);
		$message = Message::find_one($value);
		if(!Message::is_an_entity($message) || !$message->draft)
			return $this->error->property_value_should_be_a_draft_message_id('message_id', get_class($this), $value, $offset+1);	

		$this->_set_field_value('message_id', $value, 0, TRUE);
		return true;
	}
	
	protected function set_content_filepath($value, $offset=0){
		if(!$this->is->file_path($value)) return $this->error->property_value_should_be_a_nonempty_string('content', get_class($this), $value, $offset+1);
		$this->_content_filepath = $value;
		$this->_set_field_value('size', filesize($value), 0, TRUE);
		$this->_set_field_value('hash', sha1($this->size.$value), 0, TRUE); //the file itself may be too large to do a sha1 without running out of memory - this will still create an identifier
	}
	
	protected function set_content(&$value, $offset=0){
		if(!$this->is->nonempty_string($value)) 
			return $this->error->property_value_should_be_a_nonempty_string('content', get_class($this), $value, $offset+1);
		$this->_set_field_value_by_reference('content', $value, 0, TRUE);
		$this->_set_field_value('size', string_length_in_bytes($value), 0, TRUE);
		$this->_set_field_value('hash', sha1($value), 0, TRUE);
	}

////////////////////////////////
// DATA MANAGEMENT
///////////////////////////////

	protected function _values_are_valid_for_create_and_update(){ 
		if($this->property_is_empty('size')) 
			return $this->error->warning('Cannot save '.$this->describe().' with a size of '.$this->size().'. File transfers must have content or be deleted.');
		return true; 
	}

	protected function _run_before_create(){
		$user = User::find_from_session(); 

		if(!isset($this->content) && !isset($this->content_filepath))
			return $this->error->warning('Cannot create file transfer without content or content file path specified');

		if(!isset($this->created_by)) $this->_set_field_value('created_by', $user->id(), 0, TRUE);
		if(!isset($this->created_at)) $this->_set_field_value('created_at', now(), 0, TRUE);
		if(!isset($this->passcode)) $this->_set_field_value('passcode', static::random_passcode(), TRUE);

		//some file transfers will be treated as though they were attachments to normal messages - these will have a message id populated before we get here
		//for the rest of the file transfers, create a notification for them so that people know that they're there
		if(!isset($this->message_id)){
			$message_values = array('to' => $this->recipient,
									'subject' => $user->display_name().' has sent you a link to download a file');

			$message = $this->mailbox()->create_message($message_values);

			if(!Message::is_an_entity($message))
				return $this->error->warning('Could not create a message for '.$this->describe());

			$this->notification_message_id = $message->id;

			if(!isset($this->message_id)) $this->_set_field_value('message_id', $message->id(), 0, TRUE);
		}
		
		$this->original_save_queries_value = static::db()->save_queries;
		static::db()->save_queries = false; //turn this off during insert so that our giant INSERT statement never gets cached.  note that this means that it will never show up in the profiler.

		return true;
	}

	protected function _run_before_create_and_update(){
		$user = User::find_from_session();
		if(!isset($this->modified_by)) $this->_set_field_value('modified_by', $user->id(), 0, TRUE);
		if(!isset($this->modified_at)) $this->_set_field_value('modified_at', now(), 0, TRUE);

		return true;
	}

	protected function _run_after_create(){
		static::db()->save_queries = $this->original_save_queries_value; //now that we're done with our giant insert statement, set save_queries back to whatever it was to begin with	
		
		if(isset($this->notification_message_id)){
			$user = User::find_from_session();
			$message = $this->message;
			if($message->id != $this->notification_message_id){
				$this->error->warning('Notification message id does not match message_id for '.$this->describe().'; notification message will not be sent');
				return true;
			}
			$message->body = get_instance()->load->view('api/file_transfer/emails/transfer_notification', array('file_transfer' => $this, 'sender' => $user), TRUE);
			if(!$message->save() || !$this->message->send())
				return $this->error->warning('An error occurred and the notification email for '.$this->describe().' could not be sent');
		}
		
		return true;
	}
	
	protected function _run_insert_statement(){
		$id = parent::_run_insert_statement();
		if(isset($this->content_filepath) && is_file($this->content_filepath)){
			
			$i = 1;
			$chunk_size = FILE_TRANSFER_UPLOAD_CHUNK_SIZE;  //note that base64_encode increases the size to about 85MB when this is 64MB
			$file_size = filesize($this->content_filepath);
			$number_of_chunks = ceil($file_size/$chunk_size);			
			
			if(is_on_local()) log_message('error', 'Starting content update for file_transfer#'.$id.': importing '.$this->content_filepath);
			$success = $handle = fopen($this->content_filepath, 'rb');
			if($success){
				while (!feof($handle)) {
					$contents = fread($handle, $chunk_size);
					if(!$contents) break;
			
					//note - ideally we would just insert the contents of the db into the varbinary without needing to encode it, but we run into problems with the db trying to parse it as a unix strin gthat way
					//originally used bin2hex to encode it, but that doubled the length of the string; base64_encode only increases it by about 1/3
					//also, seems to be faster to use simple_query instead of update(), so going with that since we don't have user-generated content that could pose a threat
					//todo - try setting varbinary data as a SQL variable and inserting, try doing parameterized queries from query()	
					$sql_start = 'UPDATE '.static::$table.' SET upload_progress='.(100 * (min($file_size, $chunk_size * $i)/$file_size)).' , content=';
					$sql_end = "\")', 'varbinary(max)') WHERE ".static::$primary_key.'='.$id;
					
					if($i < 2)
						$success = $success && static::db()->query($sql_start."cast(N'' as xml).value('xs:base64Binary(\"".base64_encode($contents).$sql_end);
					else
						$success =  $success && static::db()->query($sql_start."content + cast(N'' as xml).value('xs:base64Binary(\"".base64_encode($contents).$sql_end);
					if(!$success) break;
					if(is_on_local()) log_message('error', 'file_transfer#'.$id.' - '.$i.' - '.byte_format(min($file_size, $chunk_size * $i)).' of '.byte_format($file_size)).' saved... ';
					$i++;
				}
				fclose($handle);
			}
			
			if(!$success){
				$this->error->warning('Unable to save content from '.$this->content_filepath.' for '.$this->describe().'; deleting entry.');
				static::delete($id);
				return false;
			}		
		}
		
		return $id;
	}
	

	protected function _set_field_value_by_reference($field, &$value, $error_offset = 0, $override_validation = FALSE){
		if(!$override_validation){
			if(!$this->field_is_writeable($field)) return $this->error->should_be_a_writeable_property($field, $error_offset+1);
			if($this->property_has_validation($field, $value) && !$this->value_is_valid_for_property($field, $value)){
				$field_should_be = 'property_value_should_be_a_'.str_replace('|', '_or_', element($field, $this->property_validation_rules()));
        	    return $this->error->$field_should_be($field, get_class($this), $value, $error_offset+1);
			}
		}
		
		$this->_values[$field] = &$value;
		$this->_altered_fields[] = $field;
		return true;
	}
	
	public function load_field_values_from_db(){
		//since content is expensive to load, we'll take a risk and assume it was saved correctly without us reloading it
		//people who are debugging any issues with this model may want to comment out the line below
		static::select_all_except_content(); 
		parent::load_field_values_from_db();
	}	
	
		
///////////////////////////////
// STATIC
//////////////////////////////

	//leave the metadata for the entries, but delete the actual file content after the transfer has expired
	public static function clear_expired_content($include_expired_messages=false){ 
		require_model('message'); //since this sometimes gets called before autoloader, we need to explicitly call the dependencies
		
		//if the file transfer has been downloaded, the content can't be downloaded again and we can delete the content
		static::db()->update(static::$table, array('content' => NULL), 'content IS NOT NULL AND downloaded_at IS NOT NULL');

		//if the message for this file transfer was sent more than FILE_TRANSFER_EXPIRATION_IN_DAYS days ago, we can clear out the data
		//this query is SLOW and requires about 30s to run, so we'll only run it on garbage collection, not when people are trying to download files
		if($include_expired_messages){
			$query = 'UPDATE '.static::$table.' SET content=NULL'."\n";
			$query .= ' FROM '.static::$table.' INNER JOIN '.Message::$table.' ON '.Message::$table.'.'.Message::$primary_key.'='.static::related_foreign_key('message', TRUE)."\n";
			$query .= ' 	  AND file_transfers.content IS NOT NULL AND messages.sent=1 AND (messages.timestamp + '.(FILE_TRANSFER_EXPIRATION_IN_DAYS * 24 * 24 * 60 ).') < '.now(); 
	
			static::db()->query($query); 
		}
	}
	
	//by default, we really really don't want to pull the content field - it's huge! it would be so much better as a filestream!  a curse upon our requirement for encrypted content
	public static function default_fields(){
		//wonder of wonders, miracle of miracles, the query that CI runs for fields() doesn't interfere with the query builder.  
		//running db->where($conditions); static::fields(); db->get($table);  will apply the where() to the ->get() and not to the query run by fields()
		//also, the value of fields is cached, so you'll only have the query run once per page load
		//in some sort of idealistic future world, it would be lovely if we, say, ran a make file as part of our build process that stored these instead of generating them on the fly
		$fields = static::fields();
		unset($fields[array_search('content', $fields)]);
		foreach($fields as $key => $field) {
			$fields[$key] = static::$table.'.'.$field;
		}
		return $fields;
	}

	public static function count_available_for_current_user($id_or_conditions = array()){
		$user = User::find_from_session();
		if(!User::is_an_entity($user)) return null;
		static::db()->where(array('file_transfers.recipient' => $user->email_address()));
 		static::where_download_is_available();
		return static::count($id_or_conditions);
	}

	public static function count_received_for_current_user($id_or_conditions = array()){
		$user = User::find_from_session();
		if(!User::is_an_entity($user)) return null;
		static::db()->where(array('recipient' => $user->email_address()));
		static::where_message_has_been_sent();
		return static::count($id_or_conditions);
	}

	public static function count_sent_for_current_user($id_or_conditions = array()){
		$user = User::find_from_session();
		if(!User::is_an_entity($user)) return null;
		static::db()->where(array('created_by' => $user->id));
		static::where_message_has_been_sent();
		return static::count($id_or_conditions);
	}
	
	public static function find($id_or_conditions = array(), $key_by = null){
		//unless the user has specifically said that they want the content field, don't load it!  it's giant and a super expensive operation
		if(!static::db()->select_is_defined())
			static::select_all_except_content();
		return parent::find($id_or_conditions, $key_by);
	}

	public static function find_available_for_current_user($id_or_conditions = array(), $key_by = null){
		$user = User::find_from_session();
		if(!User::is_an_entity($user)) return null;
		static::db()->where(array('recipient' => $user->email_address()));
		static::select_all_except_content();
		static::where_download_is_available();
		return static::find($id_or_conditions, $key_by);
	}

	public static function find_received_for_current_user($id_or_conditions = array(), $key_by = null){
		$user = User::find_from_session();
		if(!User::is_an_entity($user)) return null;
		static::db()->where(array('recipient' => $user->email_address()));
		static::select_all_except_content();
		static::where_message_has_been_sent();
		return static::find($id_or_conditions, $key_by);
	}

	public static function find_sent_for_current_user($id_or_conditions = array(), $key_by = null){
		$user = User::find_from_session();
		if(!User::is_an_entity($user)) return null;
		static::db()->where(array('created_by' => $user->id));
		static::select_all_except_content();
		static::where_message_has_been_sent();
		return static::find($id_or_conditions, $key_by);
	}

	public static function select_all_except_content(){
		static::db()->select(static::default_fields());
	}

	public static function where_message_has_been_sent(){
		//there should always be a message id for a file transfer, so we shouldn't actually need an outer join here instead of an inner join
		//but I'm leaving it in place in case any file transfers get orphaned -- MG 2016-08-08
		static::db()->join('messages', 'messages.id = file_transfers.message_id', 'left');
		static::db()->where('(messages.draft = 0 OR messages.draft IS NULL)');
	}

	public static function where_download_is_available(){
		static::where_message_has_been_sent();
		static::db()->where(array('messages.timestamp + ('.FILE_TRANSFER_EXPIRATION_IN_DAYS.' * 24 * 24 * 60) >' => now(), 'file_transfers.downloaded_at' => NULL));
	}
	
	protected static function random_passcode(){
		$passcode = '';
		$characters = array_merge(range('A','Z'), range('a','z'), range('0','9'), array('!', '#', '$', '%', '&', '(', ')', '*', '+', '-', '.', ':', ';', '=', '?', '@', ']', '^', '_', '{', '|', '}', '~'));
		$max = count($characters) - 1;
		for ($i = 0; $i < 15; $i++) {
			$rand = mt_rand(0, $max);
			$passcode .= $characters[$rand];
		}
		return $passcode;
	}
}