<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
*//** */
 
require_once APPPATH.'controllers/services/mailbox_controller.php';

/**
* API methods to manage a mailbox's messages.
*
* Inherits mailbox validation from the mailbox controller.  Mailbox name is a required field for all actions on this controller, with the exception of messages to send or save a messagage; these actions require the sender's email address to be provided.
*
* @author Adam Bell <bell_adam@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
* @author Elan Jaffee <jaffee_elan@bah.com>
*
* @package direct-as-a-service
* @subpackage controllers
*/
class Message_controller extends Mailbox_controller{	
	
	/**
	* Everything that applies to all actions should go here.
	*/
	function __construct(){
		parent::__construct();
		$this->load->helper(array('mail', 'validate'));
		require_library('attachment');
	}	
				
	/**
	* Archives a message.
	* 
	* This action is not available for inactive mailboxes.
	*
	* @param string mailbox name
	* @param int id of the message
	*/
	public function archive_message_post(){
		//set up the required fields
		$required_fields = array('id');
		$post = $this->post();
        foreach($required_fields as $required_field){
			if(array_key_exists($required_field, $post))
				$$required_field = $this->post($required_field,TRUE);
			else{
				array_push($this->missing_required_fields, $required_field);
			}
        }
		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');
		$this->respond_with_error_if_mailbox_is_inactive();

		if(!Message::formatted_like_an_id($id)){
			$this->invalid_fields[] = 'id';
		}
		$this->respond_with_error_if_fields_are_invalid();	
		
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox)){
			$this->response('Message not found.', 422);
		}
		$this->response_message['id'] = $id;
		$message->archive();
		$this->response($this->response_message, 200);
	}
	
	/**
	* Sets the custom flag of the message specified.
	* 
	* This action is not available for inactive mailboxes.
	*	
	* @todo - We may want to add funcitonality to archive messages here, since we treat that as moving them to the archive folder
	*
	* @param string mailbox name
	* @param int id of the message
	* @param int|string ID of the folder - this will be an integer for custom folders, a string for system folders (sent|draft|inbox) 
	*/	
	public function move_message_post(){
		$required_fields = array('id', 'folder');
		$post = $this->post();
        foreach($required_fields as $required_field){
        if(array_key_exists($required_field, $post))
            $$required_field = $this->post($required_field,TRUE);
        else{
        	array_push($this->missing_required_fields, $required_field);
           }
        }
		
		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');
		$this->respond_with_error_if_mailbox_is_inactive();

		if(!Message::formatted_like_an_id($id)){
			array_push($this->invalid_fields, 'id');
		}
		if($folder !== 'inbox' && $folder !== 'sent' && $folder !== 'draft' && !(Folder::formatted_like_an_id($folder) && $this->mailbox->has_related('folder', array('id' =>$folder)))) {
			array_push($this->invalid_fields, 'folder');
		}
		$this->respond_with_error_if_fields_are_invalid();
		
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox)){
			$this->response('Message not found.', 422);
		}
				
		if($message->sent){
			if($folder != 'sent') $this->response('Message with ID: '.$message->id.' was originally a sent message and cannot be moved to '.$folder, 422);
			$success = $message->restore() && !$message->archived;
		}elseif($message->draft){
			if($folder != 'draft') $this->response('Message with ID: '.$message->id.' was originally a draft message and cannot be moved to '.$folder, 422);
			$success = $message->restore() && !$message->archived;
		}elseif($folder == 'inbox'){
			$success = $message->move_to_inbox() && $message->is_in_inbox();
		}elseif(Folder::formatted_like_an_id($folder)){
			$success = $message->move_to_folder($folder) && $message->folder_id == $folder;
		}else{
			$this->response('Message with ID: '.$message->id.' was not originally a '.$folder.' message and cannot be moved to '.$folder, 422); //is this right? add a comment if you figure out why
		}
	
		if(!$success) $this->response('An unknown error occurred and the message could not be moved.', 400);				
		
		$this->response_message['id'] = $id;
		$this->response_message['folder'] = $folder;
		$this->response($this->response_message, 200);
	}
	
	
	public function process_incoming_post(){
		
		//check to make sure that this is being used by the API application
		if(!$this->request->admin_api_authorized || $this->request->public_key != WEBSERVICE_PUBLIC_KEY)
			$this->response( 'Access Denied. Use Not Authorized.', 403);
	
		$this->respond_with_error_if_mailbox_is_inactive();	
		
		//validate that we've been given a message id and that it's in the database
		if(!array_key_exists('id', $this->post()))
			$this->missing_required_fields[] = $required_field;
		$this->respond_with_error_if_fields_are_missing();
		
		$id = $this->post('id');
		if(!Message::formatted_like_an_id($id)){
			$this->invalid_fields[] = 'id';
		}
				
		$this->respond_with_error_if_fields_are_invalid();
		
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox) || !$message->is_incoming()){
			$this->response('Message not found.', 422);
		}
		$this->response_message['id'] = $id;
		
		//PROCCESS THE INCOMING MESSAGE AS NEEDED
		if($message->has_attachments() && !Patient::populate_for_message($message))
			$this->error->warning('Unable to populate patients for '.$message->describe());
		
		//if this is a message from outside of Direct with patient documents attached, save the patient documents to DAS
		if($message->has_attachments() && !$this->is->string_like_a_direct_address($message->sender)){

			require_library('attachment');

			$patient_documents = array();
			foreach($message->attachment_files as $name => $binary_string){
				$attachment = Attachment::create($name, $binary_string, array('message_id' => $message->id));
				if(is_a($attachment, 'Patient_document_attachment'))
					$patient_documents[] = $attachment;
			}
			
			if(!empty($patient_documents)){
				$success = true;
				foreach($patient_documents as $patient_document){
					$success = $patient_document->save_to_das() && $success;
				}
			
				$values = array('target_type' => 9,
								'target_id' => $message->id,
								'actor_type' => 2,
								'actor_id' => $this->application_id,
								'action' => 'Save '.count($patient_documents).' attached patient documents to DAS',
								'event_date' => time(),
								'success' => $success);
			
				Event::create($values);
			}
		}
		
		$this->response('Message was successfully processed.', 200);
	}	

	
	/**
	* Restores a message to its previous folder or folder in designated in the parameter
	* 
	* This action is not available for inactive mailboxes.
	*	
	* @param string mailbox name
	* @param int id of the message
	* @param int|string id of custom folder or inbox  
	*/
	public function restore_message_post(){
		$required_fields = array('id');
		$post = $this->post();
        foreach($required_fields as $required_field){
        if(array_key_exists($required_field, $post))
            $$required_field = $this->post($required_field,TRUE);
        else{
        	array_push($this->missing_required_fields, $required_field);
           }
        }

		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');
		$this->respond_with_error_if_mailbox_is_inactive();
		
		if(!Message::formatted_like_an_id($id)){
			$this->invalid_fields[] = 'id';
		}
		
		$folder_id = element('folder', $post, null);
		if($folder_id !== null && $folder_id !== 'inbox' && !Folder::formatted_like_an_id($folder_id) && !$this->mailbox->has_related('folder', array('id' =>$folder_id))){
			array_push($this->invalid_fields, 'folder');
		}
		
		$this->respond_with_error_if_fields_are_invalid();
		
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox)){
			$this->response('Message not found.', 422);
		}
		$this->response_message['id'] = $id;

#TODO - CHECK FOR SUCCESS FOR RESTORE() AND MOVE_TO_FOLDER AND MOVE_TO_INBOX BEFORE RETURNING

		$message->restore();
		
		if(Folder::formatted_like_an_id($folder_id)){
			$message->move_to_folder($folder_id);
		}elseif($folder_id == 'inbox'){
			$message->move_to_inbox();
		}
		
		$this->response($this->response_message, 200);
	}	
	
	/**
	* Sets the custom flag of the message specified
	* 
	* This action is not available for inactive mailboxes
	*	
	* @param string mailbox name
	* @param int id of the message
	* @param string text to set flag
	*/	
	public function set_flag_post(){
		$required_fields = array('id', 'text');
		$post = $this->post();
        foreach($required_fields as $required_field){
			if(array_key_exists($required_field, $post))
				$$required_field = $this->post($required_field,TRUE);
			else{
				array_push($this->missing_required_fields, $required_field);
           	}
        }
		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');
		$this->respond_with_error_if_mailbox_is_inactive();

		if(!Message::formatted_like_an_id($id)){
			$this->invalid_fields[] = 'id';
		}
		$this->respond_with_error_if_fields_are_invalid();
	
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox)){
			$this->response('Message not found.', 422);
		}
		$message->flags = $text;
		$message->save();
		$this->response_message['id'] = $id;
		$this->response_message['text'] = $text;
		$this->response($this->response_message, 200);
	}
		
}
?>