<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
*//** */

require_once APPPATH.'libraries/DaaS_REST_Controller.php';

/**
* @package direct-as-a-service
* @subpackage controllers
*/	
class Disclosure_controller extends DaaS_REST_Controller{
	
	function __construct() {
		parent::__construct();
		require_model('attachment_disclosure');
		
		if (!$this->request->disclosure_api_authorized) { //authorize request
				$this->response( 'Access Denied. Use Not Authorized.', 403);
		}
	}
	
	/**
	* Used by VAP/EhX to get a log of disclosures that have been made within a given time period.
	* VAP may use search criteria to get information disclosed for a specific user.  Limited to returning 1000 records at a time.
	*
	* @todo We need to apply a limit to the number of disclosures being returned - it's not scalable to allow people to pull as much as they want.  Suggested limit 1000.  (Covered by VAD-1672)
	*
	* @param string $first The first name of the patient whose information was disclosed
	* @param string $last The last name of the patient whose information was disclosed
	* @param int $ssn The SSN of the patient whose information was disclosed (9 integers, no dashes)
	* @param int $start The earliest point at which returning records should have been disclosed (Unix timestamp)
	* @param int $end The latest point at which returning records should have been disclosed (Unix timestamp) 
	* @param int $records The number of records to retrieve
	* @param int $page If we're limiting the number of records we're retrieving, which "page" are we on?  (DB offset will be $records * $page)
	* @param string $order The order in which records should be returned, formatted as a SQL order by clause (for reasons unlikely to become clear again soon)
	*/	
	public function audit_get() {
		$optional_fields = array('first' => null, 
								 'last' => null,
								 'ssn' => null,
								 'start' => null,
								 'end' => time(), 
								 'records' => null, 
								 'page' => 1, 
								 'order' => 'disclosed desc');
		
		//set variables for each optional field
		foreach($optional_fields as $optional_field => $default_value){
			$$optional_field = element($optional_field, $this->get(), $default_value); //checks to see if $field is in $get - if not, sets to null
		}
		
		if(!is_null($start) && !$this->is->unix_timestamp($start)){
			$this->invalid_fields[] = 'start';
		}
		
		if(!is_null($end) && (!$this->is->unix_timestamp($end) || (!is_null($start) && $end < $start))){
			$this->invalid_fields[] = 'end';
		}
		
		if(!is_null($ssn) && (!$this->is->unsigned_integer($ssn) || strlen($ssn) != 9)){
			$this->invalid_fields[] = 'ssn';
		}
		
		$allowed_orders = array('ssn','first','last','disclosed','title','recipient','username','purpose');
		
		if($order){
			$order_pieces = explode(' ',$order);
			if(in_array($order_pieces[0], $allowed_orders) && ($order_pieces[1] === 'desc' || $order_pieces[1] === 'asc')){
				Attachment_disclosure::db()->order_by($order_pieces[0],$order_pieces[1]);
			}
			else{
				$this->invalid_fields[] = 'order';
			}
		}
		
		if(!is_null($records) && !$this->is->unsigned_integer($records)){
			$this->invalid_fields[] = 'records';
		}
		
		if(!is_null($page) && (!$this->is->unsigned_integer($page) || is_null($records))){
			$this->invalid_fields[] = 'page';
		}
		
		$this->respond_with_error_if_fields_are_invalid();
		
		$conditions = array('recipient != ' => '');
		
		if(isset($start)) 
			$conditions['disclosed >='] = $start;
		
		if(isset($end)) 
			$conditions['disclosed <='] = $end;
			
		if(isset($records)){
			if(!isset($page)){
				$page = 1;
			}
			Attachment_disclosure::db()->limit($records,($page*$records));
		}
		
		foreach(array('first', 'last', 'ssn') as $field){
			if(isset($$field)) $conditions[$field] = $$field;
		}
		
		$disclosures = Attachment_disclosure::find($conditions);
		if(!is_array($disclosures)) $this->response('Failed to get disclosure log.', 500);
		
		foreach($disclosures as $key => $disclosure){
			$disclosures[$key] = $disclosure->values(array('recipient', 'first', 'last', 'disclosed', 'received', 'title', 'sent_to', 'received_from', 'username', 'facility', 'purpose', 'ssn', 'patient_id'));
		}
		
		//todo - would be much better to have disclosures within an array, to make it easier for end developers to pull out all the disclosures without the metadata
		$this->response($disclosures, 200);
	}
	
	/**
	* Used by VAP/EhX to get a summation of the number of disclosures made per domain within a given time period.
	*
	* @todo There should be a maximum number of possible records returned, even if it's very large.  We can't do that until we coordinate with VAP - see VAD-1672.
	* 
	* @param int $start Unix timestamp - take into account only disclosures made after this datetime
	* @param int $end Unix timestamp - take into account only disclosures made before this datetime
	* @param string $order The order in which to return domains/reports, formatted as a SQL query because we like pain
	*/
	public function summary_get() {
		$optional_fields = array('start','end','order');
	
		//set variables for each optional field
		foreach($optional_fields as $optional_field){
			$$optional_field = element($optional_field, $this->get()); //checks to see if $field is in $get - if not, sets to null
		}
		
		if(!is_null($start) && !$this->is->unix_timestamp($start)){
			$this->invalid_fields[] = 'start';
		}
		
		if(!is_null($end) && (!$this->is->unix_timestamp($end) || (!is_null($start) && $end < $start))){
			$this->invalid_fields[] = 'end';
		}
		
		$allowed_orders = array('sent_to','total');
		if($order){
			$order_piece = explode(' ',$order);
			if(in_array($order_piece[0], $allowed_orders) && ($order_piece[1] === 'desc' || $order_piece[1] === 'asc'))	{
				Attachment_disclosure::db()->order_by($order_piece[0],$order_piece[1]);
			}
			else{
				$this->invalid_fields[] = 'order';
			}	
		}
		
		$this->respond_with_error_if_fields_are_invalid();
	
		$conditions = array('recipient != ' => '');
		if(isset($start)) $conditions['disclosed >='] = $start;
		if(isset($end)) $conditions['disclosed <='] = $end;
		
		
		Attachment_disclosure::db()->select('sent_to, COUNT(*) as \'total\'');
		Attachment_disclosure::db()->group_by('sent_to');
		$disclosures = Attachment_disclosure::find($conditions,'sent_to');
		if(!is_array($disclosures)) $this->response('Failed to get disclosure log.', 500);
		$return = array();
		foreach($disclosures as $key => $disclosure){
			$return[] = $disclosure->values(array('sent_to', 'total'));
		}
		//todo - would be much better to have disclosures within an array, to make it easier for end developers to pull out all the disclosures without the metadata
		$this->response($return, 200);
	}
}