<?php defined('BASEPATH') OR exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage controllers
*/ /** */

require_once APPPATH.'controllers/admin/admin_controller.php';

/**
* @package direct-as-a-service
* @subpackage controllers
*/ 
class File_transfer_controller extends Admin_controller{
	
	var $_recipient_from_session;
	
	public function __construct(){
		parent::__construct();
		
		if($this->allow_non_va_users())
			$this->template->set('human_readable_domain_name', 'VLER Health Direct File Transfers');
		
		//we store success messages for this controller in flashdata, so check for them and add them to the template
		if(!empty($_SESSION['success_message']))
			$this->template->set('success_message', $_SESSION['success_message']);
		
	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// PAGES THAT WILL BE SHOWN TO VA USERS IN API ADMIN AND NON-VA USERS ON THE FILE TRANSFER SITE
// This is controlled by $this->allow_non_va_users(); add pages to this method if you need to add more of them (don't do this unless absolutely needed! consider security risks1)
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	function download($id, $hash){
		get_instance()->output->enable_profiler(false);        
		
		require_library('attachment');
		if(!File_transfer::formatted_like_an_id($id)) show_404();
		if(!$this->is->nonempty_string($hash)) show_404();
		
		File_transfer::select_all_except_content();
		$file_transfer = File_transfer::find_one(array('id' => $id, 'hash' => $hash));
		if(!File_transfer::is_an_entity($file_transfer)){
			$this->error->warning('Could not find a file transfer with id '.$id.' and hash '.$hash);
			show_404();
		}
		
		if(!$this->user_is_authorized_for_file_transfer($file_transfer)){
			if(!isset($this->user) || User::is_an_entity($this->user))
				redirect('file_transfer/authorize/'.$file_transfer->message_id.'/'.$hash); //if the user isn't logged in, we might just need the user to enter the passcode
			$this->error->warning('Current user has not been authorized to view '.$file_transfer->describe());
			show_404();			
		}
		
		session_write_close(); //we've gotten all the data we need from the session, so close it out to avoid concurrency issues if this takes a while
		
//todo - let's make sure the urls match the domain
		if($file_transfer->has_expired()) {
			redirect($file_transfer->url_for_view()); //send them back to the view page so that they'll see that this file has expired
			return;
		}
		
		$file_transfer->download();
		
		$file_transfer->mark_as_downloaded();	
	}
	
	function view($message_id, $hash){
		if(!Message::formatted_like_an_id($message_id)) show_404();
		if(!$this->is->nonempty_string($hash)) show_404();
		if(empty($this->recipient_from_session()) && $this->allow_non_va_users())
			redirect('file_transfer/authorize/'.$message_id.'/'.$hash); //if the user isn't logged in, we might just need the user to enter the passcode
		
		$file_transfer = $this->find_file_transfer_for_user($message_id, $hash, $allow_sender_to_view = true);
		if(!File_transfer::is_an_entity($file_transfer)){
			if(!isset($this->user) || User::is_an_entity($this->user))
				redirect('file_transfer/authorize/'.$message_id.'/'.$hash); //if the user isn't logged in, we might just need the user to enter the passcode
			
			$this->error->warning('Could not find a file transfer with id '.$message_id.' and hash '.$hash);
			show_404();
		}
		
		$sender = $file_transfer->sender;
		
		$this->template->set('page_title', 'File Transfer: View');
		$this->template->active_nav = 'file_transfer';
		$this->template->load('template', 'api/file_transfer/view', compact('file_transfer', 'sender'));
	}	
	
///////////////////////////////////////////////////////////////////////////////////////////////////////////
// ONLY AVAILABLE WHEN WE HAVE NO LOGGED-IN VA USER
///////////////////////////////////////////////////////////////////////////////////////////////////////////

	//when users aren't logged in, we require a passcode to confirm that the recipient is authorized to view the file transfer
	function authorize($message_id, $hash){
		if(!Message::formatted_like_an_id($message_id)) show_404();
		if(!File_transfer::exists(compact('message_id', 'hash'))) show_404();
		
		//if we already have a recipient authorized for this message for this session, redirect to the view (or the last attempted url)
		if($this->user_is_authorized($message_id, $hash))
			redirect('message/'.$message_id.'/file_transfer/'.$hash);
	
		
		
		require_library('form_markup_generator');
		$form = new Form_markup_generator();
		$form->set_fields( array('email_address' => array('label_text' => 'Direct Address', 'type' => 'email', 'required' => true),
								 'passcode' => array('type' => 'password', 'required' => true)));
		
		if(!empty($_POST)){
			$form->set_values_from_post();
			if(!$form->validates()){
				$error_message = '<strong>We need you to make a few changes to this form.</strong> '.ul_if_multiple($form->validation_messages);
			}elseif(!File_transfer::exists(array('message_id' => $message_id, 'hash' => $hash, 'recipient' => $form->value('email_address'), 'passcode' => trim($form->value('passcode'))))){
				$error_message = '<strong>We couldn\'t find a recipient for this file transfer with the given credentials.</strong> '
								 .'Please double-check the passcode and be sure to use the email address that the passcode was sent to.';
			}else{
				if(empty($_SESSION['file_transfer'])) $_SESSION['file_transfer'] = array();
				$_SESSION['file_transfer']['recipient'] = $form->value('email_address');
				if(empty($_SESSION['file_transfer']['messages'])) $_SESSION['file_transfer']['messages'] = array();
				if(empty($_SESSION['file_transfer']['messages'][$message_id])) $_SESSION['file_transfer']['messages'][$message_id] = array();
				
				//since we don't allow spaces and it's pretty likely people will make mistakes copying and pasting, we'll be kind and trim the passcode
				//note: if we ever allow spaces in passcodes, we'll have to stop doing this, but it would be an inadvisable thing to do
				$_SESSION['file_transfer']['messages'][$message_id][$hash] = trim($form->value('passcode')); 
				
				redirect('message/'.$message_id.'/file_transfer/'.$hash);
			}
		}
		
		$this->template->load('template', 'api/file_transfer/authorize', compact('form', 'error_message'));
	}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
// ONLY AVAILABLE TO LOGGED-IN VA USERS
// This will be enforced by DAAS_Controller and Admin_controller by checking $this->allow_non_va_users();
////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//note that this could give inaccurate results if the same user is uploading the same file twice in different windows within the same browser
	//I am pretty okay with the progress bar giving inaccurate results in that case - it's just a nice-to-have UI feature, not something that would break functionality -- MG 2016-09-28
	function ajax_upload_progress($name){
		session_write_close(); //we don't need to write new data to the session for this code, so avoid concurrency problems by closing it

		File_transfer::db()->select((array('id', 'upload_progress', 'name', 'size')))->order_by('id', 'desc');
		$file_transfer = File_transfer::find_one(array('created_by' => $this->user->id, 'downloaded_at' => null, 'name' => base64_decode($name)));
		if(!File_transfer::is_an_entity($file_transfer)){ //this could be because the file just hasn't made it to the db yet, so don't return an error
			$this->json->encode(array('upload_progress' => null)); 
			return;
		};
		
		$values = $file_transfer->values();
		$values['progress_increment'] = FILE_TRANSFER_UPLOAD_CHUNK_SIZE;
		echo $this->json->encode($values);
	}
	
	//allow logged-in users to view 
	function available_for_download(){
		$this->template->show_subnav = true;
		
		require_library('entity_lister');
		$lister = new Entity_lister(array(  'model' => 'file_transfer', 
											'method_for_list_count' => 'count_available_for_current_user',
											'method_for_list_generation' => 'find_available_for_current_user',
											'default_order_by' => 'created_at',
											'default_order' => 'desc',
											'columns' => array('name', 'size', 'created_by', 'created_at', 'expires_at'),
											'template_directory' => 'api/file_transfer/lister'));
											
		$lister->add_column_aliases(array('created_by' => 'Sent By', 'created_at' => 'Sent', 'expires_at' => 'Expiration'));
		
		$content_before_list = '<p>This table shows the files that you have received from other Direct users which are currently available for download.</p>';
		
		$this->template->load('template', 'api/file_transfer/list', compact('content_before_list', 'lister'));
	}
	
	function received_files(){
		$this->template->show_subnav = true;
		
		require_library('entity_lister');
		$lister = new Entity_lister(array(  'model' => 'file_transfer', 
											'method_for_list_count' => 'count_received_for_current_user',
											'method_for_list_generation' => 'find_received_for_current_user',
											'default_order_by' => 'created_at',
											'default_order' => 'desc',
											'columns' => array('name', 'size', 'created_by', 'created_at', 'expires_at'),
											'template_directory' => 'api/file_transfer/lister'));
											
		$lister->add_column_aliases(array('created_by' => 'Sent By', 'created_at' => 'Sent', 'expires_at' => 'Expiration'));
		
		$content_before_list = '<p>This table shows the files that you have received from other Direct users, including files which are no longer available for download.</p>';
		
		$this->template->load('template', 'api/file_transfer/list', compact('content_before_list', 'lister'));
	}
	
	function sent_files(){
		$this->template->show_subnav = true;
		
		require_library('entity_lister');
		$lister = new Entity_lister(array(  'model' => 'file_transfer', 
											'method_for_list_count' => 'count_sent_for_current_user',
											'method_for_list_generation' => 'find_sent_for_current_user',		
											'default_order_by' => 'created_at',
											'default_order' => 'desc',
											'columns' => array('name', 'size', 'recipient', 'created_at', 'expires_at', 'downloaded_at'),
											'template_directory' => 'api/file_transfer/lister' ));
		
		$lister->add_column_aliases(array('created_at' => 'Sent At', 'created_at' => 'Sent', 'expires_at' => 'Expiration', 'downloaded_at' => 'Downloaded'));
		
		$content_before_list = '<p>This table shows the files that you have sent to other Direct users. </p>';
		
		$this->template->load('template', 'api/file_transfer/list', compact('content_before_list', 'lister'));
	}		
	
	function upload(){
		ini_set('memory_limit', '1024M'); 
		ini_set('max_execution_time', '1200');

		$this->template->show_subnav = true;
		
		require_library('form_markup_generator');
		$form = new Form_markup_generator();
		$form->set_fields( array('recipient' => array('type' => 'email', 'required' => true),
								 'file_to_transfer' => array('type' => 'file', 'required' => true),
								 'notify_me' => array('type' => 'checkbox', 'label_text' => 'Notify me when this file is downloaded'),
								 'message' => array('type' => 'text_area',  'label_text' => 'Add a brief message about this file', 'attributes' => array('rows' => 5,)), 
								 'patient_information' => array('type' => 'patient_disclosures', 'comment' => 'If this file discloses patient information, please include the information below.'),
								 'progress_increment' => array('type' => 'hidden', 'value' => FILE_TRANSFER_UPLOAD_CHUNK_SIZE)));
		
		if(!empty($_POST)){
			$form->set_values_from_post();
			session_write_close(); //since we can't have concurrent session, close this while we do some time-consuming things

			if(!$form->validates()){
				$error_message = '<strong>We need you to make a few changes to this form.</strong> '.ul_if_multiple($form->validation_messages);
			}elseif(!$this->session->path_exists_in_cache($form->field('file_to_transfer')->value_full_path)){
				$error_message = $this->lang->line('error_generic');
			}else{ 
				$filename = $this->session->cache_root($form->field('file_to_transfer')->value_full_path);
				
				//save the actual file transfer
				$file_transfer = new File_transfer();
				$file_transfer->recipient = $form->field('recipient')->value;
				$file_transfer->name = $form->field('file_to_transfer')->value;
				$file_transfer->type = mime_content_type($filename);
				$file_transfer->notify_on_download = (bool)$form->field('notify_me')->value;
				if($form->field('message')->has_value)
					$file_transfer->note = $form->field('message')->value;
				$file_transfer->content_filepath = $filename;				
				
				if($file_transfer->save()){
					
					//save the patient disclosure information for this file
					$message_foreign_key = Disclosure::related_foreign_key('message');
					foreach($form->field('patient_information')->values as $patient_info){
						if(!empty($patient_info['given_name']) && !empty($patient_info['family_name']) && !empty($patient_info['ssn']) && !empty($patient_info['purpose_of_disclosure'])) {
							$disclosure = Attachment_disclosure::create(array($message_foreign_key => $file_transfer->message_id, 
																			 'hash' => $file_transfer->hash,
																			 'recipient' => $file_transfer->recipient,
																			 'disclosed' => time(),
																			 'sent_to' => substr($file_transfer->recipient, strpos($file_transfer->recipient, '@') + 1), 
																			 'first' => $patient_info['given_name'],
																			 'last' => $patient_info['family_name'],
																			 'ssn' => str_replace('-', '', $patient_info['ssn']),
																			 'purpose' => $patient_info['purpose_of_disclosure']));	
						}
					}
					
					session_start();
					$this->session->set_flashdata('success_message', '<strong>Success!</strong> A link to download the file '.link_to($file_transfer->url_for_view(), $file_transfer->name).' has been sent to '.$file_transfer->recipient);
					redirect(current_url());
				}
				
				$error_message = $this->lang->line('error_generic');
			}
			
		}
		
	
		$this->template->load('template', 'api/file_transfer/upload', compact('form', 'error_message'));
	}
	
////////////////////////////////
// PROTECTED METHODS
///////////////////////////////

	//a few of the pages in this controller can be accessed by people who are not authenticated users on the VA network - the rest can't
	//DAAS_Controller will make sure that 404s are served whenever this method does not true
	protected function allow_non_va_users(){
		return string_begins_with('https://'.API_FILETRANSFER_DOMAIN, $this->config->base_url())&& in_array($this->router->method, array('authorize', 'view', 'download'));
	}

	protected function find_file_transfer_for_user($message_id, $hash, $allow_sender_to_view = false){
		if(isset($this->user) && User::is_an_entity($this->user)){
			if($allow_sender_to_view)
				File_transfer::db()->group_start()->where('recipient', $this->recipient_from_session())->or_where('created_by', $this->user->id)->group_end();
			else
				File_transfer::db()->where('recipient', $this->recipient_from_session());
				
			File_transfer::select_all_except_content();
			return File_transfer::find_one(array('message_id' => $message_id, 'hash' => $hash));
		}
		
		$recipient = $this->non_va_recipient_from_session();
		$passcode = $this->passcode_from_session($message_id, $hash);
		if(empty($recipient) || empty($passcode))
			return null;
			
		File_transfer::select_all_except_content();
		return File_transfer::find_one(compact('message_id', 'hash', 'recipient', 'passcode'));
	}
	
	//would be more efficient to do this with an exists check, but this shoud be okay as long as we're not selecting content
	protected function user_is_authorized($message_id, $hash){
		$file_transfer = $this->find_file_transfer_for_user($message_id, $hash);
		return File_transfer::is_an_entity($file_transfer);
	}
	
	protected function user_is_authorized_for_file_transfer($file_transfer){
		if(!File_transfer::is_an_entity($file_transfer)) return $this->error->should_be_a_file_transfer_entity($file_transfer);
		
		if(isset($this->user) && User::is_an_entity($this->user)){
			return strtolower($file_transfer->recipient) == strtolower($this->recipient_from_session());
		}
	
		$recipient = $this->non_va_recipient_from_session();
		$passcode = $this->passcode_from_session($file_transfer->message_id, $file_transfer->hash);
		if(empty($recipient) || empty($passcode))
			return false;
			
		return strtolower($file_transfer->recipient) == strtolower($recipient) && $file_transfer->passcode == $passcode;
	}
	
	//this exists mainly because $this->user->email_address() involves db queries, and it messes with our other db queries to have to look up the email address while forming other queries
	protected function recipient_from_session(){
		if(empty($this->_recipient_from_session)){
			if(isset($this->user) && User::is_an_entity($this->user))
				$this->_recipient_from_session = $this->user->email_address();
			else
				$this->_recipient_from_session = $this->non_va_recipient_from_session();
		
		}
		return $this->_recipient_from_session;
	}

	//recipients are stored in $_SESSION['file_transfer']['recipient'];
	protected function non_va_recipient_from_session(){
		if(!array_key_exists('file_transfer', $_SESSION)|| empty($_SESSION['file_transfer']) || !is_array($_SESSION['file_transfer']))
			return null;
			
		$recipient = element('recipient', $_SESSION['file_transfer']);
		if(empty($recipient) || !$this->is->string_like_an_email_address($recipient))
			return null;
			
		return $recipient;
	}
	
	//passcodes are stored in $_SESSION['file_transfer']['messages'][$message_id][$hash];
	protected function passcode_from_session($message_id, $hash){
		if(!array_key_exists('file_transfer', $_SESSION)|| empty($_SESSION['file_transfer']) || !is_array($_SESSION['file_transfer']))
			return null;
			
		$messages = element('messages', $_SESSION['file_transfer']);
		if(!is_array($messages) || empty($messages))
			return null;
			
		$transfers = element($message_id, $messages);
		if(!is_array($transfers) || empty($transfers))
			return null;
		
		$passcode = element($hash, $transfers);
		if(empty($passcode)) 
			return null;
			
		return $passcode;
	}
}