<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

/** require parent */
require_model('mailbox');

/**
* @package direct-project-innovation-initiative
* @subpackage models
*/
class User extends Mailbox {
	public static $primary_key = 'user_id';
	protected $_is_admin;
	protected $_ldap_entry;
	
	
	function given_name(){
		return element('givenname', $this->ldap_entry);
	}
	
	function family_name(){
		return $this->sn();
	}
	
	//full name, not including middle name
	//currently a wrapper for cn - in part to make it clearer to the non-ldap-savvy what this is, but also in case we ever want display name to be a modified version of the cn
	function display_name(){
		return $this->cn;
	}
	
	function cn(){
		return element('cn', $this->ldap_entry);
	}
	
	function sn(){
		return element('sn', $this->ldap_entry);
	}
	
	function dn(){
		return element('dn', $this->ldap_entry);
	}

	function is_admin(){
		if(!isset($this->_is_admin)){
			$CI = get_instance();
			$CI->load->library('ldap');
			$groups = $CI->ldap->get_admin_group_membership($this->dn());
			$this->_is_admin = $CI->is->nonzero_unsigned_integer($groups['count']);
		}
		return $this->_is_admin;
	}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// ATTACHMENT CACHE FUNCTIONALITY
// Before they're attached to a message, attachments are stored in a cache specific to the logged-in user.  These functions deal with
// parsing/managing the attachments in the cache.
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////	
	
	
	function attachment_cache(){
		return APPPATH.'cache/'. $this->user_name.'/';
	}
	
	function attachment_from_cache($filename){
		if(!$this->file_name_exists_in_attachment_cache($filename)) return false;
		
		$binary_string = file_get_contents($this->attachment_cache().$filename);
		if($binary_string === FALSE) $this->error->warning('Could not read '.$path);
		
		return Attachment::create($filename, $binary_string, array('directory' => $this->attachment_cache()));
	}
	
	function attachments_from_cache(){
		require_library('attachment');
		
		if(!file_exists($this->attachment_cache())) return array();
		$filenames = get_filenames($this->attachment_cache());
		if(!is_array($filenames)) return $this->error->warning('Could not read '.$this->attachment_cache());
		
		natcasesort($filenames);

		$files = array();
		foreach($filenames as $filename){
			$attachment = $this->attachment_from_cache($filename);
			if(is_a($attachment, 'Attachment')) $files[$filename] = $attachment;
		}
		return $files;
	}	
	
	function attachment_cache_size_in_bytes(){
		$attachment_files = $this->attachments_from_cache();
		if(empty($attachment_files)) return 0;	
		return array_sum(collect('bytes', $attachment_files));
	}
	
	function file_name_exists_in_attachment_cache($file_name){
		if(empty($file_name) || !$this->is->string_like_a_file_path($file_name)) return $this->error->should_be_a_file_name($file_name);
		if(!directory_exists($this->attachment_cache())) return false;
		return file_exists($this->attachment_cache().$file_name);
	}
	
	function attachment_exists_in_cache($name, $binary_string){
		if(!file_exists($this->attachment_cache().$name)) return false;
		$current_binary_string = file_get_contents( $this->attachment_cache().$name );
		if($current_binary_string === FALSE){
			$this->error->warning('Could not read '.$this->attachment_cache().$name);
			return true; //assume that they're the same, since we can't determine otherwise
		}
		
		return ($current_binary_string == $binary_string);
	}
	
#TODO - WHAT IF THE USER HAS TWO DIFFERENT COMPOSE WINDOWS OPEN IN TWO DIFFERENT TABS?  WE NEED TO CACHE BY MESSAGE, NOT JUST BY USER
	function add_attachment_to_cache($name, $binary_string){			
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!is_string($binary_string)) return $this->error->should_be_a_binary_string($binary_string);
		
		//first, make sure we have a cache 
		if(!directory_exists($this->attachment_cache())) mkdir($this->attachment_cache()); //attempt to create the cache if needed
		if(!directory_exists($this->attachment_cache())) return $this->error->warning('Could not create an attachment cache for '.$this->describe().' at '.$this->attachment_cache());
		
		if($this->attachment_exists_in_cache($name, $binary_string)) return true; //attachment is already in cache; this is the EXACT attachment, with the same binary string & everything
		
		
#TODO - MAKE THE FILENAME UNIQUE IF NEEDED
		$path = $this->attachment_cache().$name;		
		$success = file_put_contents($path, $binary_string);	
		if($success === FALSE)
			return $this->error->warning('Unable to write '.$path.' to attachment cache');
		return true;
	}
	
	function remove_attachment_from_cache($name){
		if(!directory_exists($this->attachment_cache())) return true; //we're done!		

		$path = $this->attachment_cache().$name;
		if(!file_exists($path)) return true; //we're done!
		if(!unlink($path)) return $this->error->warning('Failed to remove '.$path);
		return true;
	}
	
	
	function clear_attachment_cache(){
		if(!directory_exists($this->attachment_cache())) return true; //we're done!
		delete_files($this->attachment_cache());
		return rmdir($this->attachment_cache());
	}
	
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// Creating a SYSTEM CACHE for Attachment Previews for now... - BH -1/06/2014
// This will be revisited to scale for other features, such as including sub-folders for messages,
// different types of attachement previews, etc.
//////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	function system_cache(){
		return APPPATH.'cache/system/'. $this->user_name.'/';
	}
	
	function attachment_from_system_cache($filename){
		if(!$this->file_name_exists_in_system_cache($filename)) return false;
	
		$binary_string = file_get_contents($this->system_cache().$filename);
		if($binary_string === FALSE) $this->error->warning('Could not read '.$path);
	
		return Attachment::create($filename, $binary_string, array('directory' => $this->system_cache()));
	}
	
	function attachments_from_system_cache(){
		require_library('attachment');
	
		if(!file_exists($this->system_cache())) return array();
		$filenames = get_filenames($this->system_cache());
		if(!is_array($filenames)) return $this->error->warning('Could not read '.$this->system_cache());
	
		natcasesort($filenames);
	
		$files = array();
		foreach($filenames as $filename){
			$attachment = $this->attachment_from_system_cache($filename);
			if(is_a($attachment, 'Attachment')) $files[$filename] = $attachment;
		}
		return $files;
	}
	
	function system_cache_size_in_bytes(){
		$attachment_files = $this->attachments_from_system_cache();
		if(empty($attachment_files)) return 0;
		return array_sum(collect('bytes', $attachment_files));
	}
	
	function file_name_exists_in_system_cache($file_name){
		if(empty($file_name) || !$this->is->string_like_a_file_path($file_name)) return $this->error->should_be_a_file_name($file_name);
		if(!directory_exists($this->system_cache())) return false;
		return file_exists($this->system_cache().$file_name);
	}
	
	function file_exists_in_system_cache($name, $binary_string){
		if(!file_exists($this->system_cache().$name)) return false;
		$current_binary_string = file_get_contents( $this->system_cache().$name );
		if($current_binary_string === FALSE){
			$this->error->warning('Could not read '.$this->system_cache().$name);
			return true; //assume that they're the same, since we can't determine otherwise
		}
	
		return ($current_binary_string == $binary_string);
	}
	
	#TODO - WHAT IF THE USER HAS TWO DIFFERENT COMPOSE WINDOWS OPEN IN TWO DIFFERENT TABS?  WE NEED TO CACHE BY MESSAGE, NOT JUST BY USER
	function add_file_to_system_cache($name, $binary_string){
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!is_string($binary_string)) return $this->error->should_be_a_binary_string($binary_string);
		
		//first, make sure we have a cache
		if(!directory_exists($this->system_cache())) mkdir($this->system_cache(),0777,true); //attempt to create the cache if needed
		if(!directory_exists($this->system_cache())) return $this->error->warning('Could not create an attachment cache for '.$this->describe().' at '.$this->system_cache());
		
		if($this->file_exists_in_system_cache($name, $binary_string)) return true; //attachment is already in cache; this is the EXACT attachment, with the same binary string & everything
		
		#TODO - MAKE THE FILENAME UNIQUE IF NEEDED
		$path = $this->system_cache().$name;
		$success = file_put_contents($path, $binary_string);
		if($success === FALSE)
			return $this->error->warning('Unable to write '.$path.' to attachment system cache');
					return true;
		}
		
		function remove_attachment_from_system_cache($name){
		if(!directory_exists($this->system_cache())) return true; //we're done!
		
		$path = $this->system_cache().$name;
		if(!file_exists($path)) return true; //we're done!
		if(!unlink($path)) return $this->error->warning('Failed to remove '.$path);
		return true;
	}
	
	
	function clear_system_cache(){
		if(!directory_exists($this->system_cache())) return true; //we're done!
		delete_files($this->system_cache(), TRUE);
		return rmdir($this->system_cache());
	}	
	
	
////////////////////////////
// GETTERS
////////////////////////////	


	//get the info about this user from LDAP *once* - to force this info to refresh, set $_ldap_entry to null
	public function ldap_entry(){
		if(!isset($this->ldap_entry)){
			$this->_ldap_entry = array(); //by default, set an empty array so that we don't keep trying to access LDAP if the call isn't working
			$CI = get_instance();
			$CI->load->library('ldap');
			$result = $CI->ldap->search(null,1,array('cn','sn','givenName'),'(uid='.$this->username.')');
			
			//make sure that the result is an array
			if(!is_array($result)){
				$this->error->warning('No LDAP entries were found for user '.$this->error->describe($this->username));				
			}else{
				//let the developer know something's going wrong if we have more than one LDAP entry for this user
				if(count($result) > 1){
					$this->error->warning(count($result).' LDAP entries were found for user '.$this->error->describe($this->username).' when only one was expected.');
				}
				
				$this->_ldap_entry = first_element($result);
			}
			
		}
		return $this->_ldap_entry;
	}

	public function username(){
		return $this->user_name;
	}
	
/////////////////////////////
// STATIC METHODS
/////////////////////////////	
	
	public static function find_from_session(){
		$username = get_instance()->session->userdata('username');
		if(empty($username)) return false;
		
		$CI = get_instance();
		if(isset($CI->user) && User::is_an_entity($CI->user) && $CI->user->username == $username)
			return $CI->user; //skip the db query
		
		return User::find_one( array('user_name' => $username) );
	}	
	
	public static function log_out($destination='auth/logout'){
		if(empty($destination)) $destination = 'auth/logout';
			
		$CI = get_instance();
		$uri_before_logout = $CI->session->flashdata('uri_before_logout');		
		if(empty($uri_before_logout)){
			if(IS_AJAX)
				$uri_before_logout = $_SERVER['HTTP_REFERER'];
			else
				$uri_before_logout = $CI->config->site_url($CI->uri->uri_string());
		}
		
		//figure out where we are
		$controller = strtolower(get_class(get_instance($CI)));
		$method = strtolower($CI->router->method);
		if($method == 'index' && !string_ends_with('index', $destination)) $method = '';
		$action = implode_nonempty('/', array($controller, $method));
		
		//destroy the session
		session_destroy(); //note that as of CI 3.0, this destroys the CI session along with the PHP session
		
		//start a new session so we can set the previous URI			
		session_start(); //not 100% sure this is the correct CI3.0 practice to replace the old sess_create, but seems to match the session_destroy practice -- MG 2015-06-02
					
		//redirect to the logout page unless we're already there		
		$destination = strtolower(strip_from_beginning('/', strip_from_end('/', $destination))); //normalize the / for destination
		if(string_contains('?', $destination)) $destination = substr($destination, 0, strpos($destination, '?')); //make sure we're compensating for a query string
		if(string_begins_with(site_url(), $uri_before_logout)) $CI->session->set_flashdata('uri_before_logout', $uri_before_logout);
					
		//redirect to the logout page unless we're already there	
		if($destination != $action){
			redirect($destination);
		}
	}
	
	public static function random_password() {
		$chars = array('a','b','c','d','e','f','g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','1','2','3','4','5','6','7','8','9','0','!','@','#','$','%','^','&','*','(',')','{','}','?');
		$pass = '';
		$count = count($chars);
		for($i = 0; $i < 32; $i++) {
			$pass .= $chars[abs(hexdec(bin2hex(openssl_random_pseudo_bytes(6)))%$count)];
		}
		return $pass;
	}
	
}
