<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

/** Include the parent model */
require_model('distribution_list_model');

/**
* Extends the Distro model to include functionality specific to public distribution lists.
*
* Public distribution lists are stored in LDAP.  For consistency with private lists, the 
* public list model will translate database field names into their LDAP equivalent 
* values when searching, and reverse the translation when returning values.  This means 
* that you can still search for a list using the criteria array('name' => 'My List'),
* and the model will find it even though name is stored in the cn field in LDAp.
*
* Public lists are not deleted completely, but instead moved to a separate ou for deleted
* lists.  You can interact with deleted public lists by using the {@link deleted_public_distro_list}
* model.  For backwards compatability, you may also append _deleted to any of the public methods of 
* this model to access the deleted model equivalent.
*
* @author M. Gibbs <gibbs_margaret@bah.com>
*
* @package direct-project-innovation-initiative
* @subpackage models
*/
class Public_distribution_list_model extends Distribution_list_model {
	
	var $ou = LDAP_DISTRUBUTION_LIST_DN;
	var $column_aliases = array('id' => 'ou',
							 	'name' => 'cn');
	
	
	/////////////////////////////
	// SEARCH
	/////////////////////////////	

	protected function _find($criteria){
		$criteria = $this->translate_to_ldap($criteria);

#TODO - DO WE ACTUALLY WANT TO DO THIS?
		//make sure that no criteria is being used that shouldn't be
		$known_criteria = array( 'ou', 'cn', 'description' );
		$unknown_criteria = array_diff_key($criteria, array_flip($known_criteria));
		if(!empty($unknown_criteria)){
			$this->error->notice('Removing unrecognized criteria from search: '.sprp_for_log($unknown_criteria, 'unknown'));
			$criteria = array_intersect_key($criteria, array_flip($known_criteria));
		}

		//generate the actual filter
		if(empty($criteria))
			$criteria['cn'] = '*';
			
		foreach($criteria as $index => $unescaped) { $criteria[$index] = $this->ldap->ldap_escape($unescaped); }
		$filter = '(&(objectclass=groupOfNames)('.key_value_implode($criteria, '=', ')(').'))';

		$results = ldap_search( $this->ldap->conn, $this->ou, $filter);
		$entries = $this->ldap->get_formatted_entries($results, 'ou');
		
#TODO - PROBABLY DON'T WANT TO PARSE THIS EVERY TIME, THIS WILL BE INEFFICIENT WHEN LISTING ALL ENTRIES		
		//format entries to return as closely to the db format as possible
		foreach($entries as &$entry){
			//find the internal members
			$members = element('member', $entry, array());
			if(is_string($members)) $members = array($members);

			$addresses = array();
			foreach($members as $member){
				if(!string_begins_with('uid=', $member)) continue; //skip the admin account
				$addresses[] = strtolower(strip_from_end(','.LDAP_ACCOUNTS_DN, strip_from_beginning('uid=', $member))).'@'.DIRECT_DOMAIN;
			}			
			
			//find the external members
			$external_member_results = ldap_search( $this->ldap->conn, 'ou='.$entry['ou'].','.$this->ou, '(objectclass=inetOrgPerson)');
			$external_members = $this->ldap->get_formatted_entries($external_member_results, 'uid');
			$addresses = array_merge($addresses, array_map('strtolower', array_keys($external_members)));

			//format the entry
			$entry = $this->translate_to_db(array_intersect_key($entry, array_flip($known_criteria)));
			sort($addresses);
			$entry['addresses'] = implode(';', $addresses);
		}

		return $entries;
	}
	
	
	///////////////////////////////////
	// MANAGE LIST C.R.U.D.
	///////////////////////////////////

	protected function _values_are_valid_for_create(&$values){
		if(!parent::_values_are_valid_for_create($values)) return false;	
		$values = $this->translate_to_ldap($values);
		
		//generate an id if it hasn't been provided
		if(empty($values['ou'])){
			
			//generate a url-friendly version of the name
			$id = url_title($values['cn'], '_', TRUE);
			
			//failsafe in case url_title isn't as reliable as we'd like
			if(!$this->formatted_like_an_id($id)){
				$id = 'public_list_'.time();
			}
			
			//check to make sure there isn't already a similar id - if so, increment
			$counter = 1;
			while($this->exists($id) || $this->exists_deleted($id)){
				if($counter > 1){
					$id = strip_from_end('_'.($counter - 1), $id);
				}
				$id .= '_'.$counter;
				$counter++;
			}
		
			$values['ou'] = $id;
		} 

		if(!array_key_exists('ou', $values))
			return $this->error->warning('Please specify an id for the distribution list');		
					
		//check that we have the right values (name & description checked by parent)
		extract($values);
		if(!$this->formatted_like_an_id($ou)) return $this->error->should_be_a_valid_ou($ou);
		if($this->exists(array('id' => $ou)) || $this->exists_deleted(array('id' => $ou))) return $this->error->should_be_an_unused_ou($ou);
		
		return true;
	}
	
#TODO - ASK AB IF IT'S OK TO MAKE THE CONNESTEOR VUBLIC / IMPLEMENT SOME OTHER WAY TO GRAB THE CONNECTION
	protected function _create($values){
		$values['objectclass'] = 'groupOfNames';
		$values['member'] = 'cn=admin,'.LDAP_BASE_DOMAIN;
		
#TODO - VALIDATE THAT THESE ARE KNOWN ATTRIBUTES?
		
		//create the record
		$success = ldap_add($this->ldap->conn, "ou=" . $values['ou'] . "," . $this->ou, $values);
		if(!$success){
			//let create() give a generic error - ldap_add will trigger its own errors as necessary
			return false; 
		}
		
		//return the id
		return $values['ou'];
	}
	
	protected function _update($id, $values){
		$values = $this->translate_to_ldap($values);
		
		//any values that are set to '', null, etc. will be removed from the entry
		$values_to_remove = array();
		foreach($values as $attribute => $value) {
			if(empty($value) && $value !== 0){ 
				$values_to_remove[$attribute] = array(); 
				unset($values[$attribute]);
			}
		}
		
		$success = true;
		if(!empty($values_to_remove))
			$success = $success && ldap_mod_del($this->ldap->conn,"ou=" . $id . "," . $this->ou,$values_to_remove);
		if(!empty($values))
			$success = ldap_modify($this->ldap->conn, "ou=" . $id. "," . $this->ou,$values);
		return $success;
	}	
	
	protected function _delete($id){
		//member attributes move gracefully, but external addresses don't.  Grab these first.
		$external_member_results = ldap_search( $this->ldap->conn, 'ou='.$id.','.$this->ou, '(objectclass=inetOrgPerson)');
		$external_members = $this->ldap->get_formatted_entries($external_member_results, 'uid');
		
		//delete the external members from our dl
		foreach($external_members as $uid => $values){
			ldap_delete($this->ldap->conn, 'uid='.$uid.',ou=' . $id . ',' . $this->ou);
		}
		
		$success = ldap_rename($this->ldap->conn, "ou=" . $id . "," . $this->ou, "ou=" . $id,LDAP_DELETED_DISTRUBUTION_LIST_DN, TRUE);
		//apparently this is one of the few places where ldap doesn't trigger an error itself ....
		if(!$success && $this->ldap->has_ldap_error()) $this->error->warning('LDAP ERROR: '.$this->ldap->get_ldap_error());
		
		//now, re-add the external members to the deleted dl entry
		foreach($external_members as $uid => $values){
			ldap_add($this->ldap->conn, 'uid='.$uid.',ou=' . $id . ',' . LDAP_DELETED_DISTRUBUTION_LIST_DN, $values);
		}
				
		return $success;
	}
	
	///////////////////////////
	// MANAGE LIST MEMBERSHIP
	///////////////////////////

	// $display_name should be last name, first name
	// $display_name is not currently being used
	function add_address_to_list($list_id, $address, $display_name=''){
		if(!$this->formatted_like_an_id($list_id)) return $this->error->should_be_a_distribution_list_id($list_id);
		if(empty($address) || !$this->is->string_like_an_email_address($address)) return $this->error->should_be_an_email_address($address);
		if(!$this->is->string($display_name)) return $this->error->should_be_a_string($display_name);
		
		$addresses = $this->addresses_for_list($list_id);
		if(!is_array($addresses)) return $this->error->warning('Could not add '.$address.' to '.strip_from_end('_model', get_class($this)).'#'.$list_id);
		if(in_array($address, $addresses)) return true; // doesn't need to be added
		
		//add internal address entry
		if($this->formatted_like_a_direct_address($address)){
			return ldap_mod_add($this->ldap->conn,"ou=" . $list_id . "," . $this->ou, array("member" => $this->direct_address_to_ldap_dn($address)));
		}
		
		//create the external address entry
		$values['uid'] = $address;
		$values['objectclass'] = 'inetOrgPerson';
		//default to using the email address as the cn/sn since they're required fields
		
		$values['displayName'] = empty($display_name) ? $address : $display_name;
		$values['sn'] = empty($display_name) ? $address : first_element(explode(', ', $display_name));
		$values['cn'] = empty($display_name) ? $address : last_element(explode(', ', $display_name));
		$values['mail'] = $address;
		
		return ldap_add($this->ldap->conn, 'uid='.$address.',ou=' . $list_id . ',' . $this->ou, $values); 
	}
	
	function remove_address_from_list($list_id, $address){
		if(!$this->formatted_like_an_id($list_id)) return $this->error->should_be_a_distribution_list_id($list_id);
		if(empty($address) || !$this->is->string_like_an_email_address($address)) return $this->error->should_be_an_email_address($address);
		
		//ldap will give us an error if we try to delete a person who doesn't exist, so check first
		if(!in_array($address, $this->addresses_for_list($list_id))) return true;
		
		if($this->formatted_like_a_direct_address($address)){
			return ldap_mod_del($this->ldap->conn,"ou=" . $list_id . "," . $this->ou, array("member" => $this->direct_address_to_ldap_dn($address)));
		}
		
		return ldap_delete($this->ldap->conn, 'uid='.$address.',ou=' . $list_id . ',' . $this->ou); 	
	}
	
	//by default, this will return the addresses; extend in child classes to use the appropriate method.
	protected function _display_names_for_external_addresses($list_id, $addresses){
		if(!is_array($addresses)) return $this->error->should_be_an_array($addresses);
		$external_addresses = $this->external_addresses_for_list($addresses);
		
		
		//find the external members
		$external_member_results = ldap_search( $this->ldap->conn, 'ou='.$list_id.','.$this->ou, '(objectclass=inetOrgPerson)');
		$external_members = $this->ldap->get_formatted_entries($external_member_results, 'uid');
		return collect('cn', $external_members);
	}	
	

	function alias($list_id){
		if(!$this->formatted_like_an_id($list_id)) return $this->error->should_be_a_distribution_list_id($list_id);
		//gdl is a relic from when we were calling these global lists.  which we possibly should still be doing.  drat.
		return 'gdl-'.$list_id;
	}
	
	function id_from_alias($alias){
		if(!$this->is->nonempty_string($alias)) return $this->error->should_be_a_nonempty_string($alias);
		return strip_from_beginning('gdl-', $alias);
	}
	
	
	////////////////////////
	// VALIDATION 
	////////////////////////

	function formatted_like_an_id($id){
		if(!$this->is->nonempty_string_with_no_whitespace($id)) return false;
		return $this->is->string_like_an_email_address($id.'@'.DIRECT_DOMAIN);
	}

	function formatted_like_an_alias($alias){
		if(!$this->is->nonempty_string($alias)) return $this->error->should_be_a_nonempty_string($alias);
		return (string_begins_with('gdl-', $alias) && $this->formatted_like_an_id(strip_from_beginning('gdl-', $alias)));
	}
	
	/**
    * Check to make sure that this name isn't in use.
    * Overrides parent to include deleted lists as well.
    * @param string 
    * @return boolean 
    */
    function name_is_available($name){
        if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
        
		$conditions = array('name' => $name);
        return !$this->exists($conditions) && !$this->exists_deleted($conditions);
    }
	
	/////////////
	//
	/////////////
	
	protected function translate_to_ldap($values){
		if(!$this->is->array($values)) return $this->error->should_be_an_array($values);
		foreach($this->column_aliases as $db_version => $ldap_version){
			if(array_key_exists($db_version, $values)){
				$values[$ldap_version] = $values[$db_version];
				unset($values[$db_version]);
			}
		}
		return $values;
	}
	
	protected function translate_to_db($values){
		if(!$this->is->array($values)) return $this->error->should_be_an_array($values);
		foreach($this->column_aliases as $db_version => $ldap_version){
			if(array_key_exists($ldap_version, $values)){
				$values[$db_version] = $values[$ldap_version];
				unset($values[$ldap_version]);
			}
		}
		return $values;
	}
	
	protected function direct_address_to_ldap_dn($address){
		if(!$this->formatted_like_a_direct_address($address)) return $this->error->should_be_a_direct_address($address);
		return 'uid='.strip_from_end('@'.DIRECT_DOMAIN, $address).','.LDAP_ACCOUNTS_DN;
	}
	
	///////////////////////////////////
	// PHP Magic Methods
	///////////////////////////////////
	
	/* Allow developers to access the deleted list methods if they append _deleted to this model's method names. */
	public function __call($name, $arguments){
		
		if(!string_begins_with('_', $name) && string_ends_with('_deleted', $name)){
			$this->load->model('deleted_public_distribution_list_model');
			$deleted_model = & $this->deleted_public_distribution_list_model;
			$name_on_deleted_model = strip_from_end('_deleted', $name);
			if($name_on_deleted_model != 'delete' && method_exists($deleted_model, $name_on_deleted_model)){
				return call_user_func_array(array($deleted_model, $name_on_deleted_model), $arguments);
			}
		} 
		
		//this is the same fatal error that PHP would be calling if we weren't overloading __call
		$this->error->fatal('Call to undefined method '.get_class($this).'::'.$name.'()', 1);
    } 
}