<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class Auth extends CI_Controller {
	protected $destination_after_login; //set by default to the inbox uri in the constructor 
	
	public function __construct(){
		parent::__construct();
		$this->destination_after_login = site_url('inbox');
		if(!empty($this->session->flashdata('uri_before_logout')) && $this->session->flashdata('uri_before_logout') != site_url('auth/logout')){
			$this->session->keep_flashdata('uri_before_logout');
			$this->destination_after_login = $this->session->flashdata('uri_before_logout');
		}
	}

	public function index(){

		//if user is logged in already, redirect them to their inbox, otherwise load the login view
		if($this->session->userdata('is_loggedin') === 'true')  
			return redirect($this->destination_after_login);
		
	
		if(USE_CAC_AUTH === TRUE || USE_PIV_AUTH === TRUE) { 
			//check both CAC and PIV credentials, as long as one is successful, log the user in
			$cert_auth = FALSE;
			if(USE_CAC_AUTH === TRUE) { $cert_auth = $this->cert_auth(); }
			if(!$cert_auth) {
				if(USE_PIV_AUTH === TRUE) { $cert_auth = $this->piv_cert_auth(); }
			}
			
			if($cert_auth) {
				$this->login();
			}
			else {
				redirect('registration');
			}
		}
		else {
			$data["title"] = PORTAL_TITLE_PREFIX . "Login";
			$data['message_class'] = 'error';
			$data['message'] = validation_errors();
			$this->load->view('auth/login', $data);
		}
		
	}

	//logs the user in
	function login() {
			
		if(USE_CAC_AUTH === TRUE || USE_PIV_AUTH === TRUE) {
			//if CAC or PIV is being used and trusted, we do not have to worry about concurrent sessions or repeated login attempts
			//because this is handled by the CAC or PIV (CAC or PIV will lock on too many failures, must have the physical CAC or PIV to login anywhere so concurrent sessions are impossible)
			$cert_auth = FALSE;
			if(USE_CAC_AUTH === TRUE) { $cert_auth = $this->cert_auth(); }
			if(!$cert_auth) {
				if(USE_PIV_AUTH === TRUE) {$cert_auth = $this->piv_cert_auth(); }
			}
			
			if(empty($cert_auth)){
				log_message('error', 'No certificate authentication found');
			}
			
			$username = $cert_auth['user_name'];
			$password = $this->encrypt->decode($cert_auth['user_ep']);
			$this->ldap_auth($username,$password,TRUE);
		}
		else {
			$username = $this->input->post('email',TRUE);
			$password = $this->input->post('password',TRUE);
			//validate form input
			$this->form_validation->set_rules('email','Email Address','required','xss_clean');
			$this->form_validation->set_rules('password','Password','required');

			//if the user is logging in, attempt a connection to the imap server
			if($this->form_validation->run() == true) {
				//queries to find out recent login attempts/established sessions for this user
				$login_attempt_arr = $this->check_login_attempts($username);
				$session_check_arr = $this->check_user_sessions($username);
				//if there have been prevoius failed logins since the last successful login
				if(is_array($login_attempt_arr)) {
					//check first if the max login attempts for the day has been reached
					if($login_attempt_arr['failed_attempts'] >= MAX_LOGIN_ATTEMPTS) {
						$data['title'] = PORTAL_TITLE_PREFIX . 'Login';
						$data['message_class'] = 'error';
						$data['message'] = 'Login attempts exceeded. Wait 24 hours or contact an administrator before attempting to login again.';
						$this->load->view('auth/login', $data);
					}
					//check if the amount of time required to wait between failed login attempts has passed
					else if(pow(3,$login_attempt_arr["failed_attempts"]) > date("U") - $login_attempt_arr["last_failed_login_time"]) {
						$data['title'] = PORTAL_TITLE_PREFIX . 'Login';
						$data['message_class'] = 'error';
						$data['message'] = 'Must wait ' . (pow(3,$login_attempt_arr['failed_attempts']) - (date('U') - $login_attempt_arr['last_failed_login_time'])) . ' seconds before making another login attempt.';
						$this->load->view('auth/login', $data);
					}
					else { 
						//if they are allowed to attempt a login as far as previous attempts go, check sessions first
						$this->session_auth_check($session_check_arr,$username,$password);
					}
				}
				else {
					//if they are allowed to attempt a login, check sessions first
					$this->session_auth_check($session_check_arr,$username,$password);
				}
			} 
			else { 
				// if the user is not logging in, and not yet logged in, 
				// then they should just be seeing the login view again
				if($this->session->userdata("is_loggedin") != "true") {
					$data['title'] = PORTAL_TITLE_PREFIX . 'Login';
					$data['message_class'] = 'error';
					$data['message'] = validation_errors();
					
					$this->load->view('auth/login', $data);
				}
				else {
					redirect($this->destination_after_login); // Redirect logged in user to inbox
				}
			}
		}
	}
	
	// Log the user out
	function logout() {
		log_message('debug', 'Logging out '.$this->session->userdata('username'). ' via auth::logout');
		
		$uri_before_logout = $this->session->flashdata('uri_before_logout'); //if we were redirected here by the User::logout() function, it will have set this
		if(empty($uri_before_logout)) $uri_before_logout = $_SERVER['HTTP_REFERER'];		
		
		User::log_out();

		//even though we're not redirect to the auth page for PIV users, this will still be accessible if user manually goes to the auth page
		$this->session->set_flashdata('uri_before_logout', $uri_before_logout); 
		
		
		if(USE_CAC_AUTH === TRUE || USE_PIV_AUTH === TRUE) {			
			$data["title"] = PORTAL_TITLE_PREFIX . "Logout";
			$data['message_class'] = 'error';
			if(USE_CAC_AUTH === TRUE && USE_PIV_AUTH !== TRUE) {
				$data['message'] = '<p>Your session has been logged out. However, to ensure a complete logout please close ALL browser windows and remove your CAC.</p>';
			}
			else if(USE_CAC_AUTH !== TRUE && USE_PIV_AUTH === TRUE) {
				$data['message'] = '<p>Your session has been logged out. However, to ensure a complete logout please close ALL browser windows and remove your PIV.</p>';
			}
			else {
				$data['message'] = '<p>Your session has been logged out. However, to ensure a complete logout please close ALL browser windows and remove your CAC or PIV.</p>';
			}
			
			$data['message'] .= '<p align="center" >If you would like to log back in, click below:</p>';
			$data['message'] .= '<div id="login_btn" style="vertical-align:center; text-align:center; "><button type="button" style="width:55px;" class="minimal" id="login_btn" name="login" aria-hidden="false">Login</button></div>';
			
//uncomment to have the logout page automatically redirect back to login for testing purposes
/*			if(is_on_local() && !IS_AJAX) $data['message'] .= '<script>$(document).ready(setTimeout(function(){window.location.href = "'.site_url('?'.$this->security->get_csrf_token_name().'='.$this->security->get_csrf_hash()).'";}, 10000));</script>';  */
			return $this->load->view("system_message",$data);
		}
		
		redirect("auth"); //redirect to login page
	}
	
	//sets variable that determines the mailbox group used when utilizing the inbox
	public function select_mailbox($mbox) {
		$mailbox_group = base64_decode(rawurldecode($mbox));
		$mailbox_group_cn = $mailbox_group; //default the display name to the mailbox name - we'll update it with the real display name if we can find it
					
		$ldapconfig['user'] = $this->session->userdata('username');
		$ldapconfig['pwd'] = $this->encrypt->decode($this->session->userdata('ep'));
		$this->load->library('ldap',$ldapconfig);
		
		//set mailbox group display name, if applicable
		$ldap_lookup = $this->ldap->search(NULL,1,array('cn'),'(&(ObjectClass=groupOfNames)(ou=' . $mailbox_group . '))',LDAP_BASE_DOMAIN);
		if(isset($ldap_lookup[0]['cn'])) $mailbox_group_cn = $ldap_lookup[0]['cn'];

		$this->session->set_userdata(compact('mailbox_group', 'mailbox_group_cn'));	//set both vars at once to reduce db writes
		
		redirect($this->destination_after_login);
	}
	
	/* This function checks the audit log of previous logins for a username.
	 * It will return an array containing the number of login attempts since the last successful login,
	 * the time of the last failed login, the number of distinct ip addresses that have failed in the last 24 hours,
	 * and the total number of failed logins for this account accross all ip address for the last 24 hours.
	 * Returns FALSE on failure (either to connect to database or query it).
	 */
	protected function check_login_attempts($username) {
			$stmt = $this->db->query(
			"SELECT username,COUNT(success) AS failed_attempts,
			(SELECT TOP(1) login_time FROM logins WHERE (success = 0) AND (username = " . $this->db->escape($username) . ") ORDER BY login_time DESC) AS last_failed_login_time,
			(SELECT COUNT(success) FROM logins WHERE (success = 0) AND (username = " . $this->db->escape($username) . ") AND (login_time > (datediff(ss, '19700101', GetUtcDate()) - 86400))) AS failed_logins_last_day,
			(SELECT COUNT(DISTINCT ip_address) FROM logins WHERE (success = 0) AND (username = " . $this->db->escape($username) . ") AND (login_time > (datediff(ss, '19700101', GetUtcDate()) - 86400))) AS distinct_failed_ips_last_day
            FROM logins
            WHERE (username = " . $this->db->escape($username) . ") AND (login_time >
                  (SELECT TOP (1) login_time
                   FROM logins AS logins_1
                   WHERE (success = 1) AND (username = " . $this->db->escape($username) . ")
                   ORDER BY login_time DESC))
            GROUP BY username");
			if($stmt === FALSE) { return FALSE; }
			
			$row = $stmt->row_array();
			if(!empty($row)){ return $row; }
			return TRUE;
	}
	
	protected function check_user_sessions($username) {
		$stmt = $this->db->query("SELECT TOP (1) session_id, ip_address
				FROM sessions
				WHERE (user_data LIKE '%' + " . $this->db->escape($username) . "+ '%')
				ORDER BY last_activity");
		if($stmt === FALSE) { return FALSE; }
	
		$row = $stmt->row_array();
		if(!empty($row)) { return $row; }
		
		return TRUE;
	}
	
	protected function session_auth_check($session_check_arr,$username,$password) {
		if(is_array($session_check_arr)) { 
			if($session_check_arr["session_id"] != session_id()) {			
				if($session_check_arr["ip_address"] != $_SERVER['REMOTE_ADDR']) {
					$data["title"] = "Login";
					$data['message_class'] = 'error';
					$data["message"] = "Your account has an active session somewhere else. If you have recently logged off, wait a few minutes before trying to log in again.";
					$this->load->view('auth/login', $data);
				}
				else {
					$destroyed = $this->destroy_old_sessions($username);
					if($destroyed) { $this->ldap_auth($username,$password); }
					else { 
						$data["title"] = "Login";
						$data['message_class'] = 'error';
						$data["message"] = "Problem logging account out of previous session. Please wait for prior session to expire then try again.";
						$this->load->view('auth/login', $data);
					}
				}
			}
			else { $this->ldap_auth($username,$password); } 
		} 
		else { $this->ldap_auth($username,$password); }
	}
	
	protected function destroy_old_sessions($username) {
		$stmt = $this->db->query("UPDATE sessions SET user_data='' WHERE user_data LIKE '%' + " . $this->db->escape($username) . " + '%'");
		return $stmt;
	}
	
	protected function ldap_auth($username, $password, $usingCert = NULL) {
		log_message('debug', 'Logging in '.$username. ' via auth::ldap_auth');
		
		$this->load->library('audit');
		
		//use provided credentials to attempt LDAP connection
		$this->load->library('ldap',array('user'=>$username,'pwd'=>$password));
		
		//this will be true only if the provided credentials were used to bind successfully
		if($this->ldap->authenticated()) {
			
			//get some user information from LDAP and store in session
			$ldap_lookup = $this->ldap->search(null,1,array('uid','mail','cn'),"(&(uid=".$username."))");
			
			$userdata = array(	'is_loggedin' => 'true', 
								'just_logged_in' => TRUE, //note the fact that the user just logged in within session
							 	'username' => $username, 
								'ep' => $this->encrypt->encode($password),
								'user_mail' => $ldap_lookup[0]['mail']);
			
			//find the user's display name, if it existsd		
			if(!empty($ldap_lookup[0]['cn'])){ 
				$userdata['user_cn'] = $ldap_lookup[0]['cn']; 
			}
			
			//check if the user has any group mailboxes
			$group_mailboxes = $this->ldap->get_group_membership('uid='.$username.','.LDAP_ACCOUNTS_DN);
			$userdata['group_mailboxes'] = $group_mailboxes;
			
			$this->session->set_userdata($userdata); //set multiple userdata at once to reduce database writes
			
			session_regenerate_id();
			
			//log the login event
			$this->audit->log_event('login',array(session_id(),$username,$_SERVER['HTTP_X_REAL_IP'],date('U'),TRUE,NULL));

//uncomment to skip the banner when when testing the session
#			if(is_on_local()) redirect($this->destination_after_login);
			
			//check if we are hiding their personal inbox
			$hidden_check = $this->ldap->search(NULL,1,array('employeeType'),'(uid='.$username.')');
			if(isset($hidden_check[0]['employeetype']) &&  $hidden_check[0]['employeetype'] == 'mailboxhidden') { 
				$personal_hidden = TRUE;
				$this->session->set_userdata('hide_personal_mailbox',TRUE);
			}
			else { $personal_hidden = FALSE; }
			
			//figure out what to load for group mailboxes/personal mailboxes
			if(count($group_mailboxes) > 0) {				
				$data['title'] = PORTAL_TITLE_PREFIX . WARNING_TEXT_TITLE;
				$data['group_mailboxes'] = $group_mailboxes;
				$this->session->set_userdata('group_mailbox_list',$group_mailboxes);
				
				//load select mailbox view if needed
				if(count($group_mailboxes) > 1 || (count($group_mailboxes) == 1 && !$personal_hidden)) {
					$data['hide_personal_mailbox'] = $personal_hidden;
					$this->load->view('auth/selectmailbox',$data);
				}
				else { 
					$keys = array_keys($group_mailboxes);
					$this->select_mailbox(rawurlencode(base64_encode($keys[0])));
					if(!$usingCert || is_null($usingCert)) { redirect($this->destination_after_login); }
				}
			}
			else if(count($group_mailboxes) <= 0 && $personal_hidden) {
				session_destroy(); //note that as of CI 3.0, this destroys the CI session along with the PHP session
				$data = array(
					'title' => PORTAL_TITLE_PREFIX . 'Configuration Error',
					'message' => 'Your account is not configured for personal use, and is not a member of any groups. Please contact an administrator to resolve the situation.',
				);
				$this->load->view('system_message',$data);
			}
			else {
				if(!$usingCert || is_null($usingCert)) { redirect($this->destination_after_login); }
				else {
					$data['title'] = PORTAL_TITLE_PREFIX . WARNING_TEXT_TITLE;
					$data['message'] = '<div id="dodwarning">'. WARNING_TEXT . '</div>' .
							'<script type="text/javascript">
								var item = $("#dodwarning").append(\'<div id="okbtn" style="margin:0 auto; text-align:center"><button class="minimal" id="ok_btn">OK</button></div>\');
								$("#okbtn").click(function() { document.location.href = \'' .$this->destination_after_login.'\'; });
								$("#dodwarning").css("position","absolute");
								$("#dodwarning").css("left","50%");
								$("#dodwarning").css("margin-left","-290px");
							</script>';
					$this->load->view('system_message',$data);
				}
			}
		}
		else {
			$msg = isset($failure_reason) ? $failure_reason : $this->ldap->auth_error;
			//log the login event
			$this->audit->log_event('login',array(session_id(),$username,$_SERVER['HTTP_X_REAL_IP'],date('U'),FALSE,$msg));
			if(!is_null($usingCert) && $usingCert) { 
				$data['title'] = 'Login Error'; 
				$data['message_class'] = 'error';
				$data['message'] = 'Login failed because a connection with the authentication server could not be established. Please contact an administrator for assistance.'; 
				$this->load->view('system_message', $data); 
			} else{
				$data['title'] = 'Login';	 
				switch($msg){
					case 'Invalid credentials':
						$data['message_class'] = 'error';
						$data['message'] = 'Login Failed. Invalid username and/or password.';
						break;
					default:
						$data['message_class'] = 'error';
						$data['message'] = 'Login failed because a connection with the authentication server could not be established. Please contact an administrator for assistance.';
				}
				$this->load->view('auth/login', $data);
			} 
		}
	}
	
	
	/* This function extracts information from the SSL certificate provided to the server.
	 * It assumes that the certificate is stored in the $_SERVER['HTTP_CERT'] variable, that the CN for the cert
	 * contains at the very least the first and last name, and a unique identication number delimited by periods.
	 * These assumptions should be valid for all DoD Common Access Cards.
	 */
	protected function cert_auth() {
		$cert = $this->clean_cert_string($_SERVER['HTTP_CERT']);
		$parsed_cert = openssl_x509_parse($cert);
		$cn_arr = explode(".",$parsed_cert["subject"]["CN"]);
		$last = $cn_arr[0];
		$first = $cn_arr[1];
		if(count($cn_arr) == 4) {
			//check if its suffix or last name, if they are more than seven generations with the same name they will just have to correct it after registration
			//since the system will assume that it is their last name
			if(
				strtolower($cn_arr[2]) == 'jr' || strtolower($cn_arr[2]) == 'sr' || strtolower($cn_arr[2]) == 'ii' || strtolower($cn_arr[2]) == 'iii' ||
				strtolower($cn_arr[2]) == 'iv' || strtolower($cn_arr[2]) == 'v' || strtolower($cn_arr[2]) == 'vi' || strtolower($cn_arr[2]) == 'vii'
			) {
				$suffix = $cn_arr[2];
				$middle = '';
			}
			else { 
				$middle = $cn_arr[2];
				$suffix = '';
			}
			$edipi = $cn_arr[3];
		}
		else if(count($cn_arr) == 5) {
			$middle = $cn_arr[2];
			$suffix = $cn_arr[3];
			$edipi = $cn_arr[4];
		}
		else { $middle = ""; $edipi = $cn_arr[2]; }
		
		if(is_numeric($edipi)) {
			$stmt = $this->db->query("SELECT TOP (1) user_name,user_ep
					FROM users
					WHERE  (user_deleted_flag=0) AND (user_edipi =" . $this->db->escape($edipi) . ")");
			if($stmt === FALSE) { return FALSE; }
			else {
				$row = $stmt->row_array();
				if($stmt->num_rows() == 1) { return $row; }
				else { return FALSE; }
			}
		}
		else { return FALSE; }	
	}
	
	/* This function extracts information from the SSL certificate provided to the server.
	 * It assumes that the certificate is stored in the $_SERVER['HTTP_CERT'] variable, that the CN for the cert
	 * contains at the very least the first and last name, and a unique identication number a space after the name.
	 * These assumptions should be valid for all VA PIV Cards.
	 */
	protected function piv_cert_auth() {
		$cert = $this->clean_cert_string($_SERVER['HTTP_CERT']);
		$parsed_cert = openssl_x509_parse($cert);
		$cn_arr = explode(' ',trim($parsed_cert['subject']['CN']));
		$first = $cn_arr[0];
		$middle = $suffix = '';
		if(count($cn_arr) === 4) {
			if(preg_match('/\(.*?\)/',$cn_arr[3]) === 0) { //if there is not something appended in parentheses after the piv id
				//check if its suffix or last name
				if(strtolower($cn_arr[2]) == 'jr' || strtolower($cn_arr[2]) == 'sr' || $this->is_roman_numeric($cn_arr[2])) {
					$suffix = $cn_arr[2];
					$last = $cn_arr[1];
				}
				else {
					$middle = $cn_arr[1];
					$last = $cn_arr[2];
					$suffix = '';
				}
				$piv_id = $cn_arr[3];
			}
			else {
				$last = $cn_arr[1];
				$piv_id = $cn_arr[2];
			}
		}
		else if(count($cn_arr) === 5) {
			if(preg_match('/\(.*?\)/',$cn_arr[4]) === 0) { //if there is not something appended in parentheses after the piv id
				$middle = $cn_arr[1];
				$last = $cn_arr[2];
				$suffix = $cn_arr[3];
				$piv_id = $cn_arr[4];
			}
			else {
				//check if its suffix or last name
				if(strtolower($cn_arr[2]) == 'jr' || strtolower($cn_arr[2]) == 'sr' || $this->is_roman_numeric($cn_arr[2])) {
					$suffix = $cn_arr[2];
					$last = $cn_arr[1];
				}
				else {
					$middle = $cn_arr[1];
					$last = $cn_arr[2];
					$suffix = '';
				}
				$piv_id = $cn_arr[3];
			}
		}
		else if(count($cn_arr) == 3) {
			$last = $cn_arr[1];
			$piv_id = $cn_arr[2];
		}
		
		if(is_numeric($piv_id)) {		
			$stmt = $this->db->query("SELECT TOP (1) user_name,user_ep FROM users WHERE(user_deleted_flag=0) AND (user_piv_id=" . $this->db->escape($piv_id) . ")");
			if($stmt){
				$user = $stmt->row_array();
				if(!empty($user)) return $user;
			}
		}
		
		return false;
	}
	
	/* This function determines if a string is a valid roman numeral. This is used in parsing names from CAC certs.
	 */
	protected function is_roman_numeric($str) {
		preg_match ('/^M{0,4}(CM|CD|D?C{0,3})(XC|XL|L?X{0,3})(IX|IV|V?I{0,3})$/',$str,$matches);
		if(count($matches) > 0) { return TRUE; }
		return FALSE;
	}
	
	/* This function cleans up the certificate string provided by the server so that it can be parsed by OpenSSL.
	 */
	protected function clean_cert_string($cert) {
		$output = preg_replace("/-----[A-Z]+\sCERTIFICATE-----/","",$cert);
		$output = preg_replace("!\s+!"," ",$output);
		$cert = str_replace(" ","\n",$output);
		$cert = "-----BEGIN CERTIFICATE-----\n".trim($cert)."\n-----END CERTIFICATE-----";
		return $cert;
	}
}
