<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models-user
*/

/**
* @package direct-as-a-service
* @subpackage models
*/
class User extends Entity {
	static $table = 'users';
	static $primary_key = 'user_id';	
	
	protected $_admin_ldap_connection;
	protected $_ldap_connection;
	protected $_ldap_info;
	protected $_navigation_tabs;
	protected $_permissions;
	
	protected static $_relationships = array( 'mailbox' => array( 'type' => 'belongs_to', 'related_foreign_key' => 'username', 'key_for_relationship' => 'name' ));
	
	protected $_property_validation_rules = array( 'user_org_id' => 'whole_number',
												   'user_ext_mail' => 'string_like_an_email_address',
												   'user_ep' => 'nonempty_string' );		
	
	protected $_readonly_fields = array( 'user_created_time' ); 
	
	

////////////////////////////////
// INSTANCE METHODS
////////////////////////////////
	
	public function activate(){
		if($this->is_active) return true; //if this ist active, we're already done		
		
		$mailbox = $this->mailbox;
		$mailbox_activated = false;
		if(Mailbox::is_an_entity($mailbox) && !$mailbox->is_active){
			$mailbox->is_active = true;
			$success = $mailbox_activated = $mailbox->save();
			if(!$success) return $this->error->warning('Cannot activate '.$this->describe().'; '.$mailbox->describe.' could not be activated.');
		}
		
		$this->_ldap_info = null; //force a reload of our ldap information when we're done
		
		if(!is_resource($this->admin_ldap_connection))return $this->error->warning('Cannot activate '.$this->describe().' without an LDAP connection');			
		$success = ldap_rename($this->admin_ldap_connection, $this->disabled_dn,'uid=' . $this->username, LDAP_ACCOUNT_GROUP, TRUE);
		if(!$success){
			$this->error->warning('Unable to activate '.$this->describe().': LDAP says '.$this->error->describe(ldap_error($this->admin_ldap_connection)));
			if($mailbox_activated){
				$mailbox->is_active = false;
				if(!$mailbox->save()) $this->error->warning('Unable to deactivate '.$mailbox->describe().'; please manually update the database.');
			}
		}
		return $success;
	}	
	
	public function deactivate(){
		if(!$this->is_active) return true; //if this isn't active, we're already done		
		
		$mailbox_deactivated = false;
		$mailbox = $this->mailbox;
		if(Mailbox::is_an_entity($mailbox) && $mailbox->is_active){
			$mailbox->is_active = false;
			$success = $mailbox_deactivated = $mailbox->save();
			if(!$success) return $this->error->warning('Cannot deactivate '.$this->describe().'; '.$mailbox->describe.' could not be deactivated.');
		}
		
		$this->_ldap_info = null; //force a reload of our ldap information when we're done
		
		if(!is_resource($this->ldap_connection))return $this->error->warning('Cannot deactivate '.$this->describe().' without an LDAP connection');			
		$success = ldap_rename($this->ldap_connection, $this->dn(),'uid=' . $this->username,LDAP_DISABLED_ACCOUNT_GROUP, TRUE);
		if(!$success){
			$this->error->warning('Unable to deactivate '.$this->describe().': LDAP says '.$this->error->describe(ldap_error($this->ldap_connection)));
			if($mailbox_deactivated){
				$mailbox->is_active = true;
				if(!$mailbox->save()) $this->error->warning('Unable to reactivate '.$mailbox->describe().'; please manually update the database.');
			}
		}
		return $success;
	}	
	
	function is_admin(){
		if($this->property_is_empty('permissions') || !is_array($this->permissions)) return false;
		if(!array_key_exists('API', $this->permissions)) return false;
		return !empty($this->permissions['API']['admins']);
	}
	
////////////////////////////////
// GETTERS
//////////////////////////////

	
	function active(){
		if($this->property_is_empty('ldap_info') || !is_object($this->ldap_info)) return false; //if we can't access LDAP, default user to inactive
		return (isset($this->ldap_info->disabled) && !$this->ldap_info->disabled); //yargh, this is why using generic objects as containers is difficult - no way to just test for empty, can't use property_exists().
	}

	function is_active(){
		return $this->active(); //currently debating the best wording, so allow both for now
	}


	public function dn(){
		return 'uid='.$this->username.','.LDAP_ACCOUNT_GROUP;
	}
	
	public function disabled_dn(){
		return 'uid='.$this->username.','.LDAP_DISABLED_ACCOUNT_GROUP;
	}
	
	public function email_address(){
		return $this->username.'@'.DIRECT_DOMAIN;
	}

	public function admin_ldap_connection() {
		if(!isset($this->_admin_ldap_connection)){
			$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
			if($ldap_conn === FALSE) return $this->error->warning('Could not connect to LDAP host: '.$this->error->describe(LDAP_HOSTNAME));
			
			if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
			if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
			
			//get the currently logged user's password from the database to bind to LDAP (NOT the user for this object);
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$this->_admin_ldap_connection = $ldap_conn;
		}
		return $this->_admin_ldap_connection;
	}	
	
	public function ldap_connection() {
		if(!isset($this->_ldap_connection)){
			$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
			if($ldap_conn === FALSE) return $this->error->warning('Could not connect to LDAP host: '.$this->error->describe(LDAP_HOSTNAME));
			
			if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
			if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
			
			//get the currently logged user's password from the database to bind to LDAP (NOT the user for this object);
			$logged_in_user = User::find_from_session();			
			$ldap_bind = ldap_bind($ldap_conn, $logged_in_user->dn, get_instance()->encrypt->decode($logged_in_user->user_ep));				
#			if(is_on_local()) $ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if(!$ldap_bind)	return $this->error->warning('Could not connect to LDAP; please check your config file to ensure that the LDAP configuration is correct');
			
			$this->_ldap_connection = $ldap_conn;
		}
		return $this->_ldap_connection;
	}	
	
	function ldap_info(){
		if(!isset($this->_ldap_info) && !$this->property_is_empty('username')){
			$this->_ldap_info = get_instance()->usersmodel->get_user_from_username($this->username); //look for info in active users
			if(is_array($this->_ldap_info) && empty($this->_ldap_info))
				$this->_ldap_info = get_instance()->usersmodel->get_user_from_username($this->username, FALSE); //look for info inactive users
		}
		return $this->_ldap_info;
	}
	
	public function navigation_tabs(){
		if(!isset($this->_navigation_tabs)){
			$this->_navigation_tabs = get_instance()->permissions->set_tab_access_from_permissions($this->permissions);
		}
		return $this->_navigation_tabs;
	}

	public function permissions() {
		if(!isset($this->_permissions)){

			get_instance()->load->model('rolesmodel');

			if(!$this->ldap_connection) return false;
	
			//search for groups that this user is in (added !(mail=*) to filter out mail groups)		
			$search = ldap_search($this->ldap_connection, LDAP_BASE_RDN, '(&(member=' . $this->dn . ')(!(mail=*)))', array('dn'));
			$entries = ldap_get_entries($this->ldap_connection, $search);
			//set default permissions to lowest possible
			$permissions = array('API'=>array('admins'=>FALSE),'Application'=>NULL,'Registered'=>TRUE,'Role'=>array());


			foreach($entries as $entry) {
				//if they are in the API admin group, set API admin permissions
				if($entry['dn'] === LDAP_API_ADMIN_GROUP) { 
					$permissions['API']['admins'] = TRUE;
				}
				//if the value is a role
				elseif(!empty($entry['dn']) && string_contains(LDAP_ROLES_GROUP, $entry['dn'])){
					$exploded_dn_values = ldap_explode_dn($entry['dn'],1);
					$permissions['Role'][]=$exploded_dn_values[0];
				}
				//everything else will be an application group, so we will have to check the group's parent to see what app it is
				elseif(isset($entry['dn'])) {
					$exploded_dn = ldap_explode_dn($entry['dn'],0); //explode dn so we can find parent dn
					unset($exploded_dn['count'], $exploded_dn[0]); //remove values unncessary for parent dn
					$parent_dn = implode(',',$exploded_dn); //impode array to get parent dn
					
					//get parent entry and grab the uid (which corresponds to the application's unique id in the database)
					$parent_search = ldap_read($this->ldap_connection, $parent_dn, '(objectclass=*)', array('uid'));
					$parent_entry = ldap_get_entries($this->ldap_connection, $parent_search);
					
					//if the parent entry is an application, it should have a uid attribute, ignore anything else
					if(isset($parent_entry[0]['uid'])) {
						$uid = $parent_entry[0]['uid'][0];
						
						//get raw values of dn parts so we can set group name  in permissions
						$exploded_dn_values = ldap_explode_dn($entry['dn'],1); 
						$permissions['Application'][$uid][$exploded_dn_values[0]] = TRUE;
					}
				}
			}

			$permissions['Permission'] = get_instance()->rolesmodel->get_permissions_from_roles($permissions['Role']);
			$this->_permissions = $permissions;
		}
		return $this->_permissions;
	}
	//attempts to give the user's first name (or if added in future, nickname)
	//will revert first to cn if first name is unavailable, and username as a last resort
	public function preferred_name(){
		if(is_object($this->ldap_info)){
			if(!empty($this->ldap_info->givenname)) return $this->ldap_info->givenname;
			if(!empty($this->ldap_info->cn)) return $this->ldap_info->cn;
		}
		return $this->username;	
	}

	
	public function readonly_fields(){
		$readonly_fields = $this->_merge_with_parent_array('readonly_fields');
		if(isset($this->id)){
			$readonly_fields[] = 'username';
			$readonly_fields[] = 'user_org_id';
		}
		
		return $readonly_fields;
    }
	
//////////////////////////////////////
// SETTERS
//////////////////////////////////////	

	public function set_username($value, $offset = 0){
		if(isset($this->id)) return $this->error->warning('Cannot change the username for '.$this->describe().', which has already been saved to the database.', $offset+1);
		if(!$this->is->nonempty_string($value)) return $this->error->property_value_should_be_a_nonempty_string('username', get_class($this), $value, $offset+1);
		if(!$this->is->string_like_an_email_address($value.'@'.DIRECT_DOMAIN)) return $this->error->property_value_should_be_a_valid_first_half_of_an_email_address('username', get_class($this), $value, $offset+1);
		#if(User::exists(array('username' => $value ))) return $this->error->warning('Cannot set '.get_class($this).'::$username to '.$this->error->describe($value).'; this username is already in use', $offset+1);
		//check against all mailboxes, including groups
		if(Mailbox::exists(array('name' => $value ))) return $this->error->warning('Cannot set '.get_class($this).'::$username to '.$this->error->describe($value).'; this username is already in use', $offset+1);
		
		return $this->_set_field_value('username', $value, $offset+1);
	}

	
//////////////////////////////////////
// DATA MANAGEMENT
//////////////////////////////////////

	protected function _run_before_create(){
		$success = $this->_set_field_value('user_created_time', time(), $error_offset = 0, $override_validation = TRUE);
		return $success;
	}
	
	protected function _values_are_valid_for_create_and_update(){
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		$required_fields = array( 'username', 'user_org_id', 'user_created_time', 'user_ep' );
		foreach($required_fields as $required_field){
			if($this->property_is_empty($required_field)) 
				return $this->error->warning(get_class($this).'::$'.$required_field.' is a required field, and must be set before saving'); 
		}
		return true;
	}	

#TODO - ADD LDAP CODE HERE INSTEAD OF NEEDING TO RUN USERSMODEL::CREATE_USER
	protected function _run_after_create(){
		$mailbox = Mailbox::create( array( 'name' => $this->username, 'is_active' => true ) );
		if(!Mailbox::is_an_entity($mailbox)){
			if(User::delete($this->id))
				return $this->error->warning('Unable to create a mailbox for '.$this->describe().'; removing user entry from the database.');
			return $this->error->warning('Unable to create a mailbox for '.$this->describe().', but the user remains in the database.  Please manually remove '.$this->describe().' from the database');
		}
		
		return true;
	} 
	
	protected static function _run_before_delete(&$user){ 
		$mailbox = $user->mailbox;
		if(Mailbox::is_an_entity($mailbox)){
			if(!Mailbox::delete($mailbox->id)) return $this->error->warning('Could not delete '.$mailbox->describe());
		}
		
		return true; 
	}
	

/////////////////////////////////
// STATIC METHODS
/////////////////////////////////

	public static function find_from_session(){
		$user_org_id = static::organization_id_from_session();
		if(empty($user_org_id)) return false; //it's OK for the user to not be in the session - don't want to trigger errors if we just need to have the user log in 
		#if(empty($user_org_id)) return $this->error->warning('Could not identify user; the user\'s organization ID was not found in the session.');

		if(!get_instance()->is->nonzero_unsigned_integer($user_org_id)) return $this->error->should_be_a_user_organization_id($user_org_id); //we do actually want to know if there's some weird invalid ID in the system
		
		$user = User::find_one( compact('user_org_id') );
# don't give an error - unregistered users will be in the session but not yet in the system.  ultimately, it would be nice to have a different way of doing this, but this will do for now.
#		if(!User::is_an_entity($user)) return $CI->error->warning('No user with an organization ID of '.$CI->error->describe($user_org_id).' was found in the system.'); 
		return $user;
	}
	
	public static function new_username($first, $last) {
		//sanitize username input to remove any non-alphanumeric characters
		$first = preg_replace("/[^\p{L}\p{M}\p{N}]/u", '', $first);
		$last = preg_replace("/[^\p{L}\p{M}\p{N}]/u", '', $last);
		
		//theoretically we could hit max username lengths for last names, so don't allow more than 100 characters total
		$last = mb_substr($last,0,99);
		
		if(empty($first) && empty($last)) return FALSE; //we can't do anything without a first and last name
			
		$username = mb_strtolower($first . '.' . $last);
		if(!User::exists(compact('username'))) return $username;
	
		for($i=2; $i <= 999; $i++){
			if(!User::exists(array('username' => $username.$i))) return  $username.$i;
		}
		
		return FALSE; //returns FALSE if more than 999 users have the same combo of first initial / last name
	}	
	
	public static function organization_id_from_session(){
		$CI = get_instance();
		return $CI->encrypt->decode($CI->session->userdata('user_id'));
	}
	
}
