<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models
*/

/** */
require_once 'Mail.php';
require_once 'Mail/mime.php';
require_once 'Mail/mimeDecode.php';  

require_library('mime_generator');
require_model('entity');

/**
* @package direct-as-a-service
* @subpackage models
*/
class Message extends Entity {
	
//////////////////
// STATIC VARS
///////////////////	
	static $database_group = 'mail_db';
	static $table = 'messages';	
	protected static $_relationships = array( 'disclosure' => array('type' => 'has_many', 'foreign_key' => 'messages_table_id'),
											  'attachment_disclosure' => array('type' => 'has_many', 'foreign_key' => 'messages_table_id', 'condition' => 'accounting_disclosure.hash IS NOT NULL'),
											  'message_disclosure' => array('type' => 'has_many', 'model' => 'disclosure',  'foreign_key' => 'messages_table_id', 'condition' => 'accounting_disclosure.hash IS NULL'),
											  'mailbox' => array('type' => 'belongs_to'),
						 					  'folder' => array('type' => 'belongs_to'),
											  'original_sender' => array('model' => 'mailbox', 
											  							 'type' => 'belongs_to', 
																		 'related_foreign_key' => 'original_sender_id'),
											  'patient' => array('type' => 'has_many', 'order_by' => 'family_name ASC, given_name ASC, date_of_birth ASC, organization ASC'));
	
	public static $parts = array( 'flags' => array( 'seen',
													'draft',
													'sent',
													'archived',
													'flags',
													),
													
								  'headers' => array( 'sender',
								  					  'to',
													  'cc',
													  'bcc',
													  'subject',
													  'timestamp',
													  'size',
													  'seen',
													  'flags',
													  'draft',
								  					  'sent',
													  'archived',
													  'attachments',
													  'headers',
													  'priority',
								  					  'folder_id'),
								  'raw_mime' => array('raw_mime'),
								  'xdm_attachments' => array('xdm_attachments'));
									

///////////////////////////
// INSTANCE VARS
///////////////////////////
	protected $_body; //either the html or the plain value, depending on mailtype
	protected $_attachment_files; //array of filename => binary_strings
	protected $_inline_attachment_files;
	protected $_mime_generator; //Mime_generator object for this message
	
	//each of these fields affect the mime - when they're changed, the mime needs to be regenerated
	protected $mime_fields = array( 'sender',
									'to',
									'cc',
									'bcc',
									'attachments',
									'subject',
									'body',
									//'flags', //I don't think this actually affects the mime, right?  this is just within our database
									'mailtype',
									'priority' );
	
	//readonly fields which are derived from the mime and may not be set by developers								
	protected $mime_derived_fields = array( 'attachments',
											'headers',
											'html',
											'plain',
											'raw_mime',
											'recipients',
											'size' );	

	protected $_readonly_fields = array( 'draft', //will be set internally for new messages
										 'sent', //will be set internally when send() is called
										 'message_id', //will be set by the RI for incoming messages
										 'timestamp', //will be generated by the model when fields that affect the mime are changed, and when the message is sent
										); 	

	//validation rules for settable properties on the method - these correspond to Validator methods		
	protected $_property_validation_rules = array('priority' => 'nonzero_unsigned_integer',
												  'to' => 'comma_and_whitespace_separated_list_of_string_like_an_email_addresses', //MG, make your peace with the poor linguistic syntax of this rule.
												  'cc' => 'comma_and_whitespace_separated_list_of_string_like_an_email_addresses', //It is too much trouble to change
												  'bcc' => 'comma_and_whitespace_separated_list_of_string_like_an_email_addresses',
												  'subject' => 'string',
												  'seen' => 'boolean',
												  'archived' => 'boolean',
												  'protected_data' => 'unsigned_integer');






///////////////////
// INSTANCE
///////////////////		
	
	/**
	* @param string
	* @param string
	* @return boolean True on success
	*/
	public function add_attachment($name, $binary_string){
		if(!$this->is->string_like_a_file_path($name)) return $this->error->should_be_a_string_like_a_file_path($name);
		//Removed check that prevents empty files from being sent, email allows this and so should we
		//if(!$this->is->nonempty_string($binary_string)) return $this->error->should_be_a_nonempty_string($binary_string);
		
		if(!$this->draft()) return $this->error->warning('Cannot attach '.$name.' to '.$this->describe().'; attachments can only be added to drafts.');
		
		if(!static::attachment_has_valid_extension($name)) return $this->error->warning('Cannot attach '.$name.' to '.$this->describe().'; invalid file extension.');
		
		if(!$this->attachment_is_within_size_limit($name, $binary_string)){
			return $this->error->warning('Adding file '.$name.' ('.byte_format(string_length_in_bytes($binary_string)).') to '.$this->describe().' would cause the message to exceed the limit of '.byte_format(DIRECT_ATTACHMENT_SIZE_LIMIT));
		}
		
#TODO - WE NEED A WAY TO SET ALL ATTACHMENTS/REMOVE ATTACHMENTS IF NECESSARY
		$attachment_files = $this->attachment_files;
		if(array_key_exists($name, $attachment_files)){
			return $this->error->warning('Cannot attach '.$name.' to '.$this->describe().'; '.$name.' has already been attached to this message.');
		}
		
#TODO - CAN WE DO ANY KIND OF VALIDATION OF FILETYPE BASED ON THE BINARY STRING?  WOULD BE NICE NOT TO JUST TRUST THE DEVELOPER AND/OR $_FILES ARRAY		
		$attachment_files[$name] = $binary_string;
		$this->_attachment_files = $attachment_files;
		
		$this->reload_mime(); //force the mime to regenerate the next time we call on a mime-derived field - this will ensure that it includes the newly attached file
		
		return true;
	}

	public function attachments_matching_hash($hash){
		$attachments_matching_hash = array();
		foreach($this->attachment_files as $name => $binary_string){
			if(sha1($binary_string) == $hash)
				$attachments_matching_hash[$name] = $binary_string;
		}
		return $attachments_matching_hash;
	}
	
	
	public function attachment_is_within_size_limit($name, $binary_string){
		$new_total = $this->size + string_length_in_bytes($binary_string);
		return $new_total <= DIRECT_ATTACHMENT_SIZE_LIMIT;
	}

	/**
	* Archives this message.
	* @return boolean True on success
	*/										  
	public function archive(){
		$this->archived = true;
		return $this->save();
	}											  


	/**
	* True if the message belongs to the given mailbox.
	* @todo Make this a generalized relationship function for objects that belong to something?
	* @todo Allow mailbox ids instead of just mailbox objects?
	* @param Mailbox
	* @return boolean
	*/
	public function belongs_to_mailbox($mailbox){
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->should_be_a_mailbox_entity($mailbox);
		$related_foreign_key = static::related_foreign_key('mailbox');
		
		//if we have the mailbox id loaded, use it
		if(isset($this->$related_foreign_key))
			return ($this->$related_foreign_key == $mailbox->id());
		
		//if we have the sender loaded, use it in case we don't have the mailbox id
		if(isset($this->sender))
			return ($this->sender == $mailbox->email_address());
			
		//you're probably going to only run into this if message hasn't been saved, or if you loaded the message using find_part()
		return $this->error->warning('Cannot compare message to mailbox; no mailbox data is loaded into this message entity.  Run $message->load_field_values_from_db() to ensure the necessary values are loaded');
	}	
	
	public function describe(){
		$description = parent::describe();
		if(!isset($this->id)) return $description;
		
		if($this->is_incoming())
			return 'incoming '.$description;
		
		if($this->sent)
			return 'sent '.$description;
			
		return 'draft '.$description;
	}		

	public function has_attachment($name, $binary_string){
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!is_string($binary_string)) return $this->error->should_be_a_string($name);
		
		if(!array_key_exists($name, $this->attachment_files)) return false;
		return $this->attachment_files[$name] == $binary_string;
	}

	public function has_xdm_attachment(){
		//if this has an id, then we determine the attachments from the attachments field.
		if($this->property_is_empty('attachments')) return false;
		$attachments = get_instance()->json->decode($this->attachments, true);
		foreach($attachments as $attachment) {
			if((string_contains($this->subject,'XDM/1.0/DDM') && pathinfo($attachment, PATHINFO_EXTENSION) === 'zip') || (string_contains('xdm',strtolower($attachment)) && pathinfo($attachment, PATHINFO_EXTENSION) === 'zip')) { 
				return true; 
			}
		}
		return false;
	}
	
	
	function has_attachments(){
		//if this has an id, then we determine the attachments from the attachments field.
		if($this->property_is_empty('attachments')) return false;
		$attachments = get_instance()->json->decode($this->attachments, true);
		return !empty($attachments);
	}
	
	
	public function has_required_attachment_disclosures(){
		foreach($this->attachment_files as $name => $binary_string){
			if(Attachment_disclosure::content_requires_disclosure($binary_string)){
				$this->db->order_by('disclosed', 'desc')->limit(1);
				$disclosure = first_element($this->attachment_disclosures( array('hash' => sha1($binary_string))));
				if(!Attachment_disclosure::is_an_entity($disclosure)) return false;
			}
		}
		return true;
	}

	public function has_recipients(){
		//checking actual recipients is not always useful, since it may have an empty json encoded value: '{}'
		return !$this->property_is_empty('to') || !$this->property_is_empty('cc') || !$this->property_is_empty('bcc');		
	}

	/**
	* @return boolean
	*/
	public function is_incoming(){	
		if(!isset($this->id)) return false;//any new messages we make, that don't have ids, are always outgoing messages - incoming messages are created in the RI
		if(!isset($this->draft)) return $this->error->warning("I can't determine if ".$this->describe().' is incoming without a value for draft; run $message->load_field_values() to load all db values into this message');  
		if(!isset($this->sent)) return $this->error->warning("I can't determine if ".$this->describe().' is incoming without a value for sent; run $message->load_field_values() to load all db values into this message'); 
		
		return !$this->is_outgoing();
	}
	
	/**
	* True if this is an outgoing message.
	* This will be true if the message has not yet been saved (all new messages *must* be outgoing), or if the sender matches the mailbox.
	* @return boolean
	*/	
	public function is_outgoing(){
		if(!isset($this->id)) return true;//any new messages we make, that don't have ids, are always outgoing messages - incoming messages are created in the RI
		if(!isset($this->draft)) return $this->error->warning("I can't determine if ".$this->describe().' is outcoming without a value for draft; run $message->load_field_values() to load all db values into this message');  
		if(!isset($this->sent)) return $this->error->warning("I can't determine if ".$this->describe().' is outgoing without a value for sent; run $message->load_field_values() to load all db values into this message'); 
		
		return ($this->draft || $this->sent);
	}
	
	/**
	* True if this message is in the inbox.
	* @return boolean
	*/
	public function is_in_inbox(){
		if(!$this->is_incoming()) return false;
		if(!isset($this->archived)) return $this->error->warning("I can't determine if ".$this->describe().' is an inbox message without a value for archived; run $message->load_field_values() to load all db values into this message'); 
		
		return !$this->archived && $this->property_is_empty('folder_id');
	}
	
	
	
	function mark_as_read(){
		if($this->is_outgoing()) return $this->error->warning( ucfirst($this->describe()).' is outgoing and cannot be marked as read' );
		if($this->seen) return true; //we're done
		$this->seen = true;
		$this->save();
	}
	
	function mark_as_unread(){
		if($this->is_outgoing()) return $this->error->warning( ucfirst($this->describe()).' is outgoing and cannot be marked as read' );
		if(!$this->seen) return true; //we're done
		$this->seen = false;
		$this->save();
	}


	/**
	* A Pear mime object representing this message.
	* Usually we use {@link mime_generator()} to get mime values, but certain of our derived values are easier to obtain using the PEAR Mime decoder class.
	* @return object
	*/
	public function mime($decode_bodies = true){
		$mime = Mail_mimeDecode::decode( array(  'include_bodies' => true,
												 'decode_bodies' => $decode_bodies,
												 'decode_headers' => true,
												 'input' => $this->raw_mime,
												 /*'crlf' => "\n" */)); //apparently we need to use \n if using PHP mail */	
		if(Pear::isError($mime)){
			$this->error->warning('PEAR error encountered while decoding the mime for '.$this->describe().': '.$mime->getMessage());
			return null;
		}

		return $mime;
	}

	/**
	* Moves this message to a custom folder.
	* @return boolean True on success
	*/	
	public function move_to_folder($folder_id){
		if(!Folder::formatted_like_an_id($folder_id))  return $this->error->should_be_a_folder_id($folder_id, 1);		
#TODO - CHECK TO MAKE SURE THAT WE HAVE SEND, DRAFT, ARCHIVED VALUES LOADED
		if($this->sent) return $this->error->warning($this->model_alias.'#'.$this->id().' is a sent message and may not be moved to folder#'.$folder_id, 1);	
		if($this->draft) return $this->error->warning($this->model_alias.'#'.$this->id().' is a draft and may not be moved to folder#'.$folder_id, 1);
		if($this->archived && !$this->restore()) return $this->error->warning($this->model_alias.'#'.$this->id().' could not be moved to folder#'.$folder_id.' because it could not be restored from the archive', 1);
		
		$foreign_key = static::related_foreign_key('folder');
		$this->$foreign_key = $folder_id;
		return $this->save();
	}
	
	/**
	* Moves this message to the inbox.
	*
	* Only received messages may be moved to the inbox - attempting to move drafts or sent messages will trigger an error. 
	*
	* @return boolean True on success
	*/	
	public function move_to_inbox(){
#TODO - CHECK TO MAKE SURE THAT WE HAVE SEND, DRAFT, ARCHIVED VALUES LOADED	
		if($this->sent) return $this->error->warning($this->model_alias.'#'.$this->id().' is a sent message and may not be moved to the inbox', 1);	
		if($this->draft) return $this->error->warning($this->model_alias.'#'.$this->id().' is a draft and may not be moved to the inbox', 1);
		if($this->archived && !$this->restore()) return $this->error->warning('Could not restore archived message '.$this->model_alias.'#'.$this->id());
		$folder_foreign_key = static::related_foreign_key('folder');
		$this->$folder_foreign_key = null;
		return $this->save();
	}
		
	/**
	* Restores an archived message to its previous location (sent messages, drafts, inbox, folder, etc.).
	* @return boolean True on success
	*/
	public function restore(){
		$this->archived = false;
		return $this->save();
	}

	/**
	* Changes the message state to 'sent' and sends the message using the CI Email Library.
	*
	* Note that the raw_mime value is derived using the Mime_generator class, a descendent of the CI Email library, and will be configured in exactly the same way as the CI Email Library using 
	* {@link _configure_mail_object}.
	*
	* @return boolean 
	*/
	public function send(){
		if(!isset($this->id)) return $this->error->warning('Please save '.$this->describe().' before sending'); //saving will validate a number of values, so we need to do this first	
		//make sure that the mailbox is active & we're allowed to send mail
		$mailbox = $this->mailbox;
		if(!$mailbox->is_active) return $this->error->warning('Cannot send '.$this->describe().'; '.$mailbox->describe().' is not active');
		//make sure that the message values are valid for send	
		if(!$this->draft) return $this->error->warning(ucfirst($this->describe()).' is not a draft and may not be sent');
		if($this->archived) return $this->error->warning(ucfirst($this->describe()).' has been archived and may not be sent until it has been restored.');
		if($this->property_is_empty('sender')) return $this->error->warning('Please specify a sender for '.$this->describe().' before sending');
		if(!$this->has_recipients()) return $this->error->warning('Please specify at least one recipient for '.$this->describe());
		
		//note - for business team reasons, we're letting applications self-enforce whether or not disclosures are required for patient documents.					
		/*
		if(!$this->has_required_attachment_disclosures()){
			return $this->error->warning('Unable to send '.$this->describe().'; missing required attachment disclosures');
		}*/
		
		//first, make sure we can record the change for this in the database
		$this->_set_field_value('draft', false, 0, TRUE); //draft is usually readonly, so override the validation in this case only
		$this->_set_field_value('sent', true, 0, TRUE); //sent is usually readonly, so override the validation in this case only
		$this->_set_field_value('timestamp', now(), 0, TRUE); //timestamp is usually readonly, so override the validation in this case only
		$this->save();
		if(!$this->sent || $this->draft){
			return $this->error->warning('Unable to send '.$this->describe().'; unable to change message state from draft to sent'); //don't try to actually send the message if we couldn't update it in the db
		}
		
		$this->attachment_files(); //make sure that attachment files is populated			
		
		//configure the CI email library && use it to send the message
		$CI = get_instance();
		$CI->load->library('email'); 
		$CI->email->clear(); //clear just in case the email library was used elsewhere & not cleared
		if(!$this->_configure_mail_object($CI->email)) return $this->error->warning('Unable to configure email library for '.$this->describe());
		$success = $CI->email->send();
		if(!$success) $this->error->warning('Unable to send '.$this->describe().'; reverting the message to draft state.');
		$CI->email->clear();
		
		//if we weren't able to send, restore the message to draft state
		if(!$success){
			$this->_set_field_value('draft', true, 0, TRUE); //draft is usually readonly, so override the validation in this case only
			$this->_set_field_value('sent', false, 0, TRUE); //sent is usually readonly, so override the validation in this case only
			$this->_set_field_value('timestamp', now(), 0, TRUE); //timestamp is usually readonly, so override the validation in this case only
			if(!$this->save() || $this->sent || !$this->draft){
				$this->error->warning('Unable to revert '.$this->describe().' to draft state, but the message was not sent.  Please correct this database entry manually if possible.');
			}
			return false;
		}		
		
		//clear any disclosures made for attachments that were not a part of the message when it's been finalized and sent
		if(!empty($this->attachment_disclosures())){
			if($this->has_attachments())
				Attachment_disclosure::db()->where('hash IS NOT NULL')->where_not_in('hash', array_map('sha1', $this->attachment_files));
			
			Attachment_disclosure::db()->where(array(static::foreign_key('attachment_disclosure') => $this->id))->delete(Attachment_disclosure::$table);
		}
		
		foreach($this->disclosures as $disclosure){			
			//for sent messages, we want a disclosure for every recipient
			$values = $disclosure->writeable_values();
			$success = true;
			
			foreach(get_instance()->json->decode($this->recipients) as $recipient){
				$values['recipient'] = $recipient;
				$values['disclosed'] = time();
				$values['sent_to'] = substr($recipient, strpos($recipient, '@') + 1);
				if($disclosure->property_is_empty('hash')){
					$success = Disclosure::create($values) && $success;
				}else{
					$values['hash'] = $disclosure->hash;
					$success = Attachment_disclosure::create($values) && $success;
				}
			}
			
			if($success)
				Disclosure::delete($disclosure->id);
			else
				$this->error->warning('Unable to populate disclosures for each recipient of '.$disclosure->describe().'; preserving '.$disclosure->describe().' with no recipients');	
		}
		
		return $success;	
	}

	//formatting that applies to all values messages returned by the API goes here, so that we don't have to duplicate code
	public function values_for_api($part = null){
		if(!empty($part) && !Message::is_a_part($part)) return $this->error->should_be_a_message_part($part);
		
		if(empty($part))
			$values = $this->values();
		else
			$values = $this->values_for_part($part);
		
		if(array_key_exists('attachments', $values) && empty($values['attachments'])) 
			$values['attachments'] = '{}';
			
		if(array_key_exists('original_sender_id', $values)){
			$original_sender = $this->original_sender;
			$values['original_sender'] =  (Mailbox::is_an_entity($original_sender)) ? $original_sender->name : '';
			unset($values['original_sender_id']);
		}

		$values['patients'] = array();
		foreach($this->patients as $patient){
			$values['patients'][] = $patient->values_for_api();
		}	
		
		
#		if($this->sent && ($part === null || $part === "status" || $part === "headers" || $part=="")){
		if($this->sent && (empty($part) || $part == "status" || $part == "headers")){	
/* NOTE - We should really rename this - shouldn't be prepended with "message_" since it's already part of a message (redundant).  Also, "status" is unclear and 
   implies a singular value when this is actually a collection of rows from the database depicting the delivery history for each recipient -- MG 2014-05-16 */
			$values['message_status'] = get_instance()->messagestatusmodel->get_status($this->message_id); 
		}
		
		//raw mime is pulled by default, but we also parse it by default unless raw mime is requested
		//^ anyone know what this comment is supposed to mean?  confusing, plus the $raw_mime variable wasn't ever set even when this was still in the controller
		if(array_key_exists('raw_mime', $values) && empty($raw_mime)) {
			#NOTE: parse_mime is meant to parse raw mime into an object which can then be
			#returned by the API as JSON, XML, etc. which can be helpful for people who want
			#the information in the raw mime without parsing it themselves.
			#If the raw_mime parameter is passed, they get the raw mime instead, but by
			#default they get the parsed object from this method - ATB 3/21/2014
			//moved this code out of parse_mime & into model; seemed like a better place for it than the controller -- MG 2014/05/16
			$mime = $this->mime(false);
			if(is_object($mime)){
				$values['mime'] = $mime;		
				unset($values['raw_mime']); 
			}
		}
		
		//There is no real xdm_attachments part, but if requested we can check if the message is likely to contain an xdm attachment and return
		//an assoc. array of it's zip contents through the API. Since this is extra work we do this through the API only if it is requested.
		if($part === 'xdm_attachments') {
			if($this->has_xdm_attachment()) {
				$xdm_attachments = array();
				foreach($this->attachment_files as $name => $binary) {
					if(pathinfo($name, PATHINFO_EXTENSION) === 'zip') {
						$xdm_file = tmpfile();
						$metadata = stream_get_meta_data($xdm_file);
						$tmp_filename = $metadata['uri'];
						fwrite($xdm_file, $binary);
						$zip = new ZipArchive;
						$zip->open($tmp_filename);
						$zip_items = array();
						for($i = 0; $i < $zip->numFiles; $i++){ 
						    $stat = $zip->statIndex($i);
						    $item = $zip->getFromIndex($i);
					    	$zip_items[$stat['name']] = array('content-transfer-encoding' => 'base64', 'name' => basename($stat['name']), 'body' => base64_encode($item));
					    }
					    array_push($xdm_attachments, array($name => explode_tree($zip_items, '/')));
					}
				}
				$values['xdm_attachments'] = $xdm_attachments;
			}
			else { $values['xdm_attachments'] = array(); }
		}

		if($this->draft){
			$disclosures = array();
			foreach($this->attachment_disclosures as $key => $disclosure){
				$disclosures[$disclosure->hash] = $disclosure->values(array('purpose', 'ssn'));
			}
			$values['disclosures'] = $disclosures;
		}		
		return $values;
	}

	/**
	* @param string
	* @return array
	*/
	public function values_for_part($part){
		if(!Message::is_a_part($part)) return $this->error->should_be_a_message_part($part);
		$primary_key = static::$primary_key;
		$part_fields = static::$parts[$part];
		return array_merge( array($primary_key => $this->id()), $this->values($part_fields));
	}


	/**
	* Helper method used to configure the Mime_generator object when editing/saving messages, and the Email class when sending.
	* @return boolean success
	*/
	protected function _configure_mail_object(&$mail_object){
		if(!is_a($mail_object, 'CI_Email')) return $this->error->should_be_an_object_that_inherits_from_the_codeigniter_email_library($mail_object);
		
		$mail_object->initialize( array( 'mailtype' => $this->mailtype(),
										 'priority' => $this->priority() ) );
			
#TODO - ARE WE SURE WE WANT \R\N?  ACCORDING TO CI'S COMMENTS, THEY DEFAULT TO \N BECAUSE IT'S THE ONLY ONE THAT ALL SERVERS SEEM TO ACCEPT (DESPITE SPECIFICATIONS TO THE CONTRARY)			
		$mail_object->set_newline("\r\n");
			
		$mail_object->from($this->sender);  //the Email library needs a value for this even if it's an empty string
			
		foreach( array('to', 'cc', 'bcc', 'subject') as $field){
			if(!$this->property_is_empty($field))
				$mail_object->$field($this->$field);
		}
		$mail_object->set_header("storage-id",$this->id());
		if(REQUEST_DISPATCHED_MDN) { 
			$mail_object->set_header("Disposition-Notification-Options","X-DIRECT-FINAL-DESTINATION-DELIVERY=optional,true");
		}
		if(isset($this->protected_data)){
			$mail_object->set_header("protected-data",$this->protected_data);
		}
		if(!$this->property_is_empty('body'))
			$mail_object->message($this->body);
			
		//add attached files.  note that we'll check the size requirement when the files are passed to the model, so we don't need to check that here
		if(isset($this->_attachment_files)){
			foreach($this->attachment_files as $name => $binary_string)
				$mail_object->string_attach($binary_string, $name); 
		}
				
		return true;	
	}	


	public function values($just_these_values=null){
		if(is_null($just_these_values)){
			$just_these_values = array();
			foreach($this->fields as $field){
				if(isset($this->$field) || array_key_exists($field, $this->_values) || !$this->property_is_empty($field))
					$just_these_values[] = $field;
			}
		}
        return parent::values($just_these_values);
    }   


/////////////////////////////////////////////////////////////////////////////////////////////////////////
// GETTERS
// These are accessors for class variables prefaced by a _ and for database field values for this row 
// of the message table.  You can access these values by just treating them as normal variables, and
// {@link __get()} will obtain the value for you.  
///////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	/**
	* A JSON-encoded array of the names of the attachments for this message.
	* This value will be automatically updated every time a value in {@link mime_fields} is altered
	* for this message.
	*
	* Getter for message.attachments, with a default derived from {@link mime()} if the message hasn't been
	* saved yet.
	*
	* @return string
	*/
	protected function attachments(){
		if(!isset($this->_values['attachments']) && !$this->property_is_empty('attachment_files')){
			return get_instance()->json->encode(array_keys($this->attachment_files)); 
		}
		return $this->_values['attachments'];
	}	
	
	/**
	* An array of the message's attached files.
	* @return array Formatted array( $file_name => $file_binary_string)
	*/
	protected function attachment_files(){
		if(is_null($this->_attachment_files)){
			$raw_mime = $this->raw_mime();
			$mime = $this->mime();
						
			if(is_object($mime) && isset($mime->parts)){
				$this->_attachment_files = array();	//don't set this unless we've managed to get the $mime
					foreach($mime->parts as $part){
						if(isset($part->disposition) && $part->disposition == 'attachment'){
							$name = $part->ctype_parameters['name'];
							$this->_attachment_files[$name] = $part->body;	
						}
					}
			}
		}
		return $this->_attachment_files;
	}

	public function encoded_attachment_files(){
		if(!is_array($this->attachment_files)) return array();
		$encoded_files = array();
		foreach($this->attachment_files as $filename => $file){
			$encoded_files[$filename] = base64_encode($file);
		}
		return $encoded_files;
	}

	/**
	* The body of the messsage.
	* This will either be the message.html value or the message.plain value depending on message.mailtype.
	* This is a getter for {@link _body}.  
	* @return string
	*/
	protected function body(){	
		if(!isset($this->_body)){
			if($this->mailtype() == 'html' && isset($this->html))
				$this->_body = $this->html;
			if($this->mailtype() == 'text' && isset($this->plain))
				$this->_body = $this->plain;
		}
		return $this->_body;
	}

	/**
	* True if this message is a draft.  
	* Getter for message.draft, with a default of true if the message hasn't been saved yet.
	* @return boolean
	*/	
	protected function draft(){
		if(!array_key_exists('draft', $this->_values)){
			//default new messages to draft, but ONLY new messages - all other messages, we need to rely on whatever was set in the databse 
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$draft has not been loaded from the database');
			else
				$this->_set_field_value('draft', true, 0, TRUE); //override validation because this is normally a read-only property
		}
		return $this->_values['draft'];
	}
	

	/**
	* A JSON-encoded array of the mime headers for this message.
	* This value will be automatically updated every time a value in {@link mime_fields} is altered
	* for this message.
	*
	* Getter for message.headers, with a default derived from {@link mime()} if the message hasn't been
	* saved yet.
	*
	* @return string
	*/
	protected function headers(){
		if(!isset($this->_values['headers'])){
			return get_instance()->json->encode($this->mime()->headers); 
		}
		return $this->_values['headers'];
	}

	/**
	* The HTML body of the message.
	* This value will be empty when message.mailtype is not 'html'.  To reliably get the body of the 
	* message regardless of mailtype, use {@link body()}
	*
	* Getter for message.html, with a default derived from {@link mime()} if the message hasn't been 
	* saved yet.
	*
	* @return string
	*/
	protected function html(){
		if(!isset($this->_values['html']) && $this->mailtype == 'html'){		
			$mime = $this->mime();
			if(is_object($mime) && property_exists($mime, 'parts')){
				$text = $this->_find_text_from_parts('html', $mime->parts);
				if(!empty($text)){
					$this->_set_field_value('html', $text, 0, TRUE); //override validation because this is normally a read-only property
				}
			}
		}
		if(isset($this->_values['html'])) 
			return $this->_values['html'];
	}
	
	function inline_attachment_files(){
		if(!isset($this->_inline_attachment_files)){
			$this->attachment_files(); //make sure we've parse the normal attachment files so that this array is set
			$this->_inline_attachment_files = $this->_attachment_files_of_type('inline');	
		}
		return $this->_inline_attachment_files;	
	}			
	
	/**
	* The mailtype for this message ('html', 'text').
	* Getter for message.mailtype, with a default of 'html'.
	*/
	protected function mailtype(){
		if(!array_key_exists('mailtype', $this->_values)){
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$mailtype has not been loaded from the database');
			else
				$this->_set_field_value('mailtype', 'html'); //set a default value for new messages, but ONLY for new messages - for existing messages, we need to get this value from the db
		}
		
		return $this->_values['mailtype'];	
	}
	
	/**
	* A Mime_generator object that represents the message.
	* This value will be updated whenever a value {@link mime_fields} is altered.
	* @return Mime_generator
	*/
	protected function mime_generator(){
		//if this variable hasn't been loaded, derive it from the information available
		if(!isset($this->_mime_generator)){
			$mime_generator = new Mime_generator(); //we're not loading this the normal way because we don't want one singleton instance for the entire application, we want an individual one for this message	
			if($this->_configure_mail_object($mime_generator)){ //configure this the same way that we would if we were sending the message
				$this->_mime_generator = $mime_generator; //as long as we didn't run into problems in the configuration, set the mime generator
				$this->_attachment_files = null; //clear _attachment_files to minimize the amount of data we have stored on the object.  We'll regenerate it before the next time that we wipe the raw mime
			}
		}
		return $this->_mime_generator;		
	}	
	

	/**
	* The plain-text body of this message.
	* This will always return the plain-text version of the message, even if it's an HTML message.
	* To get the primary body of the message regardless of {@link mailtype}, use {@link body()}.
	*
	* Getter for message.plain, with a value derived from {@link mime()} if changes to the message
	* have not been saved yet.
	*
	* @return string
	*/
	protected function plain(){
		if(!isset($this->_values['plain'])){
			if($this->mailtype == 'html'){
				$mime = $this->mime();
				if(is_object($mime) && property_exists($mime, 'parts')){
					$text = $this->_find_text_from_parts('plain', $mime->parts);
					if(!empty($text)){
						$this->_set_field_value('plain', $text, 0, TRUE); //override validation because this is normally a read-only property
					}
				}
			}else{
				$this->_set_field_value('plain', trim($this->mime()->body), 0, TRUE); //override validation, since this is normally a read-only property
			}
		}
		return $this->_values['plain'];
	}

	/**
	* A numerical value indicating the priority of this message.
	* Getter for message.priority, with a default of 3 (normal) if the message has not been saved yet.
	* @return int
	*/
	protected function priority(){
		if(!array_key_exists('priority', $this->_values)){
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$priority has not been loaded from the database');
			else
				$this->_set_field_value('priority', 3);  //set a default value for new messages, but ONLY for new messages - for existing messages, we need to get this value from the db
		}
		return $this->_values['priority'];	
	}
	
	/**
	* The raw text value of the mime for this message.
	* Getter for message.raw_mime, with a default derived from {@link mime_generator()} if the 
	* message has not been saved yet.
	* @return string
	*/
	protected function raw_mime(){
		if(!isset($this->_values['raw_mime'])){
			$mime_generator = $this->mime_generator();
			if(!is_a($mime_generator, 'CI_Email')) return $this->error->warning('Could not generate mime for '.$this->describe());
			$this->_values['raw_mime'] = $mime_generator->raw_mime();
			//set the timestamp every time the raw_mime is re-formed - this ensures that the timestamp will get set when things are changed that affect the content of the message, 
			//but not if it's archived or moved to a different folder, etc.
			$this->_set_field_value('timestamp', now(), 0, TRUE); 
		}
		return $this->_values['raw_mime'];	
	}
	
	
	/**
	* An array of all of the read-only class variables and database fields for this message.
	* Extends its parent method to include the {@link mime_derived_fields} as read-only fields.
	* Additionally, any fields which would change the mime may only be altered for drafts.
	* Getter for {@link _readonly_fields}.
	* @return array
	*/
	public function readonly_fields(){
		//the mime-derived fields are always readonly
		$readonly_fields = array_merge(parent::readonly_fields(), $this->mime_derived_fields);
		
		//any fields that would change the mime are readonly if this isn't a draft - sent messages can't be altered, incoming messages can't be altered
		if(isset($this->id) && !$this->draft) $readonly_fields = array_merge($readonly_fields, $this->mime_fields); 
		
		//if this is an outgoing message, we can't set the 'seen' flag 
		if(!isset($this->id) || $this->draft || $this->sent)
			$readonly_fields[] = 'seen';
		
		return $readonly_fields;
	}		
	

	/**
	* A JSON-encoded array of the email addresses of all recipients (to, cc, and bcc) of this message.
	* Getter for message.recipients, with a default determined from to, cc, and bcc values of this message
	* if it has not yet been saved.
	* @return string
	*/
	protected function recipients(){
		if(!isset($this->_values['recipients'])){
			$recipients = array_unique( array_filter( array_merge( explode(',', $this->to), explode(',', $this->cc), explode(',', $this->bcc) ) ) );
			if(!empty($recipients))
				$this->_values['recipients'] = get_instance()->json->encode( $recipients ); 
		}
		return $this->_values['recipients'];
	}
	
	protected function seen(){
		if($this->is_outgoing()) return true;
		return $this->_values['seen'];
	}
	
	/**
	* The size of the entire message in bytes.
	* Getter for message.size, derived from {@link mime_generator} if the message has not been saved.
	* @return int
	*/
	protected function size(){
		if(!isset($this->_values['size'])) $this->_values['size'] = $this->mime_generator->size_in_bytes(); 
		return $this->_values['size'];
	}
	
	/**
	* The subject of this message.
	* Getter for message.subject, defaults to '(No Subject)' if the message has not yet been saved.
	* @return string
	*/
# Adam says we should leave subject blanka nd let applications deal with it on their own.  I concur.	
/*	protected function subject(){
		if(!array_key_exists('subject', $this->_values)){
			if(isset($this->id))
				return $this->error->warning('The value for '.get_class($this).'::$subject has not been loaded from the database');
#			else	
#				$this->_set_field_value('subject', '(No Subject)');  //set a default value for new messages, but ONLY for new messages - for existing messages, we need to get this value from the db
		}
		return $this->_values['subject'];
	} */

	protected function _attachment_files_of_type($type){		
		if(!in_array($type, array('inline', 'attachment'))) return $this->error->should_be_a_known_type($type);
		$files = array();
		if(isset($this->id) && is_object($this->mime) && isset($this->mime->parts)){								
			foreach($this->mime->parts as $part){	
				if(isset($part->headers->{'content-transfer-encoding'}) && !empty($part->headers->{'content-transfer-encoding'}))	{
					//decode part based on content transfer encoding, if necessary
					switch (strtolower($part->headers->{'content-transfer-encoding'})) {
						case 'base64':
							$part->body = base64_decode($part->body);
							break;
						case 'quoted-printable':
							$part->body = quoted_printable_decode($part->body);
						default: //if it's 8bit, 7bit, or binary, we should be able to handle it; for x-token, we don't handle custom encodings, so we'll still go to default
							break;
					}
				}

				//useful documentation for what the value of $part can be: http://users.dcc.uchile.cl/~xnoguer/peardoc2/package.mail.mail-mimedecode.decode.html		
				if(isset($part->disposition) && $part->disposition == $type){						
					if(isset($part->ctype_parameters) && is_array($part->ctype_parameters))
						$part->ctype_parameters = (object)$part->ctype_parameters;
					if(isset($part->d_parameters) && is_array($part->d_parameters))
						$part->d_parameters = (object)$part->d_parameters;	
							
					//try to track down the name of this attachment if we can
					$name = 'Untitled';
					if(isset($part->ctype_parameters) && !empty($part->ctype_parameters->name))
						$name = $part->ctype_parameters->name;
					elseif(isset($part->ctype_parameters) && !empty($part->ctype_parameters->filename))
						$name = $part->ctype_parameters->filename;
					elseif(isset($part->d_parameters) && !empty($part->d_parameters->name))
						$name = $part->d_parameters->name;
					elseif(isset($part->d_parameters) && !empty($part->d_parameters->filename))
						$name = $part->d_parameters->filename;					
					//figure out the extension, in case we need to modify the name at all	
					$extension = '';
					if(string_contains('.', $name))
						$extension = substr($name, strpos($name, '.'));
					if($name == 'Untitled' && empty($extension) && !empty($part->ctype_secondary))
						$extension = '.'.$part->ctype_secondary; //note - this may not actually be a file type (eg "plain" or "x-unknown-content-type").  Could add a check, but risk falsely ruling out file types
					//if we have duplicate file names, make them unique by adding a number (#) to each one
					$name = strip_from_end($extension, $name);
					for($i = 1; array_key_exists($name.$extension, $files); $i++){
						//make sure to clear out any parentheses that we've added.  Note that we'll have to do this even for things we didn't add so that we don't mess things up while re-saving.
						if(preg_match('/\(\d+\)$/', $name, $matches)) {
							$name = trim(strip_from_end(first_element($matches), $name)); //clear out any numbers that we've added already
						}
						$name = $name.' ('.$i.')';
					}			
					$files[$name.$extension] = $part->body;	
				}
			}
		}				
		
		return $files;
	}

	protected function _find_text_from_parts($text_to_search_for, $parts){
		if(!is_string($text_to_search_for) && !in_array($text_to_search_for, array('html', 'plain'))) return $this->error->should_be_a_known_text_type($text_to_search_for);
		if(empty($parts) || !is_array($parts)) return null;
		foreach($parts as $part){
			if(is_object($part) && isset($part->parts)){
				$text = $this->_find_text_from_parts($text_to_search_for, $part->parts);
				if(!is_null($text)) return $text;
			}
			if($part->ctype_primary == 'text' && $part->ctype_secondary == $text_to_search_for && (!isset($part->disposition) || $part->disposition != 'attachment')){
				return trim($part->body);
			}	
		}
	}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SETTERS
// These methods will be called by __set() when the corresponding class variable or database field 
// value is set: e.g., calling $this->sender = 'gibbs_margaret@bah.com' calls $this->set_sender('gibbs_margaret@bah.com').
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	* Sets the attachment files for this message.
	* Overwrites any existing attachment files completely and regenerates the mime as needed.
	* @param array 
	* @param offset
	*/
	public function set_attachment_files($value, $offset=0, $attempting_to_restore_original_files = false){
		if(!$this->is->associative_array_of_strings($value)) return $this->error->property_value_should_be_an_array_of_filenames_mapped_to_binary_strings('attachment_files', get_class($this), $value, $offset+1);
	
		//back up our current attachments in case this doesn't work
		$original_attachment_files = $this->attachment_files;
		$this->_attachment_files = array();
		$this->reload_mime(); //force the mime to regenerate so that we're not judging whether or not we can add this attachment based on the old size
	
		$success = true;
		foreach($value as $filename => $binary_string){	
			$success = $success && $this->add_attachment($filename, $binary_string);
			$this->reload_mime(); //force the mime to regenerate so that we're not judging whether or not we can add this attachment based on the old size
		}
		
		//if we didn't succeed in attaching the files, restore our original files - but only try once, so that we avoid an infinite loop of failed attachments
		if(!$success){
			if($attempting_to_restore_original_files)
				return $this->error->warning('Unable to restore original attached files for '.$this->describe());
			return $this->set_attachment_files($original_attachment_files, $offset+1, $attempting_to_restore_original_files = true); 
		}
		
		return $success;
	}

	/**
	* Sets the body for this message.
	* This will reload the {@link mime_generator()} and update the {@link mime_derived_fields} as 
	* appropriate: message.plain and message.html will be updated as appropriate to {@link mailtype},
	* {@link size} will be updated with the new size of the message, etc.
	* Setter for {@link _body}.
	* @param string
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/
	public function set_body($value, $offset = 0){
		if(!is_string($value)) return $this->error->property_value_should_be_a_string('body', get_class($this), $value, $offset+1);
		$this->_body = $value;
		$this->reload_mime();
	}

	/**
	* Sets the mailbox id for this message.
	* Note that you cannot change the mailbox_id to a mailbox that doesn't match the message.sender
	* value; outgoing messages (which are the only messages we should be altering) must always
	* have matching values for sender and $mailbox->email_address().  
	*
	* If message.sender is not already set, setting this value will automatically populate message.sender.
	* 
	* Setter for messages.mailbox_id
	*
	* @param int
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/
	public function set_mailbox_id($value, $offset = 0){
		if(!Mailbox::formatted_like_an_id($value)) return $this->error->should_be_formatted_like_a_mailbox_id($value, $offset+1);
		
		$mailbox = Mailbox::find_one($value);
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->should_be_a_mailbox_id($value, $offset+1);
		if(!$mailbox->is_active) return $this->error->warning('Cannot add '.$this->describe().' to '.$mailbox->describe().'; '.$mailbox->describe().' is not an active mailbox');
		
		if(!$this->property_is_empty('sender') && $mailbox->email_address != $this->sender) 
			return $this->error->should_be_a_mailbox_id_that_matches_the_mailbox_for_the_message_sender($value, $offset+1);
		
		$this->_set_field_value('sender', $mailbox->email_address, $offset+1);
		return $this->_set_field_value('mailbox_id', $value, $offset+1);
	}

	/**
	* Sets the mailtype for this message.
	* Setter for message.mailtype.
	* @param string Valid values are 'text' and 'html'
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/	
	public function set_mailtype($value, $offset=0){
		$valid_values = array('html', 'text');
		if(!in_array($value, $valid_values))
			return $this->error->property_value_should_be_an_x('mailtype', get_class($this), $value, 
															   'valid value ('.array_to_human_readable_list(array_map(array($this->error, 'describe'), $valid_values), 'or').')', $offset+1);
		
		$this->_set_field_value('mailtype', $value, $offset+1);
	}

	/**
	* Sets the sender email address for this message.
	* Note that this cannot be set to a value that does not match the value of this mailbox's email address.
	* (For outgoing mail, sender and mailbox must match; for incoming mail, we can't change the value of sender.)
	* If the mailbox has not been set, it will automatically be set to the mailbox for this sender.
	* 
	* Setter for message.sender
	*
	* @param string Email address of the sender
	* @param int (Optional, internal use only) An integer offset to help {@link Error} build better backtraces
	*/
	public function set_sender($sender, $offset = 0){
		if(!$this->is->nonempty_string($sender)) return $this->error->property_value_should_be_a_nonempty_string('sender', get_class($this), $sender, $offset+1);
		if(!$this->is->string_like_a_direct_address($sender)) return $this->error->property_value_should_be_a_direct_address('sender', get_class($this), $sender, $offset+1);
	
		$mailbox_name = strip_from_end('@'.DIRECT_DOMAIN, $sender);
		$mailbox = Mailbox::find_one( array('name' => $mailbox_name) );
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->property_value_should_be_an_address_for_an_existing_mailbox('sender', get_class($this), $sender,  $offset+1);
		

		//make sure that sender isn't different than the mailbox's sender, if specified
		$mailbox_foreign_key = static::related_foreign_key('mailbox');		
		if(!$this->property_is_empty($mailbox_foreign_key) && $this->$mailbox_foreign_key != $mailbox->id)
			return $this->error->property_value_should_be_an_x('sender', get_class($this), $sender, 'email address for mailbox#'.$mailbox->id, $offset+1);
		
		if(!$this->_set_field_value($mailbox_foreign_key, $mailbox->id()))
			$this->error->warning('Could not set '.$mailbox_foreign_key.' to '.$mailbox->id(), $offset+1);
		
		return $this->_set_field_value('sender', $sender, $offset+1);
	}

	/**
	* Helper method for setting database field values - for internal use only.
	* Extends parent method to ensure that the mime is regenerated every time a {@link mime_field}  is altered.
	* Do NOT set override_validation to true without careful thought - this should be used only in special circumstances, like setting a default value for a read-only value.
	* @return boolean
	*/
	protected function _set_field_value($field, $value, $error_offset=0, $override_validation = FALSE){
		if(!parent::_set_field_value($field, $value, $error_offset+1, $override_validation)) return false;
		
		//if something that affects the mime got changed, let's make sure that the mime & all derived get wiped
		//they'll get regenerated the next time someone calls on them
		if(in_array($field, $this->mime_fields)) $this->reload_mime();
		
		return true;
	}
	
	/**
	* Clears the value for the mime and the mime-derived fields, forcing them to be reloaded when next called.
	* This method will be called whenever the value of a field in {@link mime_fields} is altered.
	* The next time that these values are called upon, their getters will re-derive their value
	* from the mime.
	*/
	protected function reload_mime(){
		$this->_mime_generator = null;
		$this->attachment_files(); //make sure that the attachment_files gets saved to the _attachment_files so that it won't get overwritten when the mime gets regenerated
		$this->body(); //make sure that the body gets saved to the _body variable so that it won't get overwritten when the mime gets regenerated
		foreach($this->mime_derived_fields as $mime_derived_field){
			$this->_values[$mime_derived_field] = null;
			$this->_altered_fields[] = $mime_derived_field;
		}
	}

	

//////////////////////
// DATA MANAGEMENT
//////////////////////
	 
	protected function _values_for_save(){
		$values_for_save = array_merge(parent::_values_for_save(), $this->values( $this->mime_derived_fields ) ); //force mime_derived_values to be saved
		
		if(!isset($this->id)){
			$values_for_save['draft'] = $this->draft; //draft isn't normally a writeable field, so make sure that the default value gets saved
			$values_for_save['seen'] = $this->seen; //seen isn't normally a writeable field, so make sure that the default value gets saved
			$values_for_save['timestamp'] = $this->timestamp;
			$values_for_save['seen'] = $this->seen;
		}
		return $values_for_save;
	}	 	

#TODO - DO SOME SORT OF CHECK TO MAKE SURE THAT WE'RE NOT SAVING CHANGES TO A MESSAGE THAT BELONGS TO AN ARCHIVED MAILBOX
	protected function _values_are_valid_for_create_and_update(){
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		$required_fields = array( static::related_foreign_key('mailbox'), 'sender' );
		foreach($required_fields as $required_field){
			if($this->property_is_empty($required_field)) return $this->error->warning(get_class($this).'::$'.$required_field.' is a required field, and must be set before saving'); 
		}
		return true;
	}
	
	//this method is run after data has been saved to the database and the object has been reloaded with the updated database values
	//it will only be run if the data has been saved successfully
	protected function _run_after_create_and_update(){ 
	
		if(!empty($this->_attachment_files)) {
			//PARSE & SAVE PATIENTS FOR ATTACHED CCDS
			Patient::populate_for_message($this);
		}
		//MANAGE MDNS / READ RECEIPTS
	
		//send a read receipt if this message is marked as read but not marked as having been displayed (i.e., the first time that this message is successfully marked as read)
		if($this->seen && $this->is_incoming() && ($this->property_is_empty('flags') || !string_contains('\\Displayed', $this->flags))){ //if the mdn has not been set
				
			//the seen toggle may be set to seen/unseen depending on whether the user wants it to appear to be marked as read
			//in contrast, the "\\Displayed" flag will be set once and not changed afterwards				
			$this->flags .= "\\Displayed"; 
			
			if($this->save()){ //note - we're re-saving after changing the flags value, so this isn't an infinite loop.
				//check to ensure that they requested delivery (dispatched) MDN, if they did not request dispatched don't give them displayed
				$headers = array_change_key_case(get_instance()->json->decode($this->headers, TRUE), CASE_LOWER);
				if(!empty($headers['disposition-notification-options']) && strtolower($headers['disposition-notification-options']) === 'x-direct-final-destination-delivery=optional,true') {
					//if we succeeded in marking the flag so that the message is marked as displayed, send a read receipt to the sender
					$CI = get_instance();
					if(string_ends_with('@'.DIRECT_DOMAIN, $this->sender)){ //for local senders, we can just do this in the db
						$CI->load->model('messagestatusmodel');
						$CI->messagestatusmodel->message_displayed($this->message_id, $this->mailbox->email_address());
					}
					elseif(SEND_DISPLAY_MDN){//for non-local senders, we'll send an actual MDN via email
						$local_address = $this->mailbox->email_address();
		
						$body = '--=_01234567890123456789' . "\r\n" .
								'Content-Type: text/plain'  . "\r\n" . "\r\n" .
						
								'Your message:'  . "\r\n" .
								'    From.....: ' . $this->sender . "\r\n" .
								'    Subject..: ' . $this->subject . "\r\n" .
								' was read on ' . gmdate('m/d/Y') . ' at ' . gmdate('H:i') . "(GMT)\r\n" .
								'--=_01234567890123456789' . "\r\n" .
								'Content-Type: message/disposition-notification' . "\r\n" . "\r\n" .
						
								'Reporting-UA: ' . DIRECT_DOMAIN . '; VLER Direct \r\n' .
								'Final-Recipient: rfc822;' . $local_address . "\r\n" .
								'Original-Message-Id: ' . $this->message_id . "\r\n"  .
								'Disposition: automatic-action/MDN-sent-automatically; displayed' . "\r\n" .
								'--=_01234567890123456789--';
						
						$headers['From'] = $local_address;
						$headers['Content-Type'] = 'multipart/report; boundary="=_01234567890123456789"; report-type=disposition-notification';
						$headers['Mime-Version'] = '1.0';
						$headers['To'] =  $this->sender;
						$headers['Subject'] = 'Read: ' . $this->subject;
						
						// Determine the method we're using to send the mdn
						$params['host'] = GATEWAY_SMTP_HOSTNAME;
						$params['port'] = GATEWAY_SMTP_PORT;
						$params['helo'] = 'HELO';
						$mail_object = & Mail::factory('smtp', $params);
						$result = $mail_object->send($this->sender,$headers,$body);					
					}
				}			
			}			
		}
	
		return true;
	}		

////////////////////
// STATIC
////////////////////													  

	/**
	* Checks to make sure that the given file has a valid filetype.
	*
	* @todo It would be better to whitelist these.
	*
	* @param string
	* @param string
	* @return boolean
	*/
	public static function attachment_has_valid_extension($name){
#TODO - WE SHOULD REALLY BE CHECKING THIS AGAINST A WHITELIST, NOT JUST BLACKLISTING EXE	
		$extension = pathinfo($name, PATHINFO_EXTENSION);
		return !empty($extension) && mb_strtolower($extension) != 'exe';
	}
													 
	public static function find_part($part, $additional_conditions=array(), $key_results_by = null){
		if(!static::is_a_part($part)) return get_instance()->error->should_be_a_part($part, 1);
		static::set_up_select_for_part($part);
		return static::find($additional_conditions, $key_results_by);
	}


	public static function set_up_select_for_part($part){
		if(!static::is_a_part($part)) return get_instance()->error->should_be_a_message_part($part, 1);
		static::db()->select('['.implode('],[', array_merge( array(static::$primary_key), static::$parts[$part])).']');
		return true;
	}
	
	public static function is_a_part($name){
		if(!is_string($name) || empty($name)) return get_instance()->error->should_be_a_nonempty_string($name, 1);
		return array_key_exists($name, static::$parts);
	}
	
}
