<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models-folder
*/

/** */
require_model('entity');

/**
* @package direct-as-a-service
* @subpackage models
*/
class Folder extends Entity {
	static $database_group = 'mail_db';
	static $table = 'folders';	
	protected static $_relationships = array( 'mailbox' => array( 'type' => 'belongs_to'),
											  //messages in folders must not be sent, drafts, or archives - shouldn't be able to create or update this, but enforce anyway
											  'message' => array( 'type' => 'has_many', 'condition' => 'sent=0 AND draft=0 AND archived=0'), 
											  'parent_folder' => array('type' => 'belongs_to', 'model' => 'folder', 'related_foreign_key' => 'parent_id' ),
											  'child_folder' => array('type' => 'has_many', 'model' => 'folder', 'foreign_key' => 'parent_id' ),
											 );
											 
	protected $_property_validation_rules = array( 'name' => 'nonempty_string',
												   'mailbox_id' => 'nonzero_unsigned_integer');						 
	
	/**
	* Archive all the messages in this folder and then delete the folder.
	* @return boolean
	*/
	public function archive(){
		$mailbox = $this->mailbox;
		if(!$mailbox->is_active) return $this->error->warning("I can't archive ".$this->describe().' while '.$mailbox->describe().' is inactive');
		foreach($this->messages as $message){
			$message->archive();
		}

		//archive the descendant folders of the archived folder too
		foreach($this->child_folders as $child_folder) {
			$child_folder->archive();
		} 
			
		return static::delete($this->id());	//foreign key constraint will cascade to null
	}	
	
	
	public function belongs_to_mailbox($mailbox){
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->should_be_a_mailbox_entity($mailbox);
		$related_foreign_key = static::related_foreign_key('mailbox');
		return $this->$related_foreign_key == $mailbox->id();
	}
	
	/**
	* True if this folder has a branch of subfolders that exceeds the given depth.
	*
	* Calculating the maximum depth of descendants of a folder is a costly operation, and not a number that we actually need - we just need to be able to judge
	* whether adding an additional child would exceed NESTING_FOLDER_MAX_DEPTH.  This method will stop checking subfolders as soon as it reaches a level 
	* at which the depth exceeds the given $depth.  
	*
	*
	*/
	public function descendant_depth_exceeds($depth){
		if(!$this->is->unsigned_integer($depth)) return $this->error->should_be_an_unsigned_integer($depth);
		if(!Folder::formatted_like_an_id($this->id)) return false; //can't have a descendants if the folder hasn't been saved yet
		
		if($depth === 0) return $this->has_child_folder();
		foreach($this->child_folders as $child_folder){
			if($child_folder->descendant_depth_exceeds($depth - 1)) return true;
		}
		return false;
	}
	
	/** 
	* True if adding the given folder as a child folder would exceed the nesting limit.
	* @param Folder $folder
	* @return boolean
	*/	
	public function child_would_exceed_nesting_limit($folder){
		if(!Folder::is_an_entity($folder) || !Folder::formatted_like_an_id($folder->id)) return $this->error->should_be_a_saved_folder_entity($folder);
		
		$allowance = NESTED_FOLDER_MAX_DEPTH - $this->number_of_ancestors() - 1; //don't forget to subtract one for this folder
		
		return ($allowance < 1 || $folder->descendant_depth_exceeds($allowance - 1)); //don't forget to subtract one for the folder you're adding		
	}
		
	/**
	* True if this folder is a descendant of the given folder.
	* @param Folder $folder Potential ancestor
	* @return boolean 
	*/
	public function is_a_descendant_of($folder){
		if(!Folder::is_an_entity($folder)) return $this->error->should_be_an_entity($folder);
		
		//can't be an ancestor if this folder doesn't have a parent
		if(!Folder::formatted_like_an_id($this->parent_id)) return false; 
		
		//can't be a an ancestor if it hasn't been saved	
		if(!Folder::formatted_like_an_id($folder->id)) return false;
		 	
		//can't be an ancestor of itself
		if($folder->id == $this->id) return false; 
		
		//is definitely an ancestor if the parent id matches 
		if($folder->id == $this->parent_id) return true; 
		
		//iterate through ancestors
		$parent_folder = $this->parent_folder;
		while(Folder::is_an_entity($parent_folder)){
			if($parent_folder->id == $folder->id || $parent_folder->parent_id == $folder->id) return true;
			if(!Folder::formatted_like_an_id($parent_folder->parent_id)) return false; 
			$parent_folder = Folder::find_one($parent_folder->parent_id);
		}
		
		return false;
	}

	//0 if this folder has no parent; 1 if it has just a parent; 2 if it has a parent and a grandparent; etc.
	public function number_of_ancestors($number_of_ancestors = 0){
		$parent_folder = $this->parent_folder;
		if(Folder::is_an_entity($parent_folder)) {
			$number_of_ancestors++;
			$number_of_ancestors = $parent_folder->number_of_ancestors($number_of_ancestors);
		}
		return $number_of_ancestors;
	}
	
	
///////////////////////
// GETTERS & SETTERS
///////////////////////	
	
	protected function set_parent_id($value, $offset = 0){
		if(!is_null($value) && !Folder::formatted_like_an_id($value)) return $this->error->property_value_should_be_a_folder_id_or_null('parent_id', get_class($this), $value, $offset+1);
		if(Folder::formatted_like_an_id($value)){
			if($value == $this->parent_id) return true; //avoid any lookups - we're done
			if($value == $this->id) return $this->error->property_value_should_be_an_id_for_another_folder('parent_id', get_class($this), $value, $offset+1);
			
			//check to make sure that the parent folder exists, is not a descendant of this folder, and that assigning this folder as a subfolder to it would not exceed the maximum number of nested folders
			$parent_folder = Folder::find_one($value);
			if(!Folder::is_an_entity($parent_folder)) 
				return $this->error->property_value_should_be_an_id_for_an_existing_folder('parent_id', get_class($this), $value, $offset+1);
			if($parent_folder->is_a_descendant_of($this)) 
				return $this->error->property_value_should_be_an_x('parent_id', get_class($this), $value, 'id for a folder that is not a descendant of '.$this->describe(), $offset+1);
			if(isset($this->id) && $parent_folder->child_would_exceed_nesting_limit($this)) 
				return $this->error->property_value_should_be_an_x('parent_id', get_class($this), $value, 'id for a parent folder for '.$this->describe().' that would not exceed the maximum nesting limit ', $offset+1);
		}
		$this->_set_field_value('parent_id', $value, $offset+1);
	}
	
//////////////////////////
// DATA MANAGEMENT
///////////////////////////	
	
	protected function _values_are_valid_for_create_and_update(){ 
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		
		$mailbox_foreign_key = static::related_foreign_key('mailbox');
		
		//make sure that we're not trying to set name or mailbox to null
		foreach( array('name', $mailbox_foreign_key) as $required_field){
			if(!isset($this->$required_field)) return $this->error->warning('Unable to save '.$this->model_alias.' without a value for '.$this->model_alias.'::'.$required_field, 3);
			if($this->property_is_empty($required_field)) return $this->error->property_value_should_be_a_nonempty_string($required_field, get_class($this), $this->$required_field, 3);
		}
		
		//since we know we have a mailbox, make sure that it's an active mailbox
		$mailbox = $this->mailbox;
		
		if(!Mailbox::is_an_entity($mailbox)) 
			return $this->error->warning("I couldn't find a mailbox where ".$mailbox_foreign_key.'='.$this->$mailbox_foreign_key);
				
		if(!$mailbox->is_active)
			return $this->error->warning("I can't save changes to ".$this->describe().' for '.$mailbox->describe().' while '.$mailbox->describe().' is inactive');
		
		
#TODO - CAN WE LOWER OVERHEAD BY ONLY CHANGING THIS IF WE'VE CHANGED NAME/USERNAME?	
		//ensure that name is unique for the mailbox				
		$values = array('name' => $this->name, $mailbox_foreign_key => $this->$mailbox_foreign_key);
		if(!$this->property_is_empty('id'))
			$values['id !='] = $this->id();
		if(static::exists($values)) return $this->error->property_value_should_be_an_x('name', get_class($this), $this->name, 'unique name for mailbox#'.$this->error->describe($this->$mailbox_foreign_key), 2);		
		
		return true; 
	}
	
	function __get($property){
		
		if(string_begins_with('message_', $property)){
			//for message relationships, allow developers to append the name of a message part to the relationship name
			//e.g. $this->inbox_message_headers will return the headers for $this->inbox_messages
			$suffix = mb_substr($property, mb_strrpos($property, '_') + 1);
			if(Message::is_a_part($suffix)){
				$relationship = strip_from_end('_'.$suffix, $property);
				$has_relationship = static::has_relationship($relationship);
				$relationship_model = static::relationships($relationship, 'model');
				if($has_relationship && $relationship_model == 'message'){
					$this->set_up_relationship_query($relationship);
					return Message::find_part($suffix);
				}
			}
		}
				
		return parent::__get($property);
	}
	
	function __call($name, $arguments){
		
		if(string_begins_with('message_', $name)){
			//for message relationships, allow developers to append the name of a message part to the relationship name
			//e.g. $this->inbox_message_headers() will return the headers for $this->inbox_messages()
			$suffix = mb_substr($name, mb_strrpos($name, '_') + 1);
			if(Message::is_a_part($suffix)){
				$relationship = strip_from_end('_'.$suffix, $name);
				if(static::has_relationship($relationship) && static::relationships($relationship, 'model') == 'message'){
					$this->set_up_relationship_query($relationship);
					return call_user_func_array('Message::find_part', array_merge( array($suffix), $arguments));
				}
			}
		}
				
		return parent::__call($name, $arguments);
	}		
	
}