<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
*//** */

require_once APPPATH.'controllers/services/mailbox_controller.php';

/**
* API methods to manage a mailbox's messages.
*
* Inherits mailbox validation from the mailbox controller.  Mailbox name is a required field for all actions on this controller, with the exception of messages to send or save a messagage; these actions require the sender's email address to be provided.
*
* @author           <bell_adam@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
* @author Elan Jaffee <jaffee_elan@bah.com>
*
* @package direct-as-a-service
* @subpackage controllers
*/
class Message_send_controller extends Mailbox_controller{

	/**
	* Everything that applies to all actions should go here.
	*/
	function __construct(){
		parent::__construct();
		$this->load->helper(array('mail', 'validate'));
	}


	public function save_post(){
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('send');
		$this->respond_with_error_if_mailbox_is_inactive();

		//id is not required, but may be provided
		$id = element('id', $this->post(), null);

		//check to make sure that id is valid
		if(!is_null($id) && !Message::formatted_like_an_id($id))
			$this->invalid_fields[] = 'id';
		$this->respond_with_error_if_fields_are_invalid();


		if( $this->save_message($id) ){
			$this->response('Message saved', 200);
		}

		$this->response('Unable to save message', 400);
	}

	/**
	* Send a saved message draft.
	*
	* Expected parameters:
	* 	- mailbox (the mailbox name)
	*	- id (the message id)
	*/
	public function send_draft_post(){
		//mailbox is set in the constructor - set up our other parameter
		$id = element('id', $this->post(), null);
		if(is_null($id))
			$this->missing_required_fields[] = 'id';

		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('send');
		$this->respond_with_error_if_mailbox_is_inactive();

		//check to make sure that id is valid
		if(!Message::formatted_like_an_id($id))
			$this->invalid_fields[] = 'id';

		//see if we've specified an original sender - this is the only value that we save
		$original_sender = element('original_sender', $this->post(), null);
		if(!is_null($original_sender)){
			$original_mailbox = Mailbox::find_one(array('name' => $original_sender));
			if(!Mailbox::is_an_entity($original_mailbox)){
				$this->invalid_fields[] = 'original_sender';
			}
		}

		$this->respond_with_error_if_fields_are_invalid();

		//find the message
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox))
			$this->response('Message not found.', 422);

#TODO - SHOULD WE INCLUDE THE ID IN THE RESPONSE EVEN IF THE MESSAGE ISN'T FOUND?
		$this->response_message['id'] = $message->id; //we want to always include the message
		if(!$message->draft) $this->response('The message could not be sent because it is not a draft', 422);
		if(!$message->has_recipients()) $this->response('The message could not be sent without a recipient', 422);
//note - for business team reasons, we're letting applications self-enforce whether or not disclosures are required for patient documents.
/*		if(!$message->has_required_attachment_disclosures())
			$this->response('The message could not be sent because it is missing one or more purpose of disclosure entries for the attached patient documents', 422); */
		if(isset($original_mailbox)) $message->original_sender_id = $original_mailbox->id();
		if(!$message->send() || !$message->sent) $this->response('The message failed to send.', 400);
		$this->response('Message sent.', 200);
	}

	public function send_post(){
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('send');
		$this->respond_with_error_if_mailbox_is_inactive();

		$message = $this->save_message();
		if(!Message::is_an_entity($message))
			$this->response('Unable to save message', 400);

		$this->response_message['id'] = $message->id; //we want to always include the message id for these responses
		if(!$message->has_recipients()) $this->response('The message could not be sent without a recipient.  It has been saved as a draft', 422);
//note - for business team reasons, we're letting applications self-enforce whether or not disclosures are required for patient documents.
/*		if(!$message->has_required_attachment_disclosures())
			$this->response('The message could not be sent because it is missing one or more purpose of disclosure entries for the attached patient documents.  It has been saved as a draft', 422); */
		if(!$message->send()) $this->response('The message failed to send.  It has been saved as a draft', 400);
		$this->response('Message sent.', 200);
	}

///////////////////////////////////////////////
// HELPER METHODS
///////////////////////////////////////////////

	protected function save_message($id = NULL){
		if(!is_null($id) && !Message::formatted_like_an_id($id)) return $this->error->should_be_a_message_id($id); //validate id before we get here? possibly reconsider
		//if we have a message id, verify that it's valid & return false otherwise
		if(!empty($id)){
			$message = Message::find_one($id);
			if(!Message::is_an_entity($message) || !$message->draft || !$message->belongs_to_mailbox($this->mailbox))
				$this->response('Message not found.', 422);
			$this->response_message['id'] = $message->id; //we want to always include the message id for these responses
		}

		if(!isset($message))
			$message = new Message( array( 'mailbox_id' => $this->mailbox->id() ) );

		//populate the message fields & make sure that the values are valid.  default values are supplied by the model, so that they'll be consistent if we ever create messages elsewhere
		$optional_fields = array('mailtype', 'priority', 'to', 'cc', 'bcc', 'subject', 'body', 'protected_data');
		foreach($optional_fields as $field){
			if(array_key_exists($field, $this->post())){
				//choose whether or not to enforce XSS filtering (the body element can contain HTML we don't want changed, rely on other methods for preventing XSS in this case)
				$intended_value = (in_array($field, array('body'))) ? purify($this->post($field, FALSE)) : $this->post($field,TRUE);
				if(in_array($field, array('to', 'cc', 'bcc')))
					$intended_value = normalize_address_string($intended_value); //address strings should get normalized before the message is saved to the database

				if($message->property_has_validation($field) && !$message->value_is_valid_for_property($field, $intended_value))
					$this->invalid_fields[] = $field;
				else{
					$message->$field = $intended_value;
					//check to make sure that the field really got set to that value - if not, it's probably because it was an invalid value.
#TODO - come up with way of doing this that won't trigger error notices
					if($message->$field != $intended_value){
						$this->invalid_fields[] = $field;
					}
				}
			}
		}
		//original_sender change from mailbox name to id
		if(array_key_exists('original_sender', $this->post())){
			$original_mailbox = Mailbox::find_one(array('name' =>element('original_sender', $this->post(), null)));
			if(!Mailbox::is_an_entity($original_mailbox)){
				$this->invalid_fields[] = 'original_sender';
			}else{
				$message->original_sender_id = $original_mailbox->id();
			}
		}

		$this->respond_with_error_if_fields_are_invalid();

		//before we save, make sure that the recipient string is trusted
		if($message->has_recipients()){
			$validation = validate($this->mailbox->email_address(), $this->json->decode($message->recipients));
			if (empty($validation['valid_message'])){
				$this->response($this->generate_invalid_recipients_message($validation['invalid_addresses']), 400);
			}
		}

		if(!$message->save()) $this->response('Failed to save message.', 400); //save the message before we try to add attachments to avoid weird behavior
		$this->response_message['id'] = $message->id; //add the id to any future reponse as soon as it's available


		$attachment_fail_message = 'The message has been saved without attachments.';
		if(!$message->property_is_empty('attachment_files'))
			$attachment_fail_message = 'The message has been saved without changes to the attachments.';

		$files_to_attach = array();

		foreach($_FILES as $file_id => $file_info){

			//make sure that we have a unique file name - we use filename as a unique identifier for some methods, so we need to rely on these being unique
			$name = $file_info['name'];
			$extension = substr($name, strpos($name, '.'));
			$name = strip_from_end($extension, $name); //temporarily remove the extension from the filename so that we can add the numbers

			//if we have duplicate file names, make them unique by adding a number (#) to each one
			for($i = 1; array_key_exists($name.$extension, $files_to_attach); $i++){
				//make sure to clear out any parentheses that we've added.  Note that we'll have to do this even for things we didn't add so that we don't mess things up while re-saving.
				if(preg_match('/\(\d+\)$/', $name, $matches)) {
 					$name = trim(strip_from_end(first_element($matches), $name)); //clear out any numbers that we've added already
				}
				$name = $name.' ('.$i.')';
			}
			$name = $name.$extension;

			//determine the binary data for this file
			if(isset($file_info['binary'])){
				//if $_FILES was set manually, grab binary directly
				$binary_string = $file_info['binary'];
			}else{
				//otherwise get binary from file
				$binary_string = file_get_contents($file_info['tmp_name']);
			}

			if(!Message::attachment_has_valid_extension($name)){
				$this->response($name.' does not have a valid file extension and cannot be attached to this message. '.$attachment_fail_message, 422);
			}

			if(!$message->has_attachment($name, $binary_string) && !$message->attachment_is_within_size_limit($name, $binary_string)){ //we only need to care about the size limit if this is a new attachment
				$attachment_length = string_length_in_bytes($binary_string);
				$message = 'Attaching file '.$name.' ('.byte_format($attachment_length).') would make the message '.byte_format(($message->size + $attachment_length)).', '.
						   ' which exceeds the '.byte_format(DIRECT_ATTACHMENT_SIZE_LIMIT).' limit. '.$attachment_fail_message;
				$this->response($message, 422);
			}

			$files_to_attach[$name] = $binary_string;
		}

		if(!$message->set_attachment_files($files_to_attach)){ //use the all-at-once attachment method so that original files are restored if the new files fail to attach
			$this->response('An error occured while attempting to add attachments to the message. '.$attachment_fail_message, 400);
		}

		if(!$message->save()) $this->response('An error occured while attempting to add attachments to the message. '.$attachment_fail_message, 400);

		return $message;
	}
}
?>