<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
*//** */
 
require_once APPPATH.'controllers/services/mailbox_controller.php';

/**
* API methods to manage a mailbox's folders.
*
* Inherits mailbox validation from the mailbox controller.  Mailbox is a required field for all actions on this controller.
*
* @author           <bell_adam@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
* @author Elan Jaffee <jaffee_elan@bah.com>
* @author Joseph Kir <kir_joseph@bah.com>
*
* @package direct-as-a-service
* @subpackage controllers
*/
class Folder_controller extends Mailbox_controller{	

	/**
	* Returns a list of all folder names for this mailbox, along with the number of new messages and the total number of messages for each folder.
	*
	* Note that in this case, "folders" does *not* correlate to the model concept of folders - it includes all mailbox locations along with the custom folders.
	*
	* This method is available to both active and inactive mailboxes.
	*
	* @param string $mailbox Mailbox name (required)
	*/
	public function folders_get(){		
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('retrieve');
		$this->respond_with_error_if_fields_are_invalid();

		//we need to make sure that these have non-numeric keys so that they won't get overwritten by folders with id < 5 -- MG 2014-11-17
		$folders = array( 'inbox' => array( 'id' =>'inbox', 
											'name' => 'Inbox'),
						  'sent' => array('id' =>'sent',
			  					  		  'name' => 'Sent'),
						  'draft' => array('id' =>'draft',
								  		   'name' => 'Drafts'),
						  'archived' => array('id' =>'archived',
			 						  		  'name' => 'Archive'));
											  
		foreach($folders as $id => $values){
			$folders[$id]['parent_id'] = null; //need this value to be consistent with the other folders, but will always be null
			$folders[$id]['depth'] = 0;
			$folders[$id]['max_child_depth'] = 0;
			
			$count_method = $id.'_message_count';
			if($id == 'draft') $count_method = 'draft_count';
			$folders[$id]['total'] = $this->mailbox->$count_method();
			$folders[$id]['new'] = $this->mailbox->$count_method(array('seen' => false));
			
			$folders[$id]['child_folders'] = array(); //need this value to be consistent with the other folders, bu twill
		}											  
											  
		Folder::db()->order_by('name');										  
		foreach($this->mailbox->folder as $folder){
			$folders[$folder->id] = array('id' =>$folder->id, 
										  'name' => $folder->name, 
										  'parent_id' => $folder->parent_id,
										  'depth' => null,
										  'max_child_depth' => 0,
										  'total' => $folder->message_count(),
										  'new'=> $folder->message_count(array('seen' => 0)),
										  'child_folders' => collect('values', $folder->child_folders) );
		}
		
		//CALCULATE FOLDER DEPTHS
		//hokay.  In an ideal world, we would do this on the folder model itself, possibly as a database field that we keep updated all the time.
		//but that's a pretty big change that we don't really have time for right now, and at the moment, we only get folders from this service, and they're always ALL the folders for a mailbox
		//so we'll calculate depth right here -- seems simplest
		$root_folder_ids = array(); //find the root folder ids - mostly because we need something to iterate through that isn't $folders, since we'll be replacing $folders every iteration
		foreach($folders as $id => $folder){
			if(is_numeric($id) && empty($folder['parent_id'])) $root_folder_ids[] = $id;
		}
		
		foreach($root_folder_ids as $root_folder_id){
			$folders = $this->populate_depth_for_folder(0, $root_folder_id, $folders);
		}
		
#TODO - SHOULDN'T THIS BE PLURAL 'FOLDERS'?  CREATE_FOLDER & RENAME_FOLDER RETURN A SINGLE FOLDER VALUES ARRAY AS 'FOLDER' AND ARCHIVE_FOLDER RETURNS THE FOLDER NAME AS 'FOLDER.'  CONSISTENCY?		
		$this->response(array('folder' => $folders), 200);
	}	
	
	/**
	 * Change parent folder of the selected folder in the given mailbox.
	 *
	 * This action is not available for inactive mailboxes.
	 *
	 * @param string $mailbox Mailbox name (required)
	 * @param string $folder_id selected folder id (required)
	 * @param string $parent_id id of the parent folder (required)
	 */
	public function change_parent_post(){		
		$required_fields = array('parent_id',  'folder_id'); 
        foreach($required_fields as $required_field){
        	if(array_key_exists($required_field, $_POST))
            	$$required_field = $this->post($required_field,TRUE);
        	else{
        		$this->missing_required_fields[] = $required_field;
           	}
        }
		
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');	
		$this->respond_with_error_if_mailbox_is_inactive();
		
		//make sure that this name is not a reserved location
		if(!Folder::formatted_like_an_id($folder_id)){
			$this->invalid_fields[] = 'folder_id';
		}
		
		//if we're unsetting the parent id, normalize to null
		if(empty($parent_id)) 
			$parent_id = null;
		
		//make sure the parent id looks like a valid folder id
		if(!is_null($parent_id) && !Folder::formatted_like_an_id($parent_id))
			$this->invalid_fields[] = 'parent_id';

		$this->respond_with_error_if_fields_are_invalid();
		
		//check to make sure that the folder exists and belongs to this mailbox
		$folder = Folder::find_one($folder_id);
		if(!Folder::is_an_entity($folder) || !$folder->belongs_to_mailbox($this->mailbox)){
			$this->response('Folder not found.', 422);
		}
		
		//avoid additional db lookups if this isn't really a change		
		if($parent_id == $folder->parent_id)
			$this->response(array('folder' => $folder->values()), 200); 
		
		//check to make sure that the parent folder exists and belongs to this mailbox
		if(!is_null($parent_id)){
			$parent_folder = Folder::find_one($parent_id);
			if(!Folder::is_an_entity($parent_folder) || !$parent_folder->belongs_to_mailbox($this->mailbox)){
				$this->response('Parent folder not found.', 422);
			}
			
			//a folder cannot be its own parent 
			if($parent_id == $folder->id) {
				$this->response('Failed to change parent because a folder cannot be its own parent.', 422);
			}
			
			//subfolders of a folder cannot become its parent
			if($parent_folder->is_a_descendant_of($folder)) {
				$this->response('Failed to change parent because you are trying to assign one of its descendants to be its parent.', 422);
			}
			
			//need to make sure that this change won't cause the folders to exceed the NESTED FOLDER MAX DEPTH
			if($parent_folder->child_would_exceed_nesting_limit($folder)){
				$this->response('Failed to change parent because it would exceed the maximum number of nested subfolders allowed.', 422);
			}			
		} //end null check for parent id
					
		$folder->parent_id = $parent_id;
		
		//since the parent_id setter makes some checks before setting, check that parent id really got set before saving
		//then check that the save reports success, and then check that the parent id really is what it's supposed to be
		if($folder->parent_id != $parent_id || !$folder->save() || $folder->parent_id != $parent_id) {
			$this->response('Failed to change parent.', 400);
		}
		
		$this->response(array('folder' => $folder->values()), 200);
	}
	
	/**
	* Creates a new folder for the given mailbox.
	*
	* This action is not available for inactive mailboxes.
	*
	* @param string $mailbox Mailbox name (required)
	* @param string $name Name for the new folder (required)
	* @param int $parent_id ID for the parent folder (optional)
	*/
	public function create_folder_post(){		
		$required_fields = array('name');
		$post = $this->post();
        foreach($required_fields as $required_field){
        	if(array_key_exists($required_field, $post))
            	$$required_field = $this->post($required_field,TRUE);
        	else{
        		$this->missing_required_fields[] = $required_field;
           	}
		}		
		
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');	
		$this->respond_with_error_if_mailbox_is_inactive();
		
		
		//make sure that this name is not a reserved location
		if(empty($name) || in_array(mb_strtolower($name), $this->valid_locations())){
			$this->invalid_fields[] = 'name';
		}
		
		if(array_key_exists('parent_id', $_POST)){
			$parent_id = $this->post('parent_id', TRUE);
			if(Folder::formatted_like_an_id($parent_id)){
				$parent_folder = Folder::find_one($parent_id);
				if(!Folder::is_an_entity($parent_folder)){
					$this->invalid_fields[] = 'parent_id';
				}
			}else{
				$this->invalid_fields[] = 'parent_id';
			}
		}

		$this->respond_with_error_if_fields_are_invalid();
		
		if($this->mailbox->has_related('folder', compact('name'))){
			$this->response('Folder name already exists for this mailbox.', 422);
		}
		
		$folder_values = compact('name');
		
		//need to make sure that this change won't cause the folders to exceed the NESTED FOLDER MAX DEPTH - check number of ancestors + parent + child
		if(isset($parent_folder)){
			if(($parent_folder->number_of_ancestors() + 2) > NESTED_FOLDER_MAX_DEPTH)
				$this->response('Failed to create folder under '.$parent_folder->name.' because it would exceed the maximum number of nested subfolders allowed.', 422);
			$folder_values['parent_id'] = $parent_id;
		}
		
		$folder = $this->mailbox->add_folder($folder_values);
		if(!Folder::is_an_entity($folder)){
			$this->response('Failed to create folder.', 400);
		}
		
		$this->response(array('id' => $folder->id, 'folder' => $folder->values()), 200);
	}
	
	/**
	* Renames an existing folder to a new name.  The name must be unique for that user.
	*
	* This action is not available for inactive mailboxes.
	*	
	* @param string $mailbox Mailbox name (required)
	* @param int $folder ID of the custom folder (required)
	* @param string $name Name to change the folder to (required)
	*/
	public function rename_folder_post(){		
		$post = $this->post();
		$required_fields = array('name', 'folder');
        foreach($required_fields as $required_field){
			if(array_key_exists($required_field, $post))
				$$required_field = $this->post($required_field,TRUE);
			else{
				$this->missing_required_fields[] = $required_field;
			   }
        }
				
		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');				
		$this->respond_with_error_if_mailbox_is_inactive();
		
		$folder_id = $folder; //since the parameter isn't named clearly, we'll set a var that's more intuitive.

		//make sure we're not using a reserved name
		if(empty($name) || in_array(mb_strtolower($name), $this->valid_locations()) ){
			$this->invalid_fields[] = 'name';
		}
		
		//make sure that the folder we're referencing exists		
		if(Folder::formatted_like_an_id($folder_id)) 
			$folder = first_element($this->mailbox->folders( array('id' => $folder_id))) ;
		if(!Folder::is_an_entity($folder) || !$folder->belongs_to_mailbox($this->mailbox)) $this->invalid_fields[] = 'folder';

		$this->respond_with_error_if_fields_are_invalid();
		
		//if the folder is already named this, we're done
		if($folder->name == $name)
			$this->response(array('folder' => $folder->values()), 200);
		
		if($this->mailbox->has_related('folder', array( 'name' => $name, 'id !=' => $folder_id)))
			$this->response('Folder name already exists for this mailbox', 422);
		
		$folder->name = $name;
				
		//since the folder model may have checks to make sure that the name is valid, make sure that the name set correctly before saving
		//then make sure that the save works and that the name really saved before return true
		if($folder->name != $name || !$folder->save() || $folder->name != $name){ 
			$this->response('Failed to change folder name', 400);
		}
		
		$this->response(array('folder' => $folder->values()), 200);
	}
	
	/**
	* Archives a custom folder and all the messages within it.
	*
	* This action is not available for inactive mailboxes.
	*
	* @param string $mailbox mailbox name (required)
	* @param int $folder id of the custom folder (required)
	*/
	public function archive_folder_post(){		
		$required_fields = array('folder');
		$post = $this->post();
        foreach($required_fields as $required_field){
        if(array_key_exists($required_field, $post))
            $$required_field = $this->post($required_field,TRUE);
        else{
        	array_push($this->missing_required_fields, $required_field);
           }
        }
		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');		
		$this->respond_with_error_if_mailbox_is_inactive();

		$folder_id = $folder; //since the parameter isn't named clearly, we'll set a var that's more intuitive.
		
#TODO - SHOULDN'T THIS BE THE FULL FOLDER VALUES ARRAY? CREATE_FOLDER AND RENAME_FOLDER BOTH RETURN THESE AS 'FOLDER.'  OTHERWISE, WE COULD CALL THIS 'FOLDER_ID' TO BE CLEAR RE:DIFFERENCE.
		$this->response_message['folder'] = $folder_id;
		if(Folder::formatted_like_an_id($folder_id))
			$folder = Folder::find_one($folder_id); 
		if(!Folder::is_an_entity($folder) || !$folder->belongs_to_mailbox($this->mailbox)){
			$this->invalid_fields[] = 'folder';
		}
			
		$this->respond_with_error_if_fields_are_invalid();
		
		$folder->archive();
		$this->response($this->response_message, 200);
	}
	
/////////////////////////////////////
// PROTECTED METHODS
////////////////////////////////////	
	
	/**
	* Set the depth for a root folder and all of its descendants.
	*
	* Depth is 0 for root folders, 1 for subfolders of root folders, etc.
	*
	* Depth can be determined for folders by calling the $folder->number_of_ancestors() method, but that's an expensive operation that requires looking up all ancestors in the database.
	* In the case of the API, we know that the folder hierarchy is not going to change between calling folders_get() and returning the results via the API.  We also have all of the
	* folders for a mailbox ready for formatting, since we don't allow users to search for individual folders / folders matching a given criteria.  In this case, it's more efficient
	* to just populate the depth for each folder before returning the values via the API.  
	*
	* @todo Could this be made more efficient? (probably!)
	*
	* @param int The depth for this folder - should be 0 when you're calling this, since you should be calling this on root folders
	* @param int The id for the root folder we're targeting
	* @param array The entire collection of folder values that we'll be returning via the API - note that these are arrays, not folder entities
	* @return array The folder values that we'll be returning via the API, updated with appropriate depths for the specified folder and its children
	*/
	protected function populate_depth_for_folder($depth, $folder_id, $folders){
		if(!$this->is->unsigned_integer($depth)) return $this->error->should_be_an_unsigned_integer($depth);
		if(!Folder::formatted_like_an_id($folder_id)) return $this->error->should_be_a_folder_id($folder_id);
		if(!$this->is->nonempty_array($folders)) return $this->error->should_be_a_nonempty_array_of_folders($folder);
		if(!array_key_exists($folder_id, $folders)) return $this->error->should_be_a_known_folder_id($folder_id);
		
		$folders[$folder_id]['depth'] = $depth;
		
		$child_folders = $folders[$folder_id]['child_folders']; //make a copy of $child_folders, since we'll be replacing the $folders array multiple times...
		foreach($child_folders as $child_folder_id => $child_folder){
			$folders = $this->populate_depth_for_folder($depth + 1, $child_folder_id, $folders);
		}
		
		//if there are no child folders, we've reached the bottom of this branch -- populate parents with max child depth
		if(empty($child_folders) && !empty($folders[$folder_id]['parent_id']))
			$folders = $this->populate_max_child_depth_for_folder(1, $folders[$folder_id]['parent_id'], $folders); 
			
		return $folders;
	}
	
	protected function populate_max_child_depth_for_folder($max_child_depth, $folder_id, $folders){
		if(!$this->is->unsigned_integer($max_child_depth)) return $this->error->should_be_an_unsigned_integer($max_child_depth);
		if(!Folder::formatted_like_an_id($folder_id)) return $this->error->should_be_a_folder_id($folder_id);
		if(!$this->is->nonempty_array($folders)) return $this->error->should_be_a_nonempty_array_of_folders($folder);
		if(!array_key_exists($folder_id, $folders)) return $this->error->should_be_a_known_folder_id($folder_id);
		
		$folder = $folders[$folder_id]; 
		
		//as long as this max child depth is larger than the current max child depth, set the max child depth
		if(!array_key_exists('max_child_depth', $folder) || $folder['max_child_depth'] < $max_child_depth){
			$folders[$folder_id]['max_child_depth'] = $max_child_depth;
		
			//populate this max depth upward
			if(!empty($folder['parent_id']))
				return $this->populate_max_child_depth_for_folder($max_child_depth + 1, $folder['parent_id'], $folders);
		}
		
		return $folders;
	}
		
}
?>