/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.vuid.rest.session;

import java.io.IOException;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import javax.inject.Singleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.glassfish.hk2.api.Rank;
import org.jvnet.hk2.annotations.Service;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import gov.vha.isaac.ochre.api.User;
import gov.vha.isaac.ochre.api.PrismeRole;
import gov.vha.vuid.rest.ApplicationConfig;

/**
 * 
 * {@link TestPrismeIntegratedUserService}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
@Service(name="rest-prismeUserService")
@Rank(value = 20)
@Singleton
public class TestPrismeIntegratedUserService extends PrismeIntegratedUserService implements PrismeUserService {
	
	private static Logger log = LogManager.getLogger(TestPrismeIntegratedUserService.class);
	
	/**
	 * 
	 */
	public TestPrismeIntegratedUserService() {
		super();
	}

	/**
	 * Return a user and roles available for that user
	 *
	 * @param ssoToken
	 *            the user's SSO token string
	 * @return the user and roles available to the user
	 * 
	 * This implementation gets a User from PRISME IFF TestPrismeItnegratedUserService is in classpath
	 * and prisme is reachable, otherwise, attempts to parse the token as a test keyword from 
	 * (i.e. TEST_JSON1) OR to parse a test string of the form {name}:{role1}[{,role2}[{,role3}[...]]]
	 */
	public Optional<User> getUser(String ssoToken) {
		
		try {
			return Optional.of(getUserFromPrisme(ssoToken));
		} catch (Exception e) {
			log.info("Failed to retreive user from prisme with ssoToken '" + ssoToken + "'", e);
			if (ApplicationConfig.getInstance().isDebugDeploy())
			{
				//fallback to local parsing for test purposes
				try {
					log.info("Using local sso token parsing");
					return getUserFromTestToken(ssoToken);
				} catch (Exception e1) {
					log.error("Failed to parse local test ssoToken '" + ssoToken + "'", e);
					return Optional.empty();
				}
			}
			else
			{
				return Optional.empty();
			}
		}
	}
	
	public static final String TEST_JSON1 = "{\"roles\":["
	+ "{\"id\":19990,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19991,\"name\":\"editor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19992,\"name\":\"reviewer\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19993,\"name\":\"administrator\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19994,\"name\":\"manager\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19995,\"name\":\"vuid_requestor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19996,\"name\":\"approver\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ "],\"token_parsed?\":true,\"user\":\"TEST\",\"type\":\"ssoi\",\"id\":10005}";
	public static final String TEST_JSON2 = "{\"roles\":["
	+ "{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ "],\"token_parsed?\":true,\"user\":\"DNS   KniazJ\",\"type\":\"ssoi\",\"id\":10005}";
	public static final String TEST_JSON3 = "{\"roles\":["
	+ "{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ ","
	+ "{\"id\":19991,\"name\":\"editor\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}"
	+ "],\"token_parsed?\":true,\"user\":\"DNS   EfronJ\",\"type\":\"ssoi\",\"id\":10005}";

	/**
	 * 
	 * Attempt to construct a user from a string of the following format:
	 * 
	 * {name}:{role1}[{,role2}[{,role3}[...]]]
	 * 
	 * @param arg
	 * @return
	 */
	private Optional<String> constructTestUser(String arg) {
		try {
			String[] components = arg.split(":");
	
			String name = null;
			Set<PrismeRole> roles = new HashSet<>();
			if (components.length == 2) {
				if (components[0].matches("[A-Za-z][A-Za-z0-9_]*")) {
					name = components[0].trim();
	
					String[] roleStrings = components[1].split(",");
	
					for (int i = 0; i < roleStrings.length; ++i) {
						roles.add(PrismeRole.safeValueOf(roleStrings[i].trim()).get());
					}
				}
			}
			
			if (name != null && name.length() > 0 && roles.size() > 0) {
				StringBuilder builder = new StringBuilder();
				builder.append("{\"roles\":[");
				boolean addedRole = false;
				for (PrismeRole role : roles) {
					if (addedRole) {
						builder.append(",");
					}
					builder.append("{\"id\":19990,\"name\":\"" + role.toString() 
					+ "\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}");
	
					addedRole = true;
				}
				
				builder.append("],\"token_parsed?\":true,\"user\":\"" + name + "\",\"type\":\"ssoi\",\"id\":10005}");
				
				return Optional.of(builder.toString());
			}
		} catch (Exception e) {
			// ignore
		}
	
		return Optional.empty();
	}
	
	private Optional<User> getUserFromTestToken(String ssoToken) throws JsonParseException, JsonMappingException, IOException {
		String jsonToUse = null;
	
		Optional<String> createdJson = constructTestUser(ssoToken);
		if (createdJson.isPresent()) {
			jsonToUse = createdJson.get();
		} else {
			if (ssoToken.equals(TEST_JSON1)) {
				jsonToUse = TEST_JSON1;
			} else if (ssoToken.equals(TEST_JSON2)) {
				jsonToUse = TEST_JSON2;
			} else if (ssoToken.equals(TEST_JSON3)) {
				jsonToUse = TEST_JSON3;
			} else if (ssoToken.equals("TEST_JSON1") || ssoToken.equals("TEST")) {
				jsonToUse = TEST_JSON1;
			} else if (ssoToken.equals("TEST_JSON2")) {
				jsonToUse = TEST_JSON2;
			} else if (ssoToken.equals("TEST_JSON3")) {
				jsonToUse = TEST_JSON3;
			} else {
				// Either a real SSO token or custom JSON
				jsonToUse = ssoToken;
			}
		}
	
		/*
		 * Example URL for get_roles_by_token
		 * URL url = new URL("https://DNS                     /rails_prisme/roles/get_roles_by_token.json?token=" + token);
		 */
		User newUser = null;
		try {
			newUser = UserServiceUtils.getUserFromJson(jsonToUse, ssoToken);
		} catch (Exception e) {
			try {
				newUser = UserServiceUtils.getUserFromJson(TEST_JSON1, ssoToken);
			} catch (IOException e1) {
				throw new RuntimeException(e1);
			}
		}
	
		return Optional.of(newUser);
	}
}
