/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.vuid.rest.session;

import java.io.IOException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import gov.vha.isaac.ochre.api.User;
import gov.vha.isaac.ochre.api.Get;
import gov.vha.isaac.ochre.api.PrismeRole;
import gov.vha.isaac.ochre.api.bootstrap.TermAux;
import gov.vha.isaac.ochre.api.util.UuidT5Generator;
import gov.vha.isaac.rest.utils.CommonPrismeServiceUtils;

/**
 * 
 * {@link UserServiceUtils}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class UserServiceUtils {
	
	private static Logger log = LogManager.getLogger(UserServiceUtils.class);
	
	static User getUserFromJson(String jsonToUse, String ssoToken) throws JsonParseException, JsonMappingException, IOException {
		ObjectMapper mapper = new ObjectMapper();
		Map<?, ?> map = null;
		map = mapper.readValue(jsonToUse, Map.class);
		
		String userName = (String)map.get("user");
		Set<PrismeRole> roleSet = new HashSet<>();
		Collection<?> roles = (Collection<?>)map.get("roles");
		for (Object roleMapObject : roles) {
			Map<?,?> roleMap = (Map<?,?>)roleMapObject;
			String roleName = (String)roleMap.get("name");
			Optional<PrismeRole> foundRole = PrismeRole.safeValueOf(roleName);
			
			if (foundRole.isPresent()) {
				roleSet.add(foundRole.get());
			}
			else
			{
				log.warn("Not adding unknown role '{}' to user '{}'", roleName, userName);
			}
		}
		
		final UUID uuidFromUserFsn = UserServiceUtils.getUuidFromUserName(userName);;
	
		User newUser = new User(userName, uuidFromUserFsn, ssoToken, roleSet);
		
		return newUser;
	}

	static UUID getUuidFromUserName(String userName) {
		return UuidT5Generator.get(TermAux.USER.getPrimordialUuid(), userName);
	}

	static User getUserFromUrl(URL url, String ssoToken) throws JsonParseException, JsonMappingException, IOException {
//		/*
//		 * Example URL for get_roles_by_token
//		 * URL url = new URL("https://DNS                     /rails_prisme/roles/get_roles_by_token.json?token=" + token);
//		 */
//		/*
//		 * Example SSO Token
//		 * AI
//		 */
//		//String json = "{\"roles\":[{\"id\":10000,\"name\":\"read_only\",\"resource_id\":null,\"resource_type\":null,\"created_at\":\"2016-09-13T14:48:18.000Z\",\"updated_at\":\"2016-09-13T14:48:18.000Z\"}],\"token_parsed?\":true,\"user\":\"DNS   ArmbrD\",\"type\":\"ssoi\",\"id\":10005}";

		log.info("Attempting to read user from url '{}'", url);
		Map<String, String> params = new HashMap<>();	
		params.put("token", ssoToken);	
		params.put("isaac_db_uuid", "");  //We don't know the datastore ID, and in the vuid server, we don't care - the vuid generate role is global.
		log.debug("Using token '{}' and no DB", ssoToken);
		String jsonResultString = PrismeServiceUtils.getResultJsonFromPrisme(CommonPrismeServiceUtils.getTargetFromUrl(url), url.getPath(), params);
		
		return UserServiceUtils.getUserFromJson(jsonResultString, ssoToken);
	}
}
