/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.vuid.rest.session;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import gov.vha.isaac.rest.api.exceptions.RestException;

/**
 * 
 * {@link RequestParameters}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 * All parameters should be added to ALL_VALID_PARAMETERS, whether grouped or individually
 * 
 */
public class RequestParameters {

	private RequestParameters() {}
	
	public final static String ssoToken = "ssoToken";
	public final static String blockSize = "blockSize";
	public final static String reason = "reason";

	public final static String vuid = "vuid";

	/**
	 * Set of all known parameters usable to detect malformed or incorrect parameters
	 */
	public final static Set<String> ALL_VALID_PARAMETERS;
	static {
		Set<String> params = new HashSet<>();
		params.addAll(unmodifiableSet(
			ssoToken,
			blockSize,
			reason,
			vuid
			));
		ALL_VALID_PARAMETERS = params;
	}

	/**
	 * This should only be modified for testing purposes.  Otherwise should always be IGNORE_CASE_VALIDATING_PARAM_NAMES_DEFAULT
	 * 
	 * Changing this to ignore case will probably break lots of things,
	 * as most comparisons do not ignore case, as currenlt coded
	 */
	public final static boolean IGNORE_CASE_VALIDATING_PARAM_NAMES_DEFAULT = false;
	public static boolean IGNORE_CASE_VALIDATING_PARAM_NAMES = IGNORE_CASE_VALIDATING_PARAM_NAMES_DEFAULT;

	/**
	 * @param parameters
	 * @param supportedParameterNames
	 * @throws RestException
	 * 
	 * This method validates the context request parameters against passed valid parameters
	 * It takes multiple parameter types in order to allow passing the constant parameter sets
	 * from RequestParameters as well as any individual parameters passed in specific methods
	 */
	public final static void validateParameterNamesAgainstSupportedNames(Map<String, List<String>> parameters, Object...supportedParameterNames) throws RestException {
		Set<String> supportedParameterNamesSet = new HashSet<>();
		if (supportedParameterNames != null && supportedParameterNames.length > 0) {
			for (Object parameter : supportedParameterNames) {
				if (parameter instanceof Iterable) {
					for (Object obj : (Iterable<?>)parameter) {
						supportedParameterNamesSet.add(obj.toString());
					}
				} else if (parameter.getClass().isArray()) {
					for (Object obj : (Object[])parameter) {
						supportedParameterNamesSet.add(obj.toString());
					}
				} else {
					supportedParameterNamesSet.add(parameter.toString());
				}
			}
		}
		for (String parameterName : parameters.keySet()) {
			String parameterNameToCompare = IGNORE_CASE_VALIDATING_PARAM_NAMES ? parameterName.toUpperCase(Locale.ENGLISH) : parameterName;
			boolean foundMatch = false;
			for (String supportedParameterName : supportedParameterNamesSet) {
				String supportedParameterNameToCompare = IGNORE_CASE_VALIDATING_PARAM_NAMES ? supportedParameterName.toUpperCase(Locale.ENGLISH) : supportedParameterName;
				if (supportedParameterNameToCompare.equals(parameterNameToCompare)) {
					foundMatch = true;
					break;
				}
			}
			if (!foundMatch) {
				throw new RestException(parameterName, Arrays.toString(parameters.get(parameterName).toArray()), "Invalid or unsupported parameter name.  Must be one of " + Arrays.toString(supportedParameterNamesSet.toArray(new String[supportedParameterNamesSet.size()])));
			}
		}
	}
	private final static <T> Set<T> unmodifiableSet(@SuppressWarnings("unchecked") T...elements) {
		Set<T> list = new HashSet<>(elements.length);
		for (T element : elements) {
			list.add(element);
		}
		return Collections.unmodifiableSet(list);
	}
}
