/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.vuid.rest.session;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.ochre.api.User;
import gov.vha.isaac.ochre.api.PrismeRole;
import gov.vha.isaac.rest.api.exceptions.RestException;
import gov.vha.vuid.rest.session.RequestParameters;

/**
 * {@link RequestInfo}
 * This class is intended to hold a cache of global request info that we tie to the request / session being processed.
 * Things like the STAMP that applies, the expandable parameters, etc.
 * We will (likely) set this up on the thread local with a request filter that looks at every request before it arrives 
 * at the implementing method.
 * 
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 */
public class RequestInfo
{
	private static Logger log = LogManager.getLogger(RequestInfo.class);

	private final static User DEFAULT_READ_ONLY_USER = new User("READ_ONLY_USER", null, null, PrismeRole.READ_ONLY);

	private Map<String, List<String>> parameters_ = new HashMap<>();

	private Optional<User> user_ = null;
	
	private static final ThreadLocal<RequestInfo> requestInfo = new ThreadLocal<RequestInfo>()
	{
		@Override
		protected RequestInfo initialValue()
		{
			return new RequestInfo();
		}
	};

	public static RequestInfo get()
	{
		return requestInfo.get();
	}

	private RequestInfo()
	{
	}

	public static void remove() {
		requestInfo.remove();
	}

	public RequestInfo readAll(Map<String, List<String>> parameters) throws Exception
	{
		parameters_.clear();
		for (Map.Entry<String, List<String>> entry : parameters.entrySet()) {
			parameters_.put(entry.getKey(), Collections.unmodifiableList(entry.getValue()));
		}
		return requestInfo.get();
	}

	/**
	 * @return parameters Map
	 */
	public Map<String, List<String>> getParameters() {
		return parameters_;
	}

	public void setDefaultReadOnlyUser() {
		user_ = Optional.of(DEFAULT_READ_ONLY_USER);
	}

	/**
	 * Lazily create and return a User
	 *
	 * @return Optional<User>
	 */
	public Optional<User> getUser() throws RestException {
		if (user_ == null) {
			String ssoTokenParameter = null;
			try {
				ssoTokenParameter = parameters_.get(RequestParameters.ssoToken).iterator().next();
				if (StringUtils.isBlank(ssoTokenParameter)) {
					throw new RuntimeException("blank " + RequestParameters.ssoToken + " parameter");
				}
			} catch (Exception e) {
				RestException re = new RestException(RequestParameters.ssoToken, null, "getUser() FAILED due to null, empty or blank " + RequestParameters.ssoToken + " parameter");
				log.info(re);
				throw re;
			}
			Optional<User> userOptional = LookupService.getService(PrismeUserService.class).getUser(parameters_.get(RequestParameters.ssoToken).iterator().next());

			user_ = userOptional;
		}
		return user_;
	}
}
