package gov.vha.vuid.rest;

import java.util.concurrent.atomic.AtomicInteger;
import javax.servlet.ServletContext;
import javax.ws.rs.ApplicationPath;
import javax.ws.rs.core.Context;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.server.spi.Container;
import org.glassfish.jersey.server.spi.ContainerLifecycleListener;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.ochre.api.util.WorkExecutors;
import gov.vha.vuid.rest.api1.data.RestSystemInfo;
import gov.vha.vuid.rest.api1.RestPaths;
import gov.vha.vuid.rest.data.VuidService;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;

@ApplicationPath(RestPaths.appPathComponent)
public class ApplicationConfig extends ResourceConfig implements ContainerLifecycleListener
{
	private static final AtomicInteger startup = new AtomicInteger(1);
	private Logger log = LogManager.getLogger(ApplicationConfig.class);

	private static ApplicationConfig instance_;

	private StringProperty status_ = new SimpleStringProperty("Not Started");
	private boolean debugMode = true;
	private boolean shutdown = false;
	private boolean initialStartupComplete = false;
	private int PORT = PORT;
	private String hostname;
	private String protocol;
	private RestSystemInfo systemInfo_;
	
	//Note - this injection works fine, when deployed as a war to tomcat.  However, when launched in the localJettyRunner from eclipse,
	//this remains null.
	@Context
	ServletContext context_;

	private String contextPath;

	public ApplicationConfig()
	{
		//If we leave everything to annotations, is picks up the eclipse moxy gson writer, which doesn't handle abstract classes properly.
		//The goal here is to force it to use Jackson, but it seems that registering jackson disables scanning, so also have to re-enable
		//scanning.  It also seems to forget to scan this class... so register itself..
		super(new ResourceConfig().packages("gov.vha.vuid.rest").register(JacksonFeature.class).register(ApplicationConfig.class));
	}

	public static ApplicationConfig getInstance()
	{
		return instance_;
	}

	@Override
	public void onReload(Container arg0)
	{
		// TODO determine if VuidService should be reinitialized onReload()
	}

	@Override
	public void onShutdown(Container arg0)
	{
		shutdown = true;
		log.info("Stopping VUID Service");
		LookupService.shutdownSystem();
		initialStartupComplete = false;
		log.info("VUID Service stopped");
	}

	@Override
	public void onStartup(Container container)
	{
		log.info("onStartup called");
		if (instance_ != null)
		{
			throw new RuntimeException("Unexpected!");
		}
		instance_ = this;

		//context is null when run from eclipse with the local jetty runner.
		if (context_ == null)
		{
			debugMode = true;
			contextPath = "vuid-rest";
		}
		else
		{
			contextPath = context_.getContextPath().replace("/", "");
			debugMode = (contextPath.contains("SNAPSHOT") ? true : false);
		}

		log.info("Context path of this deployment is '" + contextPath + "' and debug mode is " + debugMode);

		servicesInit();
	}

	public boolean isServiceReady()
	{
		return initialStartupComplete && LookupService.getService(VuidService.class).isReady();
	}
	
	public void setStatus(String statusMessage)
	{
		log.debug("Resetting ApplicationConfig status to \"" + statusMessage + "\"");
		status_.set(statusMessage);
	}

	public String getStatusMessage()
	{
		return status_.get();
	}
	
	private void servicesInit()
	{
		log.info("Services Init called");
		if (startup.getAndDecrement() == 1)
		{
			log.info("Executing initial services Init in background thread");
			Runnable r = new Runnable()
			{
				@Override
				public void run()
				{
					try
					{
						log.info("VUID Service Init thread begins");
						if (shutdown)
						{
							return;
						}
						status_.set("Starting DB init process");
						LookupService.setRunLevel(LookupService.SL_NEG_1_METADATA_STORE_STARTED_RUNLEVEL);
						
						if (shutdown)
						{
							return;
						}
						
						//go up one more level, to start up the logger, now that we have config info.
						LookupService.setRunLevel(LookupService.SL_L0);
						
						systemInfo_ = new RestSystemInfo();

						status_.set("Ready");

						if (isDebugDeploy()) {
							System.out.println("Done setting up VUID Service");

							System.out.println(String.format("Application started.\nTry out (POST) %s%s\nStop the application by pressing enter.",
									"http://localhost:8181/", "vuid-rest/write/1/vuids/allocate?blockSize=10&reason=a%20reason&ssoToken=TestUser:vuid_requestor"));
						}
					}
					catch (Exception e)
					{
						log.error("Failure starting VUID Service", e);
						status_.set("FAILED!");
					}
					finally
					{
						initialStartupComplete = true;
					}
				}
			};

			LookupService.get().getService(WorkExecutors.class).getExecutor().execute(r);
		}
	}
	
	/**
	 * A place for a filter to call in and stash the service listen port when it becomes known
	 * @param port
	 */
	public void setPort(int port)
	{
		this.port = port;
	}
	
	/**
	 * A place for a filter to call in and stash the protocol when it becomes known
	 * @param protocol
	 */
	public void setProtocol(String protocol)
	{
		this.protocol = protocol;
	}

	/**
	 * A place for a filter to call in and stash the service host name when it becomes known
	 * @param hostName
	 */
	public void setHostName(String hostName)
	{
		this.hostname = hostName;
	}
	
	/**
	 * Returns whatever port was set by setPort(int)
	 * @return
	 */
	public int getPort()
	{
		return port;
	}

	/**
	 * Returns whatever protocol was set by setProtocol(String)
	 * @return
	 */
	public String getProtocol()
	{
		return protocol;
	}

	/**
	 * Returns whatever port was set by setHostName(String)
	 * @return
	 */
	public String getHostname()
	{
		return hostname;
	}

	/**
	 * @return true if this is a debug deployment (in eclipse, or context contains SNAPSHOT)
	 */
	public boolean isDebugDeploy()
	{
		return debugMode;
	}

	/**
	 * @return String context path, which is a hard-coded value if in eclipse Jetty
	 */
	public String getContextPath() {
		return contextPath;
	}
	
	public boolean isShutdownRequested()
	{
		return shutdown;
	}

	public ServletContext getServletContext()
	{
		return context_;
	}
	
	public RestSystemInfo getSystemInfo()
	{
		return systemInfo_;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "ApplicationConfig [status_=" + status_ + ", debugMode=" + debugMode + ", shutdown="
				+ shutdown + ", initialStartupComplete=" + initialStartupComplete + ", port=" + port + ", hostname="
				+ hostname + ", protocol=" + protocol + ", systemInfo_=" + systemInfo_
				+ ", contextPath=" + contextPath + "]";
	}
}